<?php
/**
* @version $Id: cbpaidsubscriptions.sysplug.php 428 2010-01-26 11:11:34Z brunner $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

$memMax				=	trim( @ini_get( 'memory_limit' ) );
if ( $memMax ) {
	$last			=	strtolower( substr( $memMax, -1 ) );
	$memMax			=	(int) $memMax;

	switch( $last ) {
		case 'g':
			$memMax	*=	1024 * 1024 * 1024;
			break;
		case 'm':
			$memMax	*=	1024 * 1024;
			break;
		case 'k':
			$memMax	*=	1024;
			break;
	}
	if ( $memMax < 48000000 ) {
		@ini_set( 'memory_limit', '48M' );
	}
}


if ( ! is_readable( JPATH_ADMINISTRATOR . '/components/com_comprofiler/plugin.foundation.php' ) ) {
	JFactory::getApplication()->enqueueMessage( "Mandatory Community Builder package not installed!", 'error');
	return;
}

jimport( 'joomla.plugin.plugin' );

/**
 * CBSubs System Plugin class
 */
class cbpaidSysPlugin {
	/**
	 * Paid subscriptions manager
	 * @var cbpaidSubscriptionsMgr
	 */
	public $paidsubsManager			=	null;

	/**
	 * Constructor
	 */
	public function __construct( ) {
		global /** @noinspection PhpUnusedLocalVariableInspection */
		$ueConfig;		// needed for the includes below, incl. ue_config.php

		$_CB_joomla_path		=	JPATH_SITE;
		$_CB_joomla_adminpath	=	JPATH_ADMINISTRATOR;

		$_CB_adminpath				=	$_CB_joomla_adminpath. "/components/com_comprofiler";

		if ( ! file_exists( $_CB_adminpath . '/plugin.class.php' )) {
			if ( is_callable( array( 'JError', 'raiseWarning' ) ) ) {
				JError::raiseNotice( 'SOME_ERROR_CODE', 'Paid Subscriptions bot detected that Community Builder is not installed.', '' );
			} else {
				trigger_error( 'Paid Subscriptions bot detected that Community Builder is not installed.', E_USER_WARNING );
			}
			return;
		}
		if ( ! file_exists( $_CB_joomla_path . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' )) {
			if ( is_callable( array( 'JError', 'raiseWarning' ) ) ) {
				JError::raiseNotice( 'SOME_ERROR_CODE', 'Paid Subscriptions bot detected that Community Builder Paid Subscriptions plugin is not installed.', '' );
			} else {
				trigger_error( 'Paid Subscriptions bot detected that Community Builder Paid Subscriptions plugin is not installed.', E_USER_WARNING );
			}
			return;
		}
		/** @noinspection PhpIncludeInspection */
		include_once( $_CB_adminpath . '/plugin.foundation.php' );
		cbimport( 'cb.plugins' );
		//cbimport( 'cb.tabs' );		// comprofiler.class.php is not needed for sure.
		cbimport( 'cb.database' );
		cbimport( 'cb.tables' );
		/** @noinspection PhpIncludeInspection */
		include_once( $_CB_joomla_path . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

		$this->paidsubsManager		=	cbpaidSubscriptionsMgr::getInstance();
	}
	/**
	 * Checks if the logged-in user needs to be expired
	 */
	public function checkExpireMe( ) {
		if ( $this->paidsubsManager ) {
			$this->paidsubsManager->checkExpireMe( 'system_mambot', null, true );
		}
	}
	/**
	 * Register a function with a priority for onAlittleMoreAfterStart execution
	 * @static
	 *
	 * @param  mixed  $function
	 * @param  int    $priority  1..100 : default 50, max: 1, min: 100
	 */
	public static function registerOnRealStart( $function, $priority = 50 ) {
		static $registeredFunctions	=	array();
		if ( $function ) {
			$registeredFunctions[$priority][]	=	$function;
		} elseif ( $function === false ) {
			foreach ($registeredFunctions as $fncts ) {
				foreach ( $fncts as $f ) {
					call_user_func_array( $f, array() );
				}
			}
		}
	}
	/**
	 * Triggers the system plugins
	 */
	public function triggerSysPlugins() {
		// now check for content plugin:
		self::registerOnRealStart( false );
		//TODO LATER: make it more universal.
		// $_PLUGINS->trigger( 'onCPayHostSystemStart', array() );		// not needed here as plugins are not loaded depending on triggers yet
	}
}

class plgSystemcbpaidsubsbot extends JPlugin
{

	/**
	 * Loads CB and CBSubs API only once
	 *
	 * @return bool
	 */
	private function loadCBSubs()
	{
		global $_CB_framework;

		static $CBSubs_loaded	=	0;

		if ( ! $CBSubs_loaded++ ) {
			if ( ( ! file_exists( JPATH_SITE . '/libraries/CBLib/CBLib/Core/CBLib.php' ) ) || ( ! file_exists( JPATH_ADMINISTRATOR . '/components/com_comprofiler/plugin.foundation.php' ) ) ) {
				return false;
			}

			/** @noinspection PhpIncludeInspection */
			include_once( JPATH_ADMINISTRATOR . '/components/com_comprofiler/plugin.foundation.php' );

			cbimport( 'cb.html' );

			/** @noinspection PhpIncludeInspection */
			include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php');
		}

		return true;
	}

	/**
	 * Event handler for onAfterStart, run on onAfterRoute J event
	 */
	public function onAfterRoute()
	{
		if ( ! $this->loadCBSubs() ) {
			return;
		}

		$_CBPAID_SYSPLUG	=	new cbpaidSysPlugin();

		$_CBPAID_SYSPLUG->checkExpireMe();
		$_CBPAID_SYSPLUG->triggerSysPlugins();
	}

	/**
	 * @param string $context The context of the content being passed to the plugin.
	 * @param mixed  &$row    An object with a "text" property
	 * @param mixed  $params  Additional parameters. See {@see PlgContentContent()}.
	 * @param int    $page    Optional page number. Unused. Defaults to zero.
	 *
	 * @return bool
	 */
	public function onContentPrepare( $context, &$row, /** @noinspection PhpUnusedParameterInspection */ &$params, /** @noinspection PhpUnusedParameterInspection */ $page = 0 )
	{
		if ( ( $context == 'com_finder.indexer' ) || ( ! isset( $row->text ) ) ) {
			return true;
		}

		$ignore												=	$this->params->get( 'ignore_context' );

		if ( $ignore ) {
			$ignore											=	explode( ',', $ignore );

			foreach ( $ignore as $ignoreContext ) {
				if ( strpos( $context, $ignoreContext ) !== false ) {
					return true;
				}
			}
		}

		if ( ( ! $row->text ) || ( strpos( $row->text, '[cbsubs:' ) === false ) ) {
			return true;
		}

		if ( ! $this->loadCBSubs() ) {
			return true;
		}

		$cbUser												=	CBuser::getMyInstance();
		$pluginParams										=	cbpaidApp::settingsParams();

		$row->text											=	$this->substituteText( htmlspecialchars_decode( $row->text, ENT_COMPAT ), $cbUser, $pluginParams );

		return true;
	}

	/**
	 * @param UserTable $user
	 * @return array
	 */
	private function evaluateUserSubscriptions( $user )
	{
		global $_CB_framework;

		$userId												=	$user->get( 'id', 0, GetterInterface::INT );

		if ( ! $userId ) {
			static $cache									=	null;

			if ( $cache === null ) {
				$plansManager								=	cbpaidPlansMgr::getInstance();
				$cache										=	$plansManager->loadPublishedPlans( null, true, 'registration', null );
			}

			$plans											=	$cache;
			$subscriptions									=	array();
		} else {
			static $userCache								=	array();

			if ( ! isset( $userCache[$userId] ) ) {
				$subscriptions								=	array();

				$plansManager								=	cbpaidSubscriptionsMgr::getInstance();
				$plans										=	$plansManager->getUpgradeAndRenewalPossibilities( 1, $userId, $_CB_framework->now(), $subscriptions );

				$userCache[$userId]['plans']				=	$plans;
				$userCache[$userId]['subscriptions']		=	$subscriptions;
			}

			$plans											=	$userCache[$userId]['plans'];
			$subscriptions									=	$userCache[$userId]['subscriptions'];
		}

		return array( 'plans' => $plans, 'subscriptions' => $subscriptions );
	}

	/**
	 * Evaluate an user attribute
	 * @access private  Private function: DO NOT USE AS PUBLIC!
	 * (it is public because PHP 5.3 does not support $this in anonymous functions within function _evaluateIfs() below)
	 * This function is on purpose an adapted duplicate of the CBuser::_evaluateUserAttrib() function, as it's a private CB function
	 * and the whole CB substitutions should be re-factored soon, so no wish to make the CBuser function public!
	 *
	 * @param  CBuser  $cbUser
	 * @param  string  $userAttrVal  Attribute value ('#me', '#displayed', '#displayedOrMe', '1231' (user-id))
	 * @return CBuser                CB User corresponding to $userAttrValue
	 */
	public function evaluateUserAttrib( $cbUser, $userAttrVal )
	{
		global $_CB_framework;

		if ( $userAttrVal !== '' ) {
			$uid			=	null;

			if ( ( $userAttrVal == '#displayed' ) || ( $userAttrVal == '#displayedOrMe' ) ) {
				$uid		=	(int) $_CB_framework->displayedUser();
			}

			if ( ( ! $uid ) && ( ( $userAttrVal == '#displayedOrMe' ) || ( $userAttrVal == '#me' ) ) ) {
				$uid		=	(int) Application::MyUser()->getUserId();
			}

			if ( ( ! $uid ) && preg_match( '/^[1-9][0-9]*$/', $userAttrVal ) ) {
				$uid		=	(int) $userAttrVal;
			}

			if ( $uid ) {
				$userTable		=	$cbUser->getUserData();
				if ( $userTable && ( $uid == $userTable->id ) ) {
					$user = $cbUser;
				} else {
					$user	=	CBuser::getInstance( (int) $uid, false );
				}
			} else {
				$user		=	CBuser::getInstance( null );
			}
		} else {
			$user			=	$cbUser;
		}

		return $user;
	}

	/**
	 * @param string                       $text
	 * @param CBuser                       $cbUser
	 * @param ParamsInterface|cbpaidConfig $params
	 * @param bool                         $elseIf
	 * @return mixed
	 */
	private function substituteCondition( $text, $cbUser, $params, $elseIf = false )
	{
		if ( ( ! $text ) || ( strpos( $text, ( $elseIf ? '[cbsubs:elseif' : '[cbsubs:if' ) ) === false ) ) {
			// There's nothing to even check for so don't waste resources doing a preg_replace_callback when we don't have to:
			return $text;
		}

		$regex				=	'%\[cbsubs:if(?: user=(?:"|&quot;)((?:(?!&quot;)[^"/\[\] ])+)(?:"|&quot;))? ([^\]]+)]((?:[^\[]|\[(?!/?cbsubs:if[^\]]*\])|(?R))*)\[/cbsubs:if\]%i';
		$regexElseIf		=	'%\[cbsubs:elseif(?: user=(?:"|&quot;)((?:(?!&quot;)[^"/\[\] ])+)(?:"|&quot;))? ([^\]]+)]((?:[^\[]|\[(?!/?cbsubs:elseif[^\]]*\])|(?R))*)\[/cbsubs:elseif\]%i';
		$regexElse			=	'#\[cbsubs:else\]((?:[^\[]+|\[(?!/?cbsubs:else[^\]]*])|(?R))+)\[/cbsubs:else]#';

		$now				=	Application::Date( 'now', 'UTC' )->getTimestamp();
		$that				=	$this;

		$text				=	preg_replace_callback( ( $elseIf ? $regexElseIf : $regex ), function( array $matches ) use ( $now, $cbUser, $params, $elseIf, $regexElseIf, $regexElse, $that )
									{
										if ( preg_match_all( '/(?: (&&|and|\|\||or|) )?plan=(?:"|&quot;)(\d+)(?:"|&quot;)(?: status(=|<>|!=|=~|!~)(?:"|&quot;)(A|R|X|C|U|I)(?:"|&quot;))?(?: since(=|<|>|>=|<=|<>|!=)(?:"|&quot;)((?:(?!&quot;)[^"])+)(?:"|&quot;))?(?: date=(?:"|&quot;)((?:(?!&quot;)[^"])+)(?:"|&quot;))?/i', $matches[2], $conditions ) ) {
											$user							=	$that->evaluateUserAttrib( $cbUser, $matches[1] )->getUserData();
											$userSubscriptions				=	$that->evaluateUserSubscriptions( $user );

											/** @var cbpaidUsersubscriptionRecord[] $subscriptions */
											$subscriptions					=	$userSubscriptions['subscriptions'];

											$and							=	true;
											$or								=	false;

											for ( $i = 0, $n = count( $conditions[0] ); $i < $n; $i++ ) {
												$operator					=	$conditions[1][$i];
												$plan						=	$conditions[2][$i];
												$statusCompare				=	$conditions[3][$i];
												$status						=	$conditions[4][$i];
												$sinceCompare				=	$conditions[5][$i];
												$since						=	$conditions[6][$i];
												$date						=	$conditions[7][$i];

												if ( ! $status ) {
													$statusCompare			=	'=';
													$status					=	'A';
												}

												// Set default subscription state as invalid to allow IF checks against missing subscriptions:
												$subStatus					=	'I';
												$subDate					=	null;

												/** @var cbpaidUsersubscriptionRecord[] $subscriptions */
												foreach ( $subscriptions as $sub ) {

													if ( $sub->plan_id == $plan ) {

														if ( ( $subStatus == 'A' ) && ( $sub->status != 'A' ) ) {
															continue;
														}

														$subStatus			=	$sub->status;

														switch ( $date ) {
															case 'subscription':
																$subDate	=	$sub->subscription_date;
																break;
															case 'last_renewed':
																$subDate	=	$sub->last_renewed_date;
																break;
															case 'expiry':
																$subDate	=	$sub->expiry_date;
																break;
															case 'previous_expiry':
																$subDate	=	$sub->previous_expiry_date;
																break;
															default:
																$subDate	=	( $status == 'X' ? $sub->expiry_date : $sub->subscription_date );
																break;
														}
													}
												}

												switch ( $statusCompare ) {
													case '<>':
													case '!=':
														$matched			=	( $subStatus != strtoupper( $status ) );
														break;
													case '=~':
														$matched			=	( @preg_match( $status, $subStatus ) === 1 );
														break;
													case '!~':
														$matched			=	( @preg_match( $status, $subStatus ) == 0 );
														break;
													case '=':
													default:
														$matched			=	( $subStatus == strtoupper( $status ) );
														break;
												}

												if ( $matched && $since ) {
													// It's possible for there to be no expiry date (lifetime) and maybe no subscription date (in error likely); lets not cripple the page with date errors:
													if ( $subDate ) {
														$dateSince			=	Application::Date( $subDate, 'UTC' )->modify( $since )->getTimestamp();

														switch ( $sinceCompare ) {
															case '<':
																// I have been subscribed for less than the amount of time supplied (e.g. since<"1 DAY"):
																$matched	=	( $dateSince > $now );
																break;
															case '>':
																// I have been subscribed for greater than the amount of time supplied (e.g. since>"1 DAY"):
																$matched	=	( $dateSince < $now );
																break;
															case '>=':
																// I have been subscribed for greater than or equal to the amount of time supplied (e.g. since>="1 DAY"):
																$matched	=	( $dateSince <= $now );
																break;
															case '<=':
																// I have been subscribed for less than or equal to the amount of time supplied (e.g. since<="1 DAY"):
																$matched	=	( $dateSince >= $now );
																break;
															case '<>':
															case '!=':
																// I have NOT been subscribed for exactly the amount of time supplied (e.g. since="1 DAY"):
																$matched	=	Application::Date( $dateSince, 'UTC' )->diff( $now )->days;
																break;
															case '=':
															default:
																// I have been subscribed for exactly the amount of time supplied (e.g. since="1 DAY"):
																$matched	=	( ! Application::Date( $dateSince, 'UTC' )->diff( $now )->days );
																break;
														}
													}
												}

												if ( in_array( $operator, array( 'or', '||' ) ) ) {
													if ( $matched ) {
														// If at least 1 OR usage matched then $or passes entirely:
														$or					=	true;
													}
												} else {
													if ( ! $matched ) {
														// If even 1 AND usage failed then $and must fail entirely:
														$and				=	false;
													}
												}
											}

											$isMatched						=	( $and || $or );
											$string							=	$matches[3];
											$stringElse						=	'';

											if ( ! $elseIf ) {
												// Check for an elseif usage and parse:
												if ( ( strpos( $string, '[cbsubs:elseif' ) !== false ) && preg_match_all( $regexElseIf, $string, $matchesElseIf ) ) {
													for ( $i = 0, $n = count( $matchesElseIf[0] ); $i < $n; $i++ ) {
														if ( $isMatched ) {
															// The if usage matched so we need to remove these elseif strings:
															$string			=	str_replace( $matchesElseIf[0][$i], '', $string );
														} else {
															$stringElse		=	$that->substituteCondition( $matchesElseIf[0][$i], $cbUser, $params, true );

															if ( $stringElse ) {
																// We found an elseif match so stop as we don't need to check the others:
																break;
															}
														}
													}
												}

												// Check for else usage and parse if an else string (from elseif) hasn't already been found:
												if ( ( strpos( $string, '[cbsubs:else]' ) !== false ) && ( ! $stringElse ) && preg_match( $regexElse, $string, $matchesElse ) ) {
													if ( $isMatched ) {
														// The if usage matched so we need to remove this else string:
														$string				=	str_replace( $matchesElse[0], '', $string );
													} else {
														$stringElse			=	$matchesElse[1];
													}
												}
											}

											return $that->substituteCondition( ( $isMatched ? $string : $stringElse ), $cbUser, $params );
										}

										return null;
									},
									$text );

		return $text;
	}

	/**
	 * @param string                       $text
	 * @param CBuser                       $cbUser
	 * @param ParamsInterface|cbpaidConfig $params
	 * @return mixed
	 */
	private function substituteText( $text, $cbUser, $params )
	{
		global $_CB_framework;

		$ignore		=	array();
		$ignoreId	=	0;

		if ( $text && ( strpos( $text, '[cbsubs:ignore' ) !== false ) ) {
			$text	=	preg_replace_callback( '%\[cbsubs:ignore\](.*?)\[/cbsubs:ignore\]%si', function( array $matches ) use ( &$ignore, &$ignoreId )
							{
								$ignoreId++;

								$ignore[$ignoreId]		=	$matches[1];

								return '[cbsubs:ignored ' . (int) $ignoreId . ']';
							},
							$text );
		}

		$text		=	$this->substituteCondition( $text, $cbUser, $params );

		if ( $text && ( strpos( $text, '[cbsubs:url' ) !== false ) ) {
			$text	=	preg_replace_callback( '%\[cbsubs:url plans=(?:"|&quot;)((?:(?!&quot;)[^"])+)(?:"|&quot;)(?: output=(?:"|&quot;)((?:(?!&quot;)[^"])+)(?:"|&quot;))? */\]%i', function( array $matches ) use ( $_CB_framework )
							{
								switch ( ( isset( $matches[2] )? $matches[2] : 'plans' ) ) {
									case 'basket':
										return $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'addplanstobasket', 'plans' => $matches[1] ), false );
										break;
									case 'plans':
										return $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'displayplans', 'plans' => $matches[1] ), false );
										break;
								}

								return null;
							},
							$text );
		}

		if ( $text && ( strpos( $text, '[cbsubs:plans' ) !== false ) ) {
			$that	=	$this;

			$text	=	preg_replace_callback( '%\[cbsubs:plans(?: user=(?:"|&quot;)((?:(?!&quot;)[^"/\[\] ])+)(?:"|&quot;))? plan=(?:"|&quot;)(\d+)(?:"|&quot;)(?: output=(?:"|&quot;)((?:(?!&quot;)[^"])+)(?:"|&quot;))? */\]%i', function( array $matches ) use ( $_CB_framework, $cbUser, $params, $that )
							{
								$user					=	$that->evaluateUserAttrib( $cbUser, $matches[1] )->getUserData();
								$userSubscriptions		=	$that->evaluateUserSubscriptions( $user );

								/** @var cbpaidProduct[] $plans */
								$plans					=	$userSubscriptions['plans'];

								if ( ! isset( $plans[$matches[2]] ) ) {
									return null;
								}

								$plan					=	$plans[$matches[2]];

								switch ( ( isset( $matches[3] )? $matches[3] : 'name' ) ) {
									case 'basket':
										return $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'addplanstobasket', 'plans' => (int) $plan->id ), false );
										break;
									case 'url':
										return $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'displayplans', 'plans' => (int) $plan->id ), false );
										break;
									case 'name':
										return CBPTXT::replaceUserVars( CBTxt::Th( $plan->name ), $user->get( 'id', 0, GetterInterface::INT ), true, false, $plan->get( 'runcontentplugins' ) );
										break;
									case 'alias':
										return CBTxt::Th( $plan->alias );
										break;
									case 'description':
										return CBPTXT::replaceUserVars( CBTxt::Th( $plan->description ), $user->get( 'id', 0, GetterInterface::INT ), true, false, $plan->get( 'runcontentplugins' ) );
										break;
									case 'duration':
										return $plan->getFormattedValidity( null, null, 'validity' );
										break;
									case 'price_registration':
										return $plan->displayPeriodPrice( $user->get( 'id', 0, GetterInterface::INT ) );
										break;
									case 'price_upgrade':
										return $plan->displayPeriodPrice( $user->get( 'id', 0, GetterInterface::INT ), 'U' );
										break;
									case 'price_renewal':
										return $plan->displayPeriodPrice( $user->get( 'id', 0, GetterInterface::INT ), 'R' );
										break;
									case 'price':
										if ( $user->get( 'id', 0, GetterInterface::INT ) ) {
											return $plan->displayPeriodPrice( $user->get( 'id', 0, GetterInterface::INT ), 'U' );
										} else {
											return $plan->displayPeriodPrice( $user->get( 'id', 0, GetterInterface::INT ) );
										}
										break;
									case 'rate':
										return cbpaidMoney::getInstance()->renderNumber( $plan->rate, 'money', true, true );
										break;
									case 'first_rate':
										return cbpaidMoney::getInstance()->renderNumber( ( $plan->first_different ? $plan->first_rate : $plan->rate ), 'money', true, true );
										break;
								}

								return null;
							},
							$text );
		}

		if ( $text && ( strpos( $text, '[cbsubs:subscriptions' ) !== false ) ) {
			$that	=	$this;

			$text	=	preg_replace_callback( '%\[cbsubs:subscriptions(?: user=(?:"|&quot;)((?:(?!&quot;)[^"/\[\] ])+)(?:"|&quot;))? plan=(?:"|&quot;)(\d+)(?:"|&quot;)(?: output=(?:"|&quot;)((?:(?!&quot;)[^"])+)(?:"|&quot;))?(?: status=(?:"|&quot;)(A|R|X|C|U|I)(?:"|&quot;))? */\]%i', function( array $matches ) use ( $_CB_framework, $cbUser, $params, $that )
							{
								$user					=	$that->evaluateUserAttrib( $cbUser, $matches[1] )->getUserData();
								$userSubscriptions		=	$that->evaluateUserSubscriptions( $user );

								/** @var cbpaidProduct[] $plans */
								$plans					=	$userSubscriptions['plans'];
								/** @var cbpaidUsersubscriptionRecord[] $subscriptions */
								$subscriptions			=	$userSubscriptions['subscriptions'];

								if ( ! isset( $plans[$matches[2]] ) ) {
									return null;
								}

								$plan					=	$plans[$matches[2]];
								$status					=	( isset( $matches[4] )? strtoupper( $matches[4] ) : null );
								$sub					=	null;

								foreach ( $subscriptions as $subscription ) {
									if ( ( $subscription->plan_id == $plan->id ) && ( ( ! $status ) || ( $status && ( $subscription->status == $status ) ) ) ) {
										$sub			=	$subscription;
									}
								}

								if ( ! $sub ) {
									return null;
								}

								switch ( ( isset( $matches[3] )? $matches[3] : 'validity' ) ) {
									case 'status':
										switch ( $sub->status ) {
											case 'A':
												return CBTxt::Th( 'Active' );
												break;
											case 'R':
												return CBTxt::Th( 'Registered Unpaid' );
												break;
											case 'X':
												return CBTxt::Th( 'Expired' );
												break;
											case 'C':
												return CBTxt::Th( 'Unsubscribed' );
												break;
											case 'U':
												return CBTxt::Th( 'Upgraded to other' );
												break;
											case 'I':
												return CBTxt::Th( 'Invalid' );
												break;
										}

										return null;
										break;
									case 'validity':
										return $sub->getFormattedValidityIfRenewed( $_CB_framework->now(), ( $status == 'C' ? 'N' : 'R' ), 1 );
										break;
									case 'validity_remaining':
										return $sub->getFormattedValidityRemaining();
										break;
									case 'expiry_date':
										$expiry			=	null;

										if ( $sub->expiry_date && ( $sub->expiry_date != '0000-00-00 00:00:00' ) ) {
											$expiry		=	cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $sub->expiry_date );
										} elseif ( $sub->isLifetimeValidity() ) {
											$expiry		=	CBTxt::T( $params->get( 'regtextLifetime', 'Lifetime Subscription' ) );
										}

										return $expiry;
										break;
									case 'signup_date':
										return ( $sub->subscription_date ? cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $sub->subscription_date ) : null );
										break;
									case 'last_renewal_date':
										return ( $sub->last_renewed_date ? cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $sub->last_renewed_date ) : null );
										break;
								}

								return null;
							},
							$text );
		}

		foreach ( $ignore as $id => $ignored ) {
			$text			=	str_replace( '[cbsubs:ignored ' . (int) $id . ']', $ignored, $text );
		}

		return $text;
	}
}

//TODO this is temporary as we don't have yet a general CB method:
if ( JFactory::getApplication()->getClientId() == 0 )
{
	// We do check for content restrictions only in frontend:
	define( '_CBSUBS_CONTENT_BOT_FILE', dirname( __FILE__ ) . DIRECTORY_SEPARATOR
	. 'plugin' . DIRECTORY_SEPARATOR
	. 'cbsubscontent' . DIRECTORY_SEPARATOR
	. 'cbsubs.content_access.php' );

	if ( is_readable( _CBSUBS_CONTENT_BOT_FILE ) ) {
		/** @noinspection PhpIncludeInspection */
		include_once _CBSUBS_CONTENT_BOT_FILE;
	}
}
