<?php
/**
 * @version $Id: cbpaidErrorHandler.php 1569 2012-12-23 01:35:57Z beat $
 * @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
 * @subpackage Plugin for Paid Subscriptions
 * @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
 * Specific CBSubs Error handling for logging purposes
 */
class cbpaidErrorHandler {
	/**
	 * @var int  To avoid initializing it more than once by error (shouldn't happen): Use ::install(), keep track of number of installs:
	 */
	protected static $handlerInitialized		=	0;
	/**
	 * @var boolean If handler is active or not: Use ::on() or ::off()
	 */
	protected static $handlerOff				=	true;
	protected static $previousErrorHandler		=	array();
	protected static $alwaysOn					=	false;
	/**
	 * Initialize the PHP error handler with the CBSubs callable handler
	 * Memorizes previous handler to restore it when calling ::uninstall()
	 *
	 * @return void
	 */
	public static function install( ) {
		self::$previousErrorHandler[self::$handlerInitialized++]	=	set_error_handler( array( __CLASS__, '_error_handler_callable' ) );
	}
	/**
	 * Restores the previous PHP error handler memorized when calling ::install()
	 *
	 * @return void
	 */
	public static function uninstall( ) {
		// Safeguard in case it's called too much:
		if ( self::$handlerInitialized && ! self::$alwaysOn ) {
			$previousHandler	=	self::$previousErrorHandler[--self::$handlerInitialized];

			// PHP <5.5 does not accept NULL to restore system handler, but as our error handler returns false when off, it goes to system still: So check for not null (fixes bug #3896):
			if ( $previousHandler !== null ) {
				set_error_handler( $previousHandler );
			}
		}
	}
	/**
	 * Turn error handling inside CBSubs ON
	 *
	 * @return void
	 */
	public static function on( ) {
		self::$handlerOff						=	false;
	}
	/**
	 * Turn error handling inside CBSubs OFF
	 *
	 * @return void
	 */
	public static function off( ) {
		if ( ! self::$alwaysOn ) {
			self::$handlerOff					=	true;
		}
	}
	/**
	 * Keep error handling inside CBSubs always ON until end of execution
	 * This is useful to catch any errors in raw mode, after e.g. an IPN
	 *
	 * @return void
	 */
	public static function keepTurnedOn( ) {
		self::$alwaysOn							=	true;
	}

	/**
	 * Returns a print of backtrace
	 *
	 * @param	array	$backtrace		return from debug_backtrace()
	 * @param	bool	$withArguments	outputs arguments
	 * @param	int     $maxCalls
	 * @return	string
	 */
	public static function printBacktrace( $backtrace, $withArguments, $maxCalls = 9999 )
	{
		// start backtrace:
		$trace						=	'';
		foreach ($backtrace as $v) {
			if (isset($v['class'])) {

				$trace				.=	'called in class '.$v['class'].'::'.$v['function'].'(';

				if ( $withArguments && isset( $v['args'] ) ) {
					$separator		=	'';

					foreach ( $v['args'] as $arg ) {
						$trace		.=	$separator . self::cbpaidGetArgument($arg);
						$separator	=	', ';
					}
				}
				$trace				.=	')';
				if ( isset( $v['line'] ) ) {
					$trace			.=	' on line ' . $v['line'];
				}
				if ( isset( $v['file'] ) ) {
					$trace			.=	' in file ' . substr( strrchr( $v['file'], '/' ), 1 );
				}

			} elseif (isset($v['function'])) {
				if ( strtolower( $v['function'] ) != strtolower( __FUNCTION__ ) ) {
					$trace			.=	'called in function '.$v['function'].'(';

					if ( $withArguments && !empty($v['args'] ) ) {
						$separator	=	'';

						foreach ( $v['args'] as $arg ) {
							$trace	.=	$separator . self::cbpaidGetArgument($arg);
							$separator = ', ';
						}
					}
					$trace			.=	')';
					if ( isset( $v['line'] ) ) {
						$trace		.=	' on line ' . $v['line'];
					}
					if ( isset( $v['file'] ) ) {
						$trace		.=	' in file ' . substr( strrchr( $v['file'], '/' ), 1 );
					}
				}
			} else {
				$trace				.=	'????';
				$trace				.=	'::::::' . var_export( $v, true );
			}
			$trace					.=	"\n";

			if ( ( --$maxCalls ) <= 0 ) {
				$trace				.=	"...\n";
				break;
			}
		}

		return $trace;
	}

	/**
	 * Error Handling function of CBSubs to give as argument for set_error_handler
	 * @deprecated : Use cbpaidErrorHandler::init() to set it.
	 *
	 * @param $errno
	 * @param string $errstr
	 * @param string $errfile
	 * @param string $errline
	 * @return bool
	 */
	public static function _error_handler_callable( $errno, $errstr = '', $errfile = '', $errline = '' )
	{
		if ( self::$handlerOff ) {
			return false;
		}

		// if error has been supressed with an @
		if ( error_reporting() == 0 ) {
			return false;
		}

		$errorPriority = array (	//(UNIX-type): 0: Emergency, 1: Alert, 2: Critical, 3: Error, 4: Warning, 5: Notice, 6: Info, 7: Debug
			E_ERROR             => 1,
			E_WARNING           => 4,
			E_PARSE             => 1,
			E_NOTICE            => 5,
			E_CORE_ERROR        => 1,
			E_CORE_WARNING      => 4,
			E_COMPILE_ERROR     => 1,
			E_COMPILE_WARNING   => 4,
			E_USER_ERROR        => 1,
			E_USER_WARNING      => 4,
			E_USER_NOTICE       => 5,
			E_STRICT	        => 6,
			E_RECOVERABLE_ERROR => 7,
			E_DEPRECATED    	=> 6,
			E_USER_DEPRECATED   => 7
		);

		if ( array_key_exists( $errno, $errorPriority ) ) {
			$priority	=	$errorPriority[$errno];
		} else {
			$priority	=	7;
		}

		global $_CB_framework;
		$debug			=	$_CB_framework->getCfg( 'debug' );
		$maxPriority	=	( $debug ? 6 : 5 );

		if ( $priority > $maxPriority ) {
			return false;
		}

		// check if function has been called by an exception
		if(func_num_args() == 5) {
			// PHP < 8.0: called by trigger_error()
			list($errno, $errstr, $errfile, $errline) = func_get_args();

			$backtrace = debug_backtrace();

		}else {
			// caught exception
			/** @var $exc Exception */
			$exc = func_get_arg(0);
			if ( is_object( $exc ) ) {
				$errno   = $exc->getCode();
				$errstr  = $exc->getMessage();
				$errfile = $exc->getFile();
				$errline = $exc->getLine();

				$backtrace = array_reverse( $exc->getTrace() );
			} else {
				$backtrace = debug_backtrace();
			}

		}

		$errorType = array (
			E_ERROR				=> 'ERROR',
			E_WARNING			=> 'WARNING',
			E_PARSE				=> 'PARSING ERROR',
			E_NOTICE			=> 'NOTICE',
			E_CORE_ERROR		=> 'CORE ERROR',
			E_CORE_WARNING		=> 'CORE WARNING',
			E_COMPILE_ERROR		=> 'COMPILE ERROR',
			E_COMPILE_WARNING	=> 'COMPILE WARNING',
			E_USER_ERROR     	=> 'USER ERROR',
			E_USER_WARNING   	=> 'USER WARNING',
			E_USER_NOTICE    	=> 'USER NOTICE',
			E_STRICT		 	=> 'STRICT NOTICE',
			E_RECOVERABLE_ERROR	=> 'E_RECOVERABLE_ERROR',
			E_DEPRECATED		=> 'E_DEPRECATED',
			E_USER_DEPRECATED	=> 'E_USER_DEPRECATED'
		);

		// create error message
		if (array_key_exists($errno, $errorType)) {
			$err = $errorType[$errno];
		} else {
			$err = 'CAUGHT EXCEPTION';
		}

		$errMsg = $err . ': ' . $errstr . ' in ' . $errfile . ' on line ' . $errline;

		$trace		=	static::printBacktrace( $backtrace, true );

		$trace		.=	'$_GET = ' . var_export( $_GET, true ) . "\n";
		$trace		.=	'$_POST = ' . var_export( $_POST, true ) . "\n";

		$errorText	=	$errMsg . "\n"
			.	'Trace:'
			.	$trace . "\n";

		if ( $debug ) {
			echo '<h2>CBPaid Debug Message</h2>'.nl2br($errMsg).'<br />
			Trace:'.nl2br($trace).'<br />';
		}

		if ( class_exists( '\Joomla\CMS\Log\Log' ) ) {
			// Add the short error message  without backtrace to the Joomla log, this also outputs to to debug bar's log in Joomla 4.0:
			\Joomla\CMS\Log\Log::add(
				$errMsg,
				\Joomla\CMS\Log\Log::WARNING,
				'log'
			);
		}

		$log			=	new cbpaidHistory();

		$errorTextForMe	=	( strpos( $errorText, 'cbpaidsubscriptions' ) !== false ) && ( strpos( $errorText, 'parainvite' ) === false );
		if ( $errorTextForMe ) {
			$log->logError( $priority, $errorText, null );
		}

		if ( $priority === 1 ) {
			// end and display error msg
			exit( self::cbDisplayClientMessage() );
		}

		return false;
	}

	/**
	 * Error handler: Displays a user fatal error without any translations
	 *
	 * @return null
	 */
	protected static function cbDisplayClientMessage()
	{
		echo 'Fatal Error please contact admin so he can check his history log for errors.';
		return null;
	}
	/**
	 * Error handler: Displays $arg according to his type
	 *
	 * @param  mixed $arg
	 * @return string
	 */
	protected static function cbpaidGetArgument($arg)
	{
		switch (strtolower(gettype($arg))) {

			case 'string':
				return( '"'.str_replace( array("\n"), array(''), $arg ).'"' );

			case 'boolean':
				return ( $arg ? 'true' : 'false' );

			case 'object':
				return 'object('.get_class($arg).')';

			case 'array':
				$ret = 'array(';
				$separator = '';

				foreach ($arg as $k => $v) {
					$ret .= $separator . self::cbpaidGetArgument($k).' => ' . self::cbpaidGetArgument($v);
					$separator = ', ';
				}
				$ret .= ')';

				return $ret;

			case 'resource':
				return 'resource('.get_resource_type($arg).')';

			default:
				return var_export($arg, true);
		}
	}
}
