<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2018 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\FamilyPlans;

use CB\Plugin\FamilyPlans\Table\FamilyTable;
use CBLib\Registry\GetterInterface;
use CB\Database\Table\UserTable;

defined('CBLIB') or die();

class Helper
{
	/**
	 * Returns the subscription object by id for the specified user
	 *
	 * @param int $userId
	 * @param int $subId
	 * @return null|\cbpaidSomething
	 */
	static public function getSubscription( $userId, $subId )
	{
		static $cache					=	array();

		if ( ( ! $userId ) || ( ! $subId ) ) {
			return null;
		}

		if ( ! isset( $cache[$userId][$subId] ) ) {
			$user						=	\CBuser::getUserDataInstance( $userId );
			$subscription				=	null;

			foreach ( \cbpaidSomethingMgr::getAllSomethingOfUser( $user, null, true ) as $subscriptions ) {
				if ( ! isset( $subscriptions[$subId] ) ) {
					continue;
				}

				$subscription			=	$subscriptions[$subId];

				break;
			}

			$cache[$userId][$subId]		=	$subscription;
		}

		return $cache[$userId][$subId];
	}

	/**
	 * Returns the subscription object by plan id for the specified user
	 *
	 * @param int $userId
	 * @param int $planId
	 * @return null|\cbpaidSomething
	 */
	static public function getPlanSubscription( $userId, $planId )
	{
		static $cache					=	array();

		if ( ( ! $userId ) || ( ! $planId ) ) {
			return null;
		}

		if ( ! isset( $cache[$userId][$planId] ) ) {
			$user						=	\CBuser::getUserDataInstance( $userId );
			$subscription				=	null;

			foreach ( \cbpaidSomethingMgr::getAllSomethingOfUser( $user, null, true ) as $subscriptions ) {
				foreach ( $subscriptions as $sub ) {
					if ( $sub->getPlanAttribute( 'id' ) != $planId ) {
						continue;
					}

					$subscription		=	$sub;

					break 2;
				}
			}

			$cache[$userId][$planId]	=	$subscription;
		}

		return $cache[$userId][$planId];
	}

	/**
	 * Checks if users family share create limit has been reached for the supplied subscription
	 *
	 * @param null|\cbpaidSomething $subscription
	 * @param null|UserTable        $user
	 * @return bool
	 */
	static public function getCreateLimited( $subscription, $user = null )
	{
		global $_CB_database;

		static $cache						=	array();

		if ( ! $subscription ) {
			return true;
		}

		if ( ! $user ) {
			$user							=	\CBuser::getMyUserDataInstance();
		}

		$userId								=	$user->get( 'id', 0, GetterInterface::INT );

		if ( ! $userId ) {
			return true;
		}

		$subId								=	$subscription->get( 'id', 0, GetterInterface::INT );

		if ( ( ! $subId ) || ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) ) ) {
			return true;
		}

		if ( ! isset( $cache[$userId][$subId] ) ) {
			$createLimit					=	$subscription->getPlan()->getParam( 'familyplans_create_limit', 'custom', 'integrations' );

			if ( $createLimit && ( $createLimit != 'custom' ) ) {
				$limitField					=	\CBuser::getInstance( (int) $userId, false )->getField( $createLimit, null, 'php', 'none', 'profile', 0, true );

				if ( is_array( $limitField ) ) {
					$createLimit			=	array_shift( $limitField );

					if ( is_array( $createLimit ) ) {
						$createLimit		=	implode( '|*|', $createLimit );
					}
				} else {
					$createLimit			=	$user->get( $createLimit, 0, GetterInterface::INT );
				}

				$createLimit				=	(int) $createLimit;
			} else {
				$createLimit				=	(int) $subscription->getPlan()->getParam( 'familyplans_create_limit_custom', 5, 'integrations' );
			}

			if ( $createLimit ) {
				$query						=	"SELECT COUNT(*)"
											.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' )
											.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $userId
											.	"\n AND " . $_CB_database->NameQuote( 'sub_id' ) . " = " . $subId;
				$_CB_database->setQuery( $query );

				if ( $_CB_database->loadResult() >= $createLimit ) {
					$createLimited			=	true;
				} else {
					$createLimited			=	false;
				}
			} else {
				$createLimited				=	false;
			}

			$cache[$userId][$subId]			=	$createLimited;
		}

		return $cache[$userId][$subId];
	}

	/**
	 * Sends shared subscription notification
	 *
	 * @param FamilyTable $row
	 * @param int|string  $to
	 * @param string      $subject
	 * @param string      $message
	 * @return bool
	 */
	static public function sendNotification( $row, $to, $subject, $message )
	{
		global $_CB_framework;

		if ( ( ! $to ) || ( ! $subject ) || ( ! $message ) ) {
			// Notification is not mandatory so if we're missing needed information just stop here:
			return true;
		}

		$extra				=	array(	'plan_id'					=>	$row->getOriginalSubscription()->get( 'plan_id', 0, GetterInterface::INT ),
										'replaced_plan_id'			=>	$row->getOriginalSubscription()->get( 'replaces_plan', 0, GetterInterface::INT ),
										'subscription_id'			=>	$row->getOriginalSubscription()->get( 'id', 0, GetterInterface::INT ),
										'parent_plan_id'			=>	$row->getOriginalSubscription()->get( 'parent_plan', 0, GetterInterface::INT ),
										'parent_subscription_id'	=>	$row->getOriginalSubscription()->get( 'parent_subscription', 0, GetterInterface::INT ),
										'plan_url'					=>	$_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'displayplans', 'plans' => $row->getOriginalSubscription()->get( 'plan_id', 0, GetterInterface::INT ) ), true ),
										'accept_url'				=>	$_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedaccept', 'code' => $row->get( 'code', null, GetterInterface::STRING ) ) ),
										'reject_url'				=>	$_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedreject', 'code' => $row->get( 'code', null, GetterInterface::STRING ) ) )
									);

		$subject			=	$row->getOriginalSubscription()->getPersonalized( $subject, false, false, $extra, false );
		$message			=	$row->getOriginalSubscription()->getPersonalized( $message, true, false, $extra, false );

		if ( $subject && $message ) {
			$notification	=	new \cbNotification();

			if ( ! $notification->sendFromSystem( $to, $subject, $message, false, true ) ) {
				return false;
			}
		}

		return true;
	}
}
