<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2018 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\FamilyPlans\Table;

use CB\Plugin\FamilyPlans\Helper;
use CBLib\Application\Application;
use CBLib\Database\Table\Table;
use CBLib\Registry\GetterInterface;
use CB\Database\Table\UserTable;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class FamilyTable extends Table
{
	/** @var int  */
	public $id				=	null;
	/** @var int  */
	public $user_id			=	null;
	/** @var int  */
	public $sub_id			=	null;
	/** @var string  */
	public $to				=	null;
	/** @var string  */
	public $code			=	null;
	/** @var string  */
	public $sent			=	null;
	/** @var string  */
	public $accepted		=	null;
	/** @var int  */
	public $user			=	null;

	/**
	 * Table name in database
	 * @var string
	 */
	protected $_tbl			=	'#__cbsubs_family';

	/**
	 * Primary key(s) of table
	 * @var string
	 */
	protected $_tbl_key		=	'id';

	/**
	 * @return bool
	 */
	public function check()
	{
		if ( ! $this->get( 'user_id', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( 'From user not specified!' ) );

			return false;
		}

		if ( ! \CBuser::getUserDataInstance( $this->get( 'user_id', 0, GetterInterface::INT ) )->get( 'id', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( 'From user does not exist!' ) );

			return false;
		}

		if ( ! $this->get( 'sub_id', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( 'Subscription not specified!' ) );

			return false;
		}

		$subscription	=	$this->getOriginalSubscription();

		if ( ! $subscription ) {
			$this->setError( CBTxt::T( 'Subscription does not exist!' ) );

			return false;
		}

		if ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) ) {
			$this->setError( CBTxt::T( 'This subscription plan does not allow sharing!' ) );

			return false;
		}

		if ( ! Application::Cms()->getClientId() ) {
			if ( $subscription->get( 'status', null, GetterInterface::STRING ) != 'A' ) {
				$this->setError( CBTxt::T( 'Can not share inactive subscription!' ) );

				return false;
			}

			if ( Helper::getCreateLimited( $subscription, \CBuser::getUserDataInstance( $this->get( 'user_id', 0, GetterInterface::INT ) ) ) ) {
				$this->setError( CBTxt::T( 'This subscription has reached the maximum number of times it can be shared!' ) );

				return false;
			}
		}

		if ( ( ! $this->get( 'to', null, GetterInterface::STRING ) ) && ( ! $this->get( 'user', 0, GetterInterface::INT ) ) ) {
			$this->setError( CBTxt::T( 'To email address not specified!' ) );

			return false;
		}

		if ( $this->get( 'to', null, GetterInterface::STRING ) && ( ! cbIsValidEmail( $this->get( 'to', null, GetterInterface::STRING ) ) ) ) {
			$this->setError( CBTxt::T( 'To address not a valid email address!' ) );

			return false;
		}

		if ( $this->get( 'user', 0, GetterInterface::INT ) ) {
			if ( ! \CBuser::getUserDataInstance( $this->get( 'user', 0, GetterInterface::INT ) )->get( 'id', 0, GetterInterface::INT ) ) {
				$this->setError( CBTxt::T( 'Accepting user does not exist!' ) );

				return false;
			}

			if ( $this->get( 'user_id', 0, GetterInterface::INT ) == $this->get( 'user', 0, GetterInterface::INT ) ) {
				$this->setError( CBTxt::T( 'Accepting user can not be the from user!' ) );

				return false;
			}

			$userSubscription	=	Helper::getPlanSubscription( $this->get( 'user', 0, GetterInterface::INT ), $subscription->getPlanAttribute( 'id' ) );

			if ( $userSubscription
				 && ( $userSubscription->get( 'status', null, GetterInterface::STRING ) == 'A' )
				 && ( $userSubscription->get( 'user_id', 0, GetterInterface::INT ) == $this->get( 'user', 0, GetterInterface::INT ) )
			) {
				$this->setError( CBTxt::T( 'Accepting user already has this subscription!' ) );

				return false;
			}
		}

		return true;
	}

	/**
	 * Returns the original subscription based off the subscription id
	 *
	 * @return null|\cbpaidSomething
	 */
	public function getOriginalSubscription()
	{
		if ( ( ! $this->get( 'user_id', 0, GetterInterface::INT ) ) || ( ! $this->get( 'sub_id', 0, GetterInterface::INT ) ) ) {
			return null;
		}

		$subscription	=	Helper::getSubscription( $this->get( 'user_id', 0, GetterInterface::INT ), $this->get( 'sub_id', 0, GetterInterface::INT ) );

		if ( ! $subscription ) {
			return null;
		}

		return $subscription;
	}

	/**
	 * Returns the pseudo subscription based off the subscription id
	 *
	 * @return null|\cbpaidSomething
	 */
	public function getSubscription()
	{
		if ( ( ! $this->get( 'user_id', 0, GetterInterface::INT ) ) || ( ! $this->get( 'sub_id', 0, GetterInterface::INT ) ) ) {
			return null;
		}

		$subscription				=	$this->getOriginalSubscription();

		if ( ! $subscription ) {
			return null;
		}

		if ( $subscription instanceof \cbpaidMerchandiseRecord ) {
			$pseudoSubscription		=	new PseudoMerchandiseTable();
		} elseif ( $subscription instanceof \cbpaidDonationRecord ) {
			$pseudoSubscription		=	new PseudoDonationTable();
		} else {
			$pseudoSubscription		=	new PseudoSubscriptionTable();
		}

		$pseudoSubscription->bind( $subscription );

		// Replace IDs so we can't perform any actions against this objects IDs:
		$pseudoSubscription->set( '_id', $pseudoSubscription->get( 'id', 0, GetterInterface::INT ) );
		$pseudoSubscription->set( '_parent_subscription', $pseudoSubscription->get( 'parent_subscription', 0, GetterInterface::INT ) );
		$pseudoSubscription->set( 'id', 0 );
		$pseudoSubscription->set( 'parent_subscription', 0 );

		if ( $pseudoSubscription instanceof PseudoSubscriptionTable ) {
			$pseudoSubscription->set( '_replaces_subscription', $pseudoSubscription->get( 'replaces_subscription', 0, GetterInterface::INT ) );
			$pseudoSubscription->set( 'replaces_subscription', 0 );
		}

		// Now lets pretend the accepting user is the owner of this subscription:
		$pseudoSubscription->set( '_user_id', $pseudoSubscription->get( 'user_id', 0, GetterInterface::INT ) );
		$pseudoSubscription->set( 'user_id', $this->get( 'user', 0, GetterInterface::INT ) );

		return $pseudoSubscription;
	}

	/**
	 * @param bool $updateNulls
	 * @return bool
	 */
	public function store( $updateNulls = false )
	{
		global $_PLUGINS;

		$new						=	( $this->get( 'id', 0, GetterInterface::INT ) ? false : true );

		$this->set( 'code', $this->get( 'code', md5( uniqid( $this->get( 'user_id', 0, GetterInterface::INT ) . $this->get( 'sub_id', 0, GetterInterface::INT ) ) ), GetterInterface::STRING ) );

		if ( ! parent::store( $updateNulls ) ) {
			return false;
		}

		if ( $new ) {
			// If this is a new share being stored we either need to activate the subscription if a user has been supplied (e.g. backend case) or send email to the recipient user:
			if ( $this->get( 'user', 0, GetterInterface::INT ) ) {
				$user				=	\CBuser::getUserDataInstance( $this->get( 'user', 0, GetterInterface::INT ) );

				if ( $user->get( 'id', 0, GetterInterface::INT ) ) {
					$subscription	=	$this->getSubscription();

					if ( $subscription && ( $subscription->get( 'status', null, GetterInterface::STRING ) == 'A' ) ) {
						$paidUserExtension	=	\cbpaidUserExtension::getInstance( $user->get( 'id', 0, GetterInterface::INT ) );

						$paidUserExtension->getUserSubscriptions( 'clearcache' );

						$subscription->triggerIntegrations( $user, 'PaidSubscription', null, null, 'N', 0 );

						if ( $subscription instanceof PseudoSubscriptionTable ) {
							$null			=	null;

							$paidUserExtension->checkUserSubscriptions( true, $null, 'N' );
						}

						$_PLUGINS->trigger( 'familyplans_onAfterAcceptSharedSub', array( $this, $user, $subscription ) );
					}
				}
			} elseif ( $this->get( 'to', null, GetterInterface::STRING ) ) {
				$this->send();
			}
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function canDelete( $id = null )
	{
		if ( ! parent::canDelete( $id ) ) {
			return false;
		}

		if ( ! Application::Cms()->getClientId() ) {
			$userId		=	Application::MyUser()->getUserId();

			if ( ( $userId == $this->get( 'user_id', 0, GetterInterface::INT ) )
				 && ( $userId == $this->get( 'user', 0, GetterInterface::INT ) )
				 && ( ! \cbpaidApp::authoriseAction( 'cbsubs.usersubscriptionmanage' ) )
			) {
				return false;
			}
		}

		return true;
	}

	/**
	 * @param null|int $id
	 * @return bool
	 */
	public function delete( $id = null )
	{
		global $_PLUGINS;

		if ( ! parent::delete( $id ) ) {
			return false;
		}

		$user					=	\CBuser::getUserDataInstance( $this->get( 'user', 0, GetterInterface::INT ) );

		if ( $user->get( 'id', 0, GetterInterface::INT ) ) {
			// If the shared subscription is being deleted we need to be sure the subscription deactivation is triggered for the recipient user (if there is one) for plan behavior reversals:
			$subscription		=	$this->getSubscription();

			if ( ! $subscription ) {
				return true;
			}

			$userSubscription	=	Helper::getPlanSubscription( $user->get( 'id', 0, GetterInterface::INT ), $subscription->getPlanAttribute( 'id' ) );

			if ( ( ! $userSubscription ) || ( $userSubscription->get( 'status', null, GetterInterface::STRING ) != 'A' ) ) {
				// Only trigger the deactivation behavior if they're not actively subscribed to the plan (e.g. they subscribed to it themselves):
				if ( $subscription->get( 'status', null, GetterInterface::STRING ) == 'A' ) {
					$paidUserExtension	=	\cbpaidUserExtension::getInstance( $user->get( 'id', 0, GetterInterface::INT ) );

					$paidUserExtension->getUserSubscriptions( 'clearcache' );

					$subscription->set( 'status', 'X' );

					$subscription->triggerIntegrations( $user, 'SubscriptionDeactivated', null, null, null, 0 );

					if ( $subscription instanceof PseudoSubscriptionTable ) {
						$paidUserExtension->checkUserSubscriptions( false, $subscription, 'X' );
					}
				}
			}

			if ( ! Application::Cms()->getClientId() ) {
				if ( Application::MyUser()->getUserId() == $this->get( 'user', 0, GetterInterface::INT ) ) {
					// Let the subscription owner know the recipient has cancelled:
					$subject	=	$subscription->getPlan()->getParam( 'familyplans_notify_cancel_sub', null, 'integrations' );
					$message	=	$subscription->getPlan()->getParam( 'familyplans_notify_cancel_msg', null, 'integrations' );

					Helper::sendNotification( $this, $this->get( 'user_id', 0, GetterInterface::INT ), $subject, $message );
				} else {
					// Let the recipient know the shared subscription has been deleted:
					$subject	=	$subscription->getPlan()->getParam( 'familyplans_notify_delete_sub', null, 'integrations' );
					$message	=	$subscription->getPlan()->getParam( 'familyplans_notify_delete_msg', null, 'integrations' );

					Helper::sendNotification( $this, $this->get( 'user', 0, GetterInterface::INT ), $subject, $message );
				}
			}

			$_PLUGINS->trigger( 'familyplans_onAfterDeleteSharedSub', array( $this, $user, $subscription ) );
		}

		return true;
	}

	/**
	 * Sends the shared subscription email
	 *
	 * @param bool $resend
	 * @return bool
	 */
	public function send( $resend = false )
	{
		$subscription	=	$this->getSubscription();

		if ( ! $subscription ) {
			return false;
		}

		$lastSent		=	$this->get( 'sent', null, GetterInterface::STRING );

		if ( $lastSent ) {
			if ( ! $resend ) {
				$this->setError( CBTxt::T( 'This shared subscription has already been sent!' ) );

				return false;
			} else {
				$resendDelay	=	(int) $subscription->getPlan()->getParam( 'familyplans_notify_share_sub', 15, 'integrations' );

				if ( $resendDelay && ( floor( ( Application::Date( 'now', 'UTC' )->getTimestamp() - Application::Date( $lastSent, 'UTC' )->getTimestamp() ) / 60 ) <= $resendDelay ) ) {
					$this->setError( CBTxt::T( 'This shared subscription has already been sent recently. Please retry later.' ) );

					return false;
				}
			}
		}

		$subject		=	$subscription->getPlan()->getParam( 'familyplans_notify_share_sub', 'Shared Subscription', 'integrations' );
		$message		=	$subscription->getPlan()->getParam( 'familyplans_notify_share_msg', '[cb:userfield field="formatname" reason="list" /] has shared their <a href="[plan_url]">[PLAN_NAME]</a> subscription with you. <a href="[accept_url]">Click here if you would like to accept this shared subscription.</a> If you do not want this shared subscription or believe this was done in error you may <a href="[reject_url]">click here to reject this shared subscription</a>.', 'integrations' );

		if ( ! Helper::sendNotification( $this, $this->get( 'to', null, GetterInterface::STRING ), $subject, $message ) ) {
			return false;
		}

		$this->set( 'sent', Application::Database()->getUtcDateTime() );

		if ( ! $this->store() ) {
			return false;
		}

		return true;
	}

	/**
	 * Checks if the supplied user can accept the shared subscription
	 *
	 * @param int $userId
	 * @return bool
	 */
	public function canAccept( $userId = 0 )
	{
		global $_PLUGINS;

		if ( $this->get( 'user', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( 'This shared subscription has already been accepted!' ) );

			return false;
		}

		if ( $userId == $this->get( 'user_id', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( ' You can not accept a shared subscription from yourself!' ) );

			return false;
		}

		$subscription	=	$this->getSubscription();

		if ( ! $subscription ) {
			$this->setError( CBTxt::T( 'Shared subscription does not exist!' ) );

			return false;
		}

		if ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) ) {
			$this->setError( CBTxt::T( 'This shared subscriptions plan does not allow sharing!' ) );

			return false;
		}

		if ( $subscription->get( 'status', null, GetterInterface::STRING ) != 'A' ) {
			$this->setError( CBTxt::T( 'You can not accept an inactive subscription!' ) );

			return false;
		}

		if ( $userId ) {
			$userSubscription	=	Helper::getPlanSubscription( $userId, $subscription->getPlanAttribute( 'id' ) );

			if ( $userSubscription
				 && ( $userSubscription->get( 'status', null, GetterInterface::STRING ) == 'A' )
				 && ( $userSubscription->get( 'user_id', 0, GetterInterface::INT ) == $userId )
			) {
				$this->setError( CBTxt::T( 'You can not accept a shared subscription for a plan you are already subscribed to!' ) );

				return false;
			}
		}

		if ( in_array( false, $_PLUGINS->trigger( 'familyplans_onCanAcceptSharedSub', array( &$this, $userId, $subscription ) ), true ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Accepts the shared subscription for $user
	 *
	 * @param UserTable $user
	 * @return bool
	 */
	public function accept( $user )
	{
		global $_PLUGINS;

		if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
			$this->setError( CBTxt::T( 'Accepting user not specified!' ) );

			return false;
		}

		$subscription		=	$this->getSubscription();

		if ( ! $subscription ) {
			$this->setError( CBTxt::T( 'Shared subscription does not exist!' ) );

			return false;
		}

		$this->set( 'user', $user->get( 'id', 0, GetterInterface::INT ) );
		$this->set( 'accepted', Application::Database()->getUtcDateTime() );

		if ( in_array( false, $_PLUGINS->trigger( 'familyplans_onBeforeAcceptSharedSub', array( &$this, $user, $subscription ) ), true ) ) {
			return false;
		}

		if ( ! $this->store() ) {
			return false;
		}

		$paidUserExtension	=	\cbpaidUserExtension::getInstance( $user->get( 'id', 0, GetterInterface::INT ) );

		$paidUserExtension->getUserSubscriptions( 'clearcache' );

		$subscription->triggerIntegrations( $user, 'PaidSubscription', null, null, 'N', 0 );

		if ( $subscription instanceof PseudoSubscriptionTable ) {
			$null			=	null;

			$paidUserExtension->checkUserSubscriptions( true, $null, 'N' );
		}

		$subject			=	$subscription->getPlan()->getParam( 'familyplans_notify_accept_sub', null, 'integrations' );
		$message			=	$subscription->getPlan()->getParam( 'familyplans_notify_accept_msg', null, 'integrations' );

		Helper::sendNotification( $this, $this->get( 'user_id', 0, GetterInterface::INT ), $subject, $message );

		$_PLUGINS->trigger( 'familyplans_onAfterAcceptSharedSub', array( $this, $user, $subscription ) );

		return true;
	}
}