<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\PlanOptions\Entity;

use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use cbpaidProduct;
use cbpaidSomething;
use moscomprofilerHTML;

\defined('CBLIB') or die();

class OptionEntity
{
	/** @var cbpaidProduct */
	protected cbpaidProduct $plan;
	/** @var ParamsInterface */
	protected ParamsInterface $option;
	/** @var null|cbpaidSomething */
	protected ?cbpaidSomething $subscription	=	null;
	/** @var null|PriceEntity */
	protected ?PriceEntity $selectedPlanPrice	=	null;

	/**
	 * @param cbpaidProduct        $plan
	 * @param ParamsInterface      $option
	 * @param null|cbpaidSomething $subscription
	 * @param null|PriceEntity     $selectedPlanPrice
	 */
	public function __construct( cbpaidProduct $plan, ParamsInterface $option, ?cbpaidSomething $subscription = null, ?PriceEntity $selectedPlanPrice = null )
	{
		$this->plan					=	$plan;
		$this->option				=	$option;
		$this->subscription			=	$subscription;
		$this->selectedPlanPrice	=	$selectedPlanPrice;
	}

	/**
	 * @return cbpaidProduct
	 */
	public function getPlan(): cbpaidProduct
	{
		return $this->plan;
	}

	/**
	 * @return null|cbpaidSomething
	 */
	public function getSubscription(): ?cbpaidSomething
	{
		return $this->subscription;
	}

	/**
	 * @return null|PriceEntity
	 */
	public function getSelectedPlanPrice(): ?PriceEntity
	{
		if ( ! $this->selectedPlanPrice ) {
			return null;
		}

		return $this->selectedPlanPrice;
	}

	/**
	 * @return string
	 */
	public function getName(): string
	{
		return $this->option->getString( 'name', '' );
	}

	/**
	 * @return string
	 */
	public function getType(): string
	{
		return $this->option->getString( 'type', 'select' );
	}

	/**
	 * @return string
	 */
	public function getTitle(): string
	{
		if ( $this->subscription ) {
			return $this->subscription->getPersonalized( $this->option->getString( 'title', '' ), false, false, null, false );
		}

		return $this->plan->getPersonalized( $this->option->getString( 'title', '' ), Application::MyUser()->getUserId(), false, false, null, false );
	}

	/**
	 * @return string
	 */
	public function getBlankText(): string
	{
		if ( $this->subscription ) {
			return $this->subscription->getPersonalized( $this->option->getString( 'blanktext', '' ), false, false, null, false );
		}

		return $this->plan->getPersonalized( $this->option->getString( 'blanktext', '' ), Application::MyUser()->getUserId(), false, false, null, false );
	}

	/**
	 * @return string
	 */
	public function getDefaultValue(): string
	{
		$default	=	$this->option->getString( 'default', '' );

		if ( ! $this->hasValue( $default ) ) {
			return '';
		}

		return $default;
	}

	/**
	 * @return int
	 */
	public function getDisplay(): int
	{
		return $this->option->getInt( 'display', 1 );
	}

	/**
	 * @return bool
	 */
	public function isRequired(): bool
	{
		return $this->option->getBool( 'required', false );
	}

	/**
	 * @return ValueEntity[]
	 */
	public function getValues(): array
	{
		$values				=	[];

		foreach ( $this->option->subTree( 'values' ) as $value ) {
			/** @var ParamsInterface $value */
			$name			=	$value->getString( 'value', '' );

			if ( $name === '' ) {
				continue;
			}

			$values[$name]	=	new ValueEntity( $this, $value );
		}

		return $values;
	}

	/**
	 * @param string $value
	 * @return bool
	 */
	public function hasValue( string $value = '' ): bool
	{
		if ( $value === '' ) {
			return false;
		}

		return \array_key_exists( $value, $this->getValues() );
	}

	/**
	 * @param string $value
	 * @return null|ValueEntity
	 */
	public function getValue( string $value = '' ): ?ValueEntity
	{
		if ( ! $this->hasValue( $value ) ) {
			return null;
		}

		return $this->getValues()[$value];
	}

	/**
	 * @return string
	 */
	public function getInputName(): string
	{
		return 'planoptions[' . $this->plan->getInt( 'id', 0 ) . '][' . $this->getName() . ']';
	}

	/**
	 * @return string
	 */
	public function getInputId(): string
	{
		return moscomprofilerHTML::htmlId( $this->getInputName() );
	}

	/**
	 * @return string
	 */
	public function renderInput(): string
	{
		$selected			=	$this->getDefaultValue();

		if ( $this->subscription ) {
			$selected		=	$this->subscription->getParams( 'integrations' )->getRaw( 'planoptions.' . $this->getName(), '' );
		}

		$valueAttr			=	'';

		if ( \in_array( $this->getType(), [ 'checkbox_buttons', 'radio_buttons' ], true ) ) {
			$valueAttr		=	'class="' . ( $this->isRequired() ? ( $this->subscription ? 'required ' : 'cbsubsPlanOptionRequired ' ) : '' ) . 'cbsubsPlanOptionsSelect"';
		}

		$isGrid				=	( \in_array( $this->getType(), [ 'radio_grid', 'checkbox_grid' ], true ) );
		$values				=	[];

		if ( $this->getBlankText() ) {
			$label			=	$this->getBlankText();

			if ( $isGrid ) {
				$label		=	'<div class="cbsubsPlanOptionGridLabel">' . $label . '</div>';
			}

			$values[]		=	moscomprofilerHTML::makeOption( '', $label, 'value', 'text', null, null, $valueAttr );
		}

		foreach ( $this->getValues() as $value ) {
			$label			=	$value->renderOption();

			if ( $isGrid ) {
				$label		=	'<div class="cbsubsPlanOptionGridLabel">' . $value->getLabel() . '</div>'
							.	'<div class="cbsubsPlanOptionGridPrice">' . $value->renderOptionPrice() . '</div>';
			}

			$values[]		=	moscomprofilerHTML::makeOption( $value->getValue(), $label, 'value', 'text', null, null, $valueAttr );
		}

		switch ( $this->getType() ) {
			case 'radio':
			case 'radio_grid':
			case 'checkbox':
			case 'checkbox_grid':
				$classes			=	[ 'cbsubsPlanOptionsSelect' ];

				if ( $this->isRequired() ) {
					$classes[]		=	'cbsubsPlanOptionRequired';

					if ( $this->subscription ) {
						$classes[]	=	'required';
					}
				}

				if ( \in_array( $this->getType(), [ 'checkbox', 'checkbox_grid' ], true ) ) {
					$input			=	moscomprofilerHTML::checkboxListTable( $values, $this->getInputName() . '[]', '', 'value', 'text', $selected, 0, 0, 0, false, $classes, '', false );
				} else {
					$input			=	moscomprofilerHTML::radioListTable( $values, $this->getInputName(), '', 'value', 'text', $selected, 0, 0, 0, false, $classes, '', false );
				}

				if ( $isGrid ) {
					$input			=	'<div class="cbsubsPlanOptionGrid">' . $input . '</div>';
				}

				return '<div class="form-control-plaintext">' . $input . '</div>';
			case 'radio_buttons':
				return moscomprofilerHTML::radioListButtons( $values, $this->getInputName(), null, 'value', 'text', $selected, false, null, null, false );
			case 'checkbox_buttons':
				return moscomprofilerHTML::checkboxListButtons( $values, $this->getInputName() . '[]', null, 'value', 'text', $selected, false, null, null, false );
			default:
				return moscomprofilerHTML::selectList( $values, $this->getInputName(), 'class="w-auto form-control' . ( $this->isRequired() ? ( $this->subscription ? ' required' : ' cbsubsPlanOptionRequired' ) : '' ) . ' cbsubsPlanOptionsSelect"', 'value', 'text', $selected, $this->isRequired(), false, false );
		}
	}
}