<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\PlanOptions\Entity;

use CB\Plugin\PlanOptions\Helper;
use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;

\defined('CBLIB') or die();

class ValueEntity
{
	/** @var OptionEntity */
	protected OptionEntity $option;
	/** @var ParamsInterface */
	protected ParamsInterface $value;
	/** @var array */
	protected array $price;

	/**
	 * @param OptionEntity    $option
	 * @param ParamsInterface $value
	 */
	public function __construct( OptionEntity $option, ParamsInterface $value )
	{
		$this->option	=	$option;
		$this->value	=	$value;
		$this->price	=	[];

		if ( $this->option->getSubscription() ) {
			$price		=	$this->option->getSubscription()->getPersonalized( $value->getString( 'price', '' ), false, false, null, false );
		} else {
			$price		=	$this->option->getPlan()->getPersonalized( $value->getString( 'price', '' ), Application::MyUser()->getUserId(), false, false, null, false );
		}

		preg_match( '#([+*/-])?([\d.]+)(%)?$#', $price, $this->price );
	}

	/**
	 * @return OptionEntity
	 */
	public function getOption(): OptionEntity
	{
		return $this->option;
	}

	/**
	 * @return float
	 */
	protected function getPlanRate(): float
	{
		$selected	=	$this->option->getSelectedPlanPrice();

		if ( $selected ) {
			return $selected->getRate();
		}

		return Helper::getBasePlanPrice( $this->option->getPlan(), $this->option->getSubscription() )['rate'];
	}

	/**
	 * @return float
	 */
	protected function getPlanFirstRate(): float
	{
		if ( $this->option->getPlan()->getInt( 'first_different', 0 ) ) {
			$selected	=	$this->option->getSelectedPlanPrice();

			if ( $selected ) {
				return $selected->getFirstRate();
			}

			return Helper::getBasePlanPrice( $this->option->getPlan(), $this->option->getSubscription() )['first_rate'];
		}

		return $this->getPlanRate();
	}

	/**
	 * @return string
	 */
	public function getValue(): string
	{
		return $this->value->getString( 'value', '' );
	}

	/**
	 * @return string
	 */
	public function getLabel(): string
	{
		if ( $this->option->getSubscription() ) {
			return $this->option->getSubscription()->getPersonalized( $this->value->getString( 'label', '' ), false, false, null, false );
		}

		return $this->option->getPlan()->getPersonalized( $this->value->getString( 'label', '' ), Application::MyUser()->getUserId(), false, false, null, false );
	}

	/**
	 * @return string
	 */
	public function getModifier(): string
	{
		return ( $this->price[1] ?? '' );
	}

	/**
	 * @return float
	 */
	public function getAmount(): float
	{
		return (float) ( $this->price[2] ?? 0.0 );
	}

	/**
	 * @return bool
	 */
	public function isPercent(): bool
	{
		return ( ( $this->price[3] ?? '' ) === '%' );
	}

	/**
	 * @return float
	 */
	public function getRate(): float
	{
		return $this->getModifiedRate( $this->getAmount(), $this->getPlanRate() );
	}

	/**
	 * @return float
	 */
	public function getFirstRate(): float
	{
		return $this->getModifiedRate( $this->getAmount(), $this->getPlanFirstRate() );
	}

	/**
	 * @param float $amount
	 * @param float $rate
	 * @return float
	 */
	protected function getModifiedRate( float $amount, float $rate ): float
	{
		if ( $this->isPercent() ) {
			$amount	=	( $rate * ( $amount / 100 ) );
		}

		switch ( $this->getModifier() ) {
			case '+':
				return ( $rate + $amount );
			case '-':
				return ( $rate - $amount );
			case '*':
				return ( $rate * $amount );
			case '/':
				return ( $rate / $amount );
		}

		return $amount;
	}

	/**
	 * @return string
	 */
	public function renderPrice(): string
	{
		if ( $this->option->getSubscription() ) {
			if ( ( ! $this->option->getSubscription()->getOccurrence() ) && $this->option->getPlan()->getInt( 'first_different', 0 ) && ( $this->getRate() !== $this->getFirstRate() ) ) {
				$price	=	$this->option->getPlan()->renderPrice( $this->getFirstRate() ) . CBTxt::T( ', then ' ) . $this->option->getPlan()->renderPrice( $this->getRate() );
			} else {
				$price	=	$this->option->getPlan()->renderPrice( $this->getRate() );
			}
		} elseif ( $this->option->getPlan()->getInt( 'first_different', 0 ) && ( $this->getRate() !== $this->getFirstRate() ) ) {
			$price		=	$this->option->getPlan()->renderPrice( $this->getFirstRate() ) . CBTxt::T( ', then ' ) . $this->option->getPlan()->renderPrice( $this->getRate() );
		} else {
			$price		=	$this->option->getPlan()->renderPrice( $this->getRate() );
		}

		return trim( \strip_tags( $price ) );
	}

	/**
	 * @return string
	 */
	public function renderRelativePrice(): string
	{
		if ( ! $this->getModifier() ) {
			return '';
		}

		if ( ! $this->getAmount() ) {
			return $this->option->getPlan()->renderPrice( $this->getAmount() );
		}

		if ( $this->isPercent() ) {
			return CBTxt::T( 'PLAN_OPTIONST_RELATIVE_PRICE_PERCENT', '[modifier][percent]%', [ '[modifier]' => $this->getModifier(), '[percent]' => $this->getAmount() ] );
		}

		return CBTxt::T( 'PLAN_OPTIONST_RELATIVE_PRICE', '[modifier][amount]', [ '[modifier]' => $this->getModifier(), '[amount]' => $this->option->getPlan()->renderPrice( $this->getAmount() ) ] );
	}

	/**
	 * @return string
	 */
	public function renderOptionPrice(): string
	{
		if ( $this->option->getSubscription() ) {
			if ( $this->getModifier() && Helper::hasPlanPrices( $this->option->getPlan() ) && $this->option->getPlan()->getParam( 'options_prices_changeable', 1, 'integrations' ) ) {
				return $this->renderRelativePrice();
			}
		} elseif ( $this->getModifier() && Helper::hasPlanPrices( $this->option->getPlan() ) ) {
			return $this->renderRelativePrice();
		}

		return $this->renderPrice();
	}

	/**
	 * @return string
	 */
	public function renderAmount(): string
	{
		return trim( \strip_tags( $this->option->getPlan()->renderPrice( $this->getAmount() ) ) );
	}

	/**
	 * @return string
	 */
	public function renderLabel(): string
	{
		$key	=	strtoupper( preg_replace( '/[^-a-zA-Z0-9_]/', '', $this->getValue() ) );

		// CBTxt::T( 'PLAN_OPTIONS_SELECTED', '[label] – [price]' )
		return CBTxt::T( 'PLAN_OPTIONS_SELECTED_' . $key . ' PLAN_OPTIONS_SELECTED', '[label] – [price]', [ '[label]' => $this->getLabel(), '[price]' => $this->renderPrice(), '[price_relative]' => $this->renderRelativePrice(), '[modifier]' => $this->getModifier(), '[amount]' => $this->renderAmount() ] );
	}

	/**
	 * @return string
	 */
	public function renderOption(): string
	{
		$key			=	strtoupper( preg_replace( '/[^-a-zA-Z0-9_]/', '', $this->getValue() ) );

		// CBTxt::T( 'PLAN_OPTIONS_SELECT', '[label] – [price]' )
		return CBTxt::T( 'PLAN_OPTIONS_SELECT_' . $key . ' PLAN_OPTIONS_SELECT', '[label] – [price]', [ '[label]' => $this->getLabel(), '[price]' => $this->renderOptionPrice(), '[price_relative]' => $this->renderRelativePrice(), '[modifier]' => $this->getModifier(), '[amount]' => $this->renderAmount() ] );
	}
}