<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\PlanOptions\Trigger;

use CB\Database\Table\UserTable;
use CB\Plugin\PlanOptions\Entity\PriceEntity;
use CB\Plugin\PlanOptions\Helper;
use CB\Plugin\PlanOptions\TemplateHandler;
use CBLib\Registry\ParamsInterface;
use cbpaidPaymentBasket;
use cbpaidPaymentItem;
use cbpaidPaymentNotification;
use cbpaidProduct;
use cbpaidSomething;

\defined('CBLIB') or die();

class BasketTrigger extends \cbPluginHandler
{
	/**
	 * Handles rendering of selected pricing options in basket and invoice
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param cbpaidPaymentItem   $item
	 * @param null|string         $column
	 * @param string              $variable
	 * @param string              $output
	 * @return void
	 */
	public function onCPayRenderBasketItem( cbpaidPaymentBasket $paymentBasket, cbpaidPaymentItem $item, ?string &$column, string $variable, string $output ): void
	{
		if ( ( $variable !== 'description' ) || ( $output !== 'html' ) ) {
			return;
		}

		$integrationParams	=	$paymentBasket->getParams( 'integrations' );

		if ( ! $integrationParams->has( 'planoptions' ) ) {
			return;
		}

		$subscription		=	$item->loadSubscription();

		if ( ! $subscription ) {
			return;
		}

		$plan				=	$subscription->getPlan();

		if ( $paymentBasket->getString( 'proformainvoice', '' ) ) {
			if ( ( ! $plan->getParam( 'options_display_invoice', 1, 'integrations' ) ) || ( ! Helper::getPaidBasketValues( $paymentBasket, $subscription ) ) ) {
				return;
			}
		} elseif ( ( ! $plan->getParam( 'options_display_basket', 1, 'integrations' ) ) || ( ! Helper::getBasketValues( $paymentBasket, $subscription, Helper::getBasketPlanPrice( $paymentBasket, $subscription ) ) ) ) {
			return;
		}

		/** @var $viewer \cbpaidPlanOptionsView */
		$viewer				=	TemplateHandler::getViewer( null, 'planoptions' );

		$column				.=	$viewer->drawBasket( $subscription, $paymentBasket );
	}

	/**
	 * Handles promoting pricing option selection to basket parameters
	 * Also handles re-applying pricing options on basket update from basket parameters
	 *
	 * @param string                   $event
	 * @param cbpaidSomething          $subscription
	 * @param null|cbpaidPaymentBasket $paymentBasket
	 * @return void
	 */
	public function onCPayPaymentItemEvent( string $event, cbpaidSomething $subscription, ?cbpaidPaymentBasket $paymentBasket = null ): void
	{
		if ( ! \in_array( $event, [ 'beforeUpdatePaymentItem', 'addSomethingToBasket' ], true ) ) {
			return;
		}

		$plan					=	$subscription->getPlan();

		Helper::setBasePlanPrice( $plan, $subscription );

		if ( ( ! Helper::hasPlanPrices( $plan ) ) && ( ! Helper::hasPlanOptions( $plan ) ) ) {
			return;
		}

		if ( $event === 'beforeUpdatePaymentItem' ) {
			// Update the subscriptions plan object pricing from the basket parameters
			$selected			=	Helper::getBasketPlanPrice( $paymentBasket, $subscription );

			if ( $selected ) {
				Helper::setPlanOverrides( $plan, $selected );
			}

			Helper::setPlanOverrides( $plan, Helper::getBasketPrice( $paymentBasket, $subscription, $selected ) );
		} else {
			// Promote selected pricing and options to the basket parameters
			$integrationParams	=	$paymentBasket->getParams( 'integrations' );

			$this->setPaymentItemOptions( $plan, $subscription, $integrationParams, $this->setPaymentItemPrice( $plan, $subscription, $integrationParams ) );

			$paymentBasket->set( 'integrations', $integrationParams->asJson() );

			// Override the subscriptions plan object pricing from the selected options
			$selected			=	Helper::getSelectedPlanPrice( $plan, $subscription );

			if ( $selected ) {
				Helper::setPlanOverrides( $plan, $selected );
			}

			Helper::setPlanOverrides( $plan, Helper::getSelectedPrice( $plan, $subscription, $selected ) );
		}
	}

	/**
	 * @param cbpaidProduct   $plan
	 * @param cbpaidSomething $subscription
	 * @param ParamsInterface $integrationParams
	 * @return null|PriceEntity
	 */
	private function setPaymentItemPrice( cbpaidProduct $plan, cbpaidSomething $subscription, ParamsInterface $integrationParams ): ?PriceEntity
	{
		if ( ! Helper::hasPlanPrices( $plan ) ) {
			return null;
		}

		$selected			=	Helper::getSelectedPlanPrice( $plan, $subscription );

		if ( ! $selected ) {
			$integrationParams->set( 'planprices.' . $plan->getInt( 'id', 0 ), '' );

			return null;
		}

		$integrationParams->set( 'planprices.' . $plan->getInt( 'id', 0 ), $selected->getName() );

		return $selected;
	}

	/**
	 * @param cbpaidProduct    $plan
	 * @param cbpaidSomething  $subscription
	 * @param ParamsInterface  $integrationParams
	 * @param null|PriceEntity $selectedPlanPrice
	 * @return void
	 */
	private function setPaymentItemOptions( cbpaidProduct $plan, cbpaidSomething $subscription, ParamsInterface $integrationParams, ?PriceEntity $selectedPlanPrice = null ): void
	{
		if ( ! Helper::hasPlanOptions( $plan ) ) {
			return;
		}

		$planId						=	$plan->getInt( 'id', 0 );
		$selected					=	Helper::getSelectedValues( $plan, $subscription, $selectedPlanPrice );

		if ( ! $selected ) {
			$integrationParams->set( 'planoptions.' . $planId, [] );
			$integrationParams->set( 'planoptionspaid.' . $planId, [] );

			return;
		}

		$options					=	[];
		$paid						=	[];

		foreach ( $selected as $v ) {
			$k						=	$v->getOption()->getName();

			if ( ! \array_key_exists( $k, $options ) ) {
				$options[$k]		=	[];
			}

			if ( ! \array_key_exists( $k, $paid ) ) {
				$paid[$k]			=	[ 'title' => $v->getOption()->getTitle(), 'values' => [] ];
			}

			$options[$k][]			=	$v->getValue();

			$paid[$k]['values'][]	=	[ 'label' => $v->getLabel(), 'first_rate' => $v->getFirstRate(), 'rate' => $v->getRate() ];
		}

		$integrationParams->set( 'planoptions.' . $planId, $options );
		$integrationParams->set( 'planoptionspaid.' . $planId, $paid );
	}

	/**
	 * @param UserTable                 $user
	 * @param cbpaidPaymentBasket       $paymentBasket
	 * @param cbpaidSomething[]         $subscriptions
	 * @param null|string               $unifiedStatus
	 * @param null|string               $previousUnifiedStatus
	 * @param null|string               $eventType
	 * @param cbpaidPaymentNotification $notification
	 * @return void
	 */
	public function onCPayBeforePaymentStatusUpdateEvent( UserTable $user, cbpaidPaymentBasket $paymentBasket, array $subscriptions, ?string $unifiedStatus, ?string $previousUnifiedStatus, ?string $eventType, cbpaidPaymentNotification $notification ): void
	{
		if ( ( ( $previousUnifiedStatus === 'NotInitiated' ) && ( $unifiedStatus !== 'RegistrationCancelled' ) ) || ( $notification->getString( 'payment_status', '' ) === 'Canceled_Reversal' ) ) {
			// Basket was either initialized or a reversal was cancelled (e.g. dispute won) so promote the planoption parameters to the subscriptions
			$integrationParams			=	$paymentBasket->getParams( 'integrations' );

			$hasPrices					=	$integrationParams->has( 'planprices' );
			$hasOptions					=	$integrationParams->has( 'planoptions' );

			if ( ( ! $hasPrices ) && ( ! $hasOptions ) ) {
				return;
			}

			foreach ( $subscriptions as $subscription ) {
				// Promote basket pricing and options parameters to the subscription parameters
				$subscriptionParams		=	$subscription->getParams( 'integrations' );

				if ( $hasPrices ) {
					$this->setBasketSubscriptionPrices( $integrationParams, $subscriptionParams, $subscription );
				}

				if ( $hasOptions ) {
					$this->setBasketSubscriptionOptions( $integrationParams, $subscriptionParams, $subscription );
				}

				$subscription->set( 'integrations', $subscriptionParams->asJson() );

				// Override the subscriptions plan object pricing from the basket
				$plan					=	$subscription->getPlan();
				$selectedPlanPrice		=	null;

				if ( $hasPrices ) {
					$selectedPlanPrice	=	Helper::getBasketPlanPrice( $paymentBasket, $subscription );

					if ( $selectedPlanPrice ) {
						Helper::setPlanOverrides( $plan, $selectedPlanPrice );
					}
				}

				if ( $hasOptions ) {
					Helper::setPlanOverrides( $plan, Helper::getBasketPrice( $paymentBasket, $subscription, $selectedPlanPrice ) );
				}
			}
		} elseif ( \in_array( $unifiedStatus, [ 'Refunded', 'Reversed' ], true ) ) {
			// Basket was fully refunded or reversed (e.g. dispute) so revert to the previous plan options
			$integrationParams			=	$paymentBasket->getParams( 'integrations' );

			$hasPrices					=	$integrationParams->has( 'planprices' );
			$hasOptions					=	$integrationParams->has( 'planoptions' );

			if ( ( ! $hasPrices ) && ( ! $hasOptions ) ) {
				return;
			}

			foreach ( $subscriptions as $subscription ) {
				$subscriptionParams		=	$subscription->getParams( 'integrations' );

				if ( $hasPrices ) {
					$this->revertBasketSubscriptionPrices( $subscriptionParams, $subscription );
				}

				if ( $hasOptions ) {
					$this->revertBasketSubscriptionOptions( $subscriptionParams, $subscription );
				}

				$subscription->set( 'integrations', $subscriptionParams->asJson() );
			}
		}
	}

	/**
	 * @param ParamsInterface $integrationParams
	 * @param ParamsInterface $subscriptionParams
	 * @param cbpaidSomething $subscription
	 * @return void
	 */
	private function setBasketSubscriptionPrices( ParamsInterface $integrationParams, ParamsInterface $subscriptionParams, cbpaidSomething $subscription ): void
	{
		if ( ! Helper::hasPlanOptions( $subscription->getPlan() ) ) {
			return;
		}

		$newPrice			=	$integrationParams->getString( 'planprices.' . $subscription->getPlan()->getInt( 'id', 0 ), '' );
		$currentPrice		=	$subscriptionParams->getString( 'planprices', '' );

		if ( $newPrice === $currentPrice ) {
			return;
		}

		if ( $currentPrice && ( ! $subscription->getPlan()->getParam( 'options_prices_changeable', 1, 'integrations' ) ) ) {
			return;
		}

		if ( ! $newPrice ) {
			$subscriptionParams->unsetEntry( 'planprices' );
		} else {
			$subscriptionParams->set( 'planprices', $newPrice );
		}

		$subscriptionParams->set( 'previousplanprices', $currentPrice );
	}

	/**
	 * @param ParamsInterface $subscriptionParams
	 * @param cbpaidSomething $subscription
	 * @return void
	 */
	private function revertBasketSubscriptionPrices( ParamsInterface $subscriptionParams, cbpaidSomething $subscription ): void
	{
		if ( ! Helper::hasPlanOptions( $subscription->getPlan() ) ) {
			return;
		}

		$currentPrice		=	$subscriptionParams->getString( 'planprices', '' );
		$previousPrice		=	$subscriptionParams->getString( 'previousplanprices', '' );

		if ( $currentPrice === $previousPrice ) {
			return;
		}

		if ( ! $previousPrice ) {
			$subscriptionParams->unsetEntry( 'planprices' );
		} else {
			$subscriptionParams->set( 'planprices', $previousPrice );
		}

		$subscriptionParams->unsetEntry( 'previousplanprices' );
	}

	/**
	 * @param ParamsInterface $integrationParams
	 * @param ParamsInterface $subscriptionParams
	 * @param cbpaidSomething $subscription
	 * @return void
	 */
	private function setBasketSubscriptionOptions( ParamsInterface $integrationParams, ParamsInterface $subscriptionParams, cbpaidSomething $subscription ): void
	{
		if ( ! Helper::hasPlanOptions( $subscription->getPlan() ) ) {
			return;
		}

		$newOptions			=	$integrationParams->subTree( 'planoptions.' . $subscription->getPlan()->getInt( 'id', 0 ) )->asArray();
		$currentOptions		=	$subscriptionParams->subTree( 'planoptions' )->asArray();

		if ( $newOptions === $currentOptions ) {
			return;
		}

		if ( $currentOptions && ( ! $subscription->getPlan()->getParam( 'options_changeable', 1, 'integrations' ) ) ) {
			return;
		}

		if ( ! $newOptions ) {
			$subscriptionParams->unsetEntry( 'planoptions' );
		} else {
			$subscriptionParams->set( 'planoptions', $newOptions );
		}

		$subscriptionParams->set( 'previousplanoptions', $currentOptions );
	}

	/**
	 * @param ParamsInterface $subscriptionParams
	 * @param cbpaidSomething $subscription
	 * @return void
	 */
	private function revertBasketSubscriptionOptions( ParamsInterface $subscriptionParams, cbpaidSomething $subscription ): void
	{
		if ( ! Helper::hasPlanOptions( $subscription->getPlan() ) ) {
			return;
		}

		$currentOptions		=	$subscriptionParams->subTree( 'planoptions' )->asArray();
		$previousOptions	=	$subscriptionParams->subTree( 'previousplanoptions' )->asArray();

		if ( $currentOptions === $previousOptions ) {
			return;
		}

		if ( ! $previousOptions ) {
			$subscriptionParams->unsetEntry( 'planoptions' );
		} else {
			$subscriptionParams->set( 'planoptions', $previousOptions );
		}

		$subscriptionParams->unsetEntry( 'previousplanoptions' );
	}
}