<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\PlanOptions\Trigger;

use CB\Plugin\PlanOptions\Entity\PriceEntity;
use CB\Plugin\PlanOptions\Helper;
use CB\Plugin\PlanOptions\TemplateHandler;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use cbpaidItem;
use cbpaidProduct;
use cbpaidSomething;

\defined('CBLIB') or die();

class PlanTrigger extends \cbPluginHandler
{
	/**
	 * Sets the base plan price if plan price override is enabled
	 *
	 * @param cbpaidItem $plan
	 * @return void
	 */
	public function onCPayBeforeDisplayProductPeriodPrice( cbpaidItem $plan ): void
	{
		if ( ! $plan instanceof cbpaidProduct ) {
			return;
		}

		Helper::setBasePlanPrice( $plan );
	}

	/**
	 * Draws plan options and price selection
	 *
	 * @param cbpaidProduct $plan
	 * @param null|string   $childrenRendering
	 * @param null|string   $reason
	 * @param null|bool     $drawOnlyAsContainer
	 * @param null|int      $userId
	 * @param bool          $displayPrice
	 * @return string
	 */
	public function onCPayBeforeDrawPlan( cbpaidProduct $plan, ?string &$childrenRendering, ?string $reason, ?bool $drawOnlyAsContainer, ?int $userId, bool &$displayPrice ): string
	{
		if ( ( ! Helper::getPlanPrices( $plan ) ) && ( ! Helper::getPlanOptions( $plan ) ) ) {
			return '';
		}

		/** @var $viewer \cbpaidPlanOptionsView */
		$viewer		=	TemplateHandler::getViewer( null, 'planoptions' );

		return $viewer->drawForm( $plan, null, $displayPrice );
	}

	/**
	 * Checks for missing required plan options for new subscriptions
	 * Then overrides plan pricing based off selected options
	 *
	 * @param cbpaidProduct   $plan
	 * @param ParamsInterface $integrationParams
	 * @return void
	 */
	public function onCPayAfterPlanSelected( cbpaidProduct $plan, ParamsInterface $integrationParams ): void
	{
		Helper::setBasePlanPrice( $plan );

		$this->setPlanSelectedOptions( $plan, $integrationParams, $this->setPlanSelectedPrice( $plan, $integrationParams ) );
	}

	/**
	 * @param cbpaidProduct   $plan
	 * @param ParamsInterface $integrationParams
	 * @return null|PriceEntity
	 */
	private function setPlanSelectedPrice( cbpaidProduct $plan, ParamsInterface $integrationParams ): ?PriceEntity
	{
		$selected	=	Helper::getSelectedPlanPrice( $plan );

		if ( ! $selected ) {
			return null;
		}

		$integrationParams->set( 'planprices', $selected->getName() );

		Helper::setPlanOverrides( $plan, $selected );

		return $selected;
	}

	/**
	 * @param cbpaidProduct    $plan
	 * @param ParamsInterface  $integrationParams
	 * @param null|PriceEntity $selectedPlanPrice
	 * @return void
	 */
	private function setPlanSelectedOptions( cbpaidProduct $plan, ParamsInterface $integrationParams, ?PriceEntity $selectedPlanPrice = null ): void
	{
		global $_PLUGINS;

		if ( ! Helper::hasPlanOptions( $plan ) ) {
			return;
		}

		$missing				=	Helper::getMissingOptions( $plan );

		if ( $missing ) {
			$labels				=	[];

			foreach ( $missing as $v ) {
				$labels[]		=	$v->getTitle();
			}

			$_PLUGINS->_setErrorMSG( CBTxt::T( 'PLAN_OPTIONS_MISSING', 'The following required options are missing: [missing]', [ '[missing]' => implode( ', ', $labels ) ] ) );
			$_PLUGINS->raiseError();
			return;
		}

		$selected				=	Helper::getSelectedValues( $plan, null, $selectedPlanPrice );

		if ( ! $selected ) {
			return;
		}

		$options				=	[];

		foreach ( $selected as $v ) {
			$k					=	$v->getOption()->getName();

			if ( ! \array_key_exists( $k, $options ) ) {
				$options[$k]	=	[];
			}

			$options[$k][]		=	$v->getValue();
		}

		$integrationParams->set( 'planoptions', $options );

		Helper::setPlanOverrides( $plan, Helper::getSelectedPrice( $plan, null, $selectedPlanPrice ) );
	}

	/**
	 * Checks for missing required plan options for renew/reactivate/resubscribe
	 * Then overrides plan pricing based off selected options
	 *
	 * @param cbpaidProduct   $plan
	 * @param cbpaidSomething $subscription
	 * @return void
	 */
	public function onCPayAfterPlanRenewalSelected( cbpaidProduct $plan, cbpaidSomething $subscription ): void
	{
		Helper::setBasePlanPrice( $plan, $subscription );

		$this->setPlanSelectedRenewalOptions( $plan, $subscription, $this->setPlanSelectedRenewalPrice( $plan, $subscription ) );
	}

	/**
	 * @param cbpaidProduct   $plan
	 * @param cbpaidSomething $subscription
	 * @return null|PriceEntity
	 */
	private function setPlanSelectedRenewalPrice( cbpaidProduct $plan, cbpaidSomething $subscription ): ?PriceEntity
	{
		$selected	=	Helper::getSelectedPlanPrice( $plan, $subscription );

		if ( ! $selected ) {
			return null;
		}

		Helper::setPlanOverrides( $plan, $selected );

		return $selected;
	}

	/**
	 * @param cbpaidProduct    $plan
	 * @param cbpaidSomething  $subscription
	 * @param null|PriceEntity $selectedPlanPrice
	 * @return void
	 */
	private function setPlanSelectedRenewalOptions( cbpaidProduct $plan, cbpaidSomething $subscription, ?PriceEntity $selectedPlanPrice = null ): void
	{
		global $_PLUGINS;

		if ( ! Helper::hasPlanOptions( $plan ) ) {
			return;
		}

		$missing			=	Helper::getMissingOptions( $plan, $subscription );

		if ( $missing ) {
			$labels			=	[];

			foreach ( $missing as $v ) {
				$labels[]	=	$v->getTitle();
			}

			$_PLUGINS->_setErrorMSG( CBTxt::T( 'PLAN_OPTIONS_MISSING', 'The following required options are missing: [missing]', [ '[missing]' => implode( ', ', $labels ) ] ) );
			$_PLUGINS->raiseError();
			return;
		}

		$selected			=	Helper::getSelectedValues( $plan, $subscription, $selectedPlanPrice );

		if ( ! $selected ) {
			return;
		}

		Helper::setPlanOverrides( $plan, Helper::getSelectedPrice( $plan, $subscription, $selectedPlanPrice ) );
	}

	/**
	 * Overrides plan core parameters (price and validity) based off selected (or stored) options when a plan is loaded in a product
	 * This is primarily to cover ->getPlan() accesses AFTER having subscribed to ensure all future calls take into account the plan options
	 *
	 * @param cbpaidProduct   $plan
	 * @param cbpaidSomething $subscription
	 * @return void
	 */
	public function onCPayLoadSomethingPlan( cbpaidProduct $plan, cbpaidSomething $subscription ): void
	{
		Helper::setBasePlanPrice( $plan, $subscription );

		$this->setPlanOptions( $plan, $subscription, $this->setPlanPrice( $plan, $subscription ) );
	}

	/**
	 * @param cbpaidProduct   $plan
	 * @param cbpaidSomething $subscription
	 * @return null|PriceEntity
	 */
	private function setPlanPrice( cbpaidProduct $plan, cbpaidSomething $subscription ): ?PriceEntity
	{
		$selected	=	Helper::getSelectedPlanPrice( $plan, $subscription );

		if ( ! $selected ) {
			return null;
		}

		Helper::setPlanOverrides( $plan, $selected );

		return $selected;
	}

	/**
	 * @param cbpaidProduct    $plan
	 * @param cbpaidSomething  $subscription
	 * @param null|PriceEntity $selectedPlanPrice
	 * @return void
	 */
	private function setPlanOptions( cbpaidProduct $plan, cbpaidSomething $subscription, ?PriceEntity $selectedPlanPrice = null ): void
	{
		if ( ! Helper::getSelectedValues( $plan, $subscription ) ) {
			return;
		}

		Helper::setPlanOverrides( $plan, Helper::getSelectedPrice( $plan, $subscription, $selectedPlanPrice ) );
	}
}