<?php
/**
* @version $Id: cbsubs.url.php 1489 2012-07-16 14:57:45Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_PLUGINS;

$_PLUGINS->registerFunction( 'onCPayUserStateChange', 'onCPayUserStateChange', 'getcbsubsurlTab' );

/**
 * CBSubs URL integration plugin class
 */
class getcbsubsurlTab extends cbTabHandler
{

	/**
	 * Called at each change of user subscription state due to a plan activation or deactivation
	 *
	 * @param  UserTable        $user
	 * @param  string           $status
	 * @param  int              $planId
	 * @param  int              $replacedPlanId
	 * @param  ParamsInterface  $integrationParams
	 * @param  string           $cause              'PaidSubscription' (first activation only), 'SubscriptionActivated' (renewals, cancellation reversals), 'SubscriptionDeactivated', 'Denied'
	 * @param  string           $reason             'N' new subscription, 'R' renewal, 'U'=update
	 * @param  int              $now                Unix time
	 * @param cbpaidSomething   $subscription
	 */
	public function onCPayUserStateChange( &$user, $status, $planId, $replacedPlanId, &$integrationParams, $cause, $reason, /** @noinspection PhpUnusedParameterInspection */ $now, &$subscription )
	{
		if ( ! $user ) {
			return;
		}

		$event											=	null;

		if ( ( $status == 'A' ) && ( $cause == 'PaidSubscription' ) && ( $reason != 'R' ) ) {
			$event										=	'activation';
		} elseif ( ( $status == 'A' ) && ( $cause == 'PaidSubscription' ) && ( $reason == 'R' ) ) {
			$event										=	'renewal';
		} elseif ( ( $status == 'X' ) && ( $cause != 'Pending' ) ) {
			$event										=	'expiration';
		} elseif ( ( $status == 'C' ) && ( $cause != 'Pending' ) ) {
			$event										=	'deactivation';
		}

		if ( $event ) {
			$path										=	$integrationParams->get( 'url_path_' . $event, null, GetterInterface::RAW );
			$method										=	$integrationParams->get( 'url_method_' . $event, 'GET', GetterInterface::STRING );
			$results									=	$integrationParams->get( 'url_results_' . $event, 0, GetterInterface::INT );

			if ( $path ) {
				// add substitutions for: [plan_id], [replaced_plan_id], [subscription_id], [parent_plan_id], [parent_subscription_id]
				$extraStringsLocal						=	array(	'plan_id'					=>	(int) $planId,
																	'replaced_plan_id'			=>	(int) $replacedPlanId,
																	'subscription_id'			=>	$subscription->get( 'id', 0, GetterInterface::INT ),
																	'parent_plan_id'			=>	$subscription->get( 'parent_plan', 0, GetterInterface::INT ),
																	'parent_subscription_id'	=>	$subscription->get( 'parent_subscription', 0, GetterInterface::INT )
																);

				$extraStrings							=	array_merge( $subscription->substitutionStrings( false ), $extraStringsLocal );

				$cbUser									=	CBuser::getInstance( $user->get( 'id', 0, GetterInterface::INT ) );

				$logObject								=	new cbpaidHistory();

				if ( ! $cbUser ) {
					$logObject->logError( 5, CBTxt::T( 'CBSubs URL skipped due to missing user.' ), null );

					return;
				}

				$path									=	trim( $cbUser->replaceUserVars( $path, array( $this, '_urlencode' ), false, $extraStrings, false ) );

				if ( ! $path ) {
					$logObject->logError( 5, CBTxt::T( 'CBSubs URL skipped due to missing path.' ), null );

					return;
				}

				$client									=	new \GuzzleHttp\Client();

				try {
					$options							=	array( 'timeout' => 30 );
					$body								=	array();
					$headers							=	array();

					$request							=	$integrationParams->get( 'url_' . strtolower( $method ) . '_' . $event, null, GetterInterface::RAW );

					if ( $request ) {
						if ( $method == 'XML' ) {
							$headers['Content-Type']	=	'text/xml';

							$body						=	trim( $cbUser->replaceUserVars( $request, array( $this, '_htmlspecialchars' ), false, $extraStrings, false ) );
						} else {
							foreach ( explode( "\n", $request ) as $vars ) {
								$var					=	explode( '=', trim( $vars ), 2 );

								if ( count( $var ) == 2 ) {
									$key				=	trim( $var[0] );
									$value				=	trim( $cbUser->replaceUserVars( $var[1], false, false, $extraStrings, false ) );

									$body[$key]			=	$value;
								}
							}
						}
					}

					$header								=	$integrationParams->get( 'url_header_' . $event, null, GetterInterface::RAW );

					if ( $header ) {
						foreach ( explode( "\n", $header ) as $vars ) {
							$var						=	explode( '=', trim( $vars ), 2 );

							if ( count( $var ) == 2 ) {
								$key					=	trim( $var[0] );
								$value					=	trim( $cbUser->replaceUserVars( $var[1], false, false, $extraStrings, false ) );

								$headers[$key]			=	$value;
							}
						}
					}

					if ( $headers ) {
						$options['headers']				=	$headers;
					}

					if ( in_array( $method, array( 'POST', 'XML' ) ) ) {
						if ( $body ) {
							if ( cbGuzzleVersion() >= 6 ) {
								$options['form_params']	=	$body;
							} else {
								$options['body']		=	$body;
							}
						}

						$result							=	$client->post( $path, $options );
					} else {
						if ( $body ) {
							$options['query']			=	$body;
						}

						$result							=	$client->get( $path, $options );
					}

					if ( $result->getStatusCode() != 200 ) {
						$logObject->logError( 3, CBTxt::T( 'CBSUBS_URL_FAILED_ERROR', 'CBSubs URL failed due to error: [error] ([code])', array( '[code]' => $result->getStatusCode(), '[error]' => $result->getReasonPhrase() ) ), null );

						return;
					}
				} catch ( \GuzzleHttp\Exception\ClientException $e ) {
					$logObject->logError( 3, CBTxt::T( 'CBSUBS_URL_FAILED_ERROR_REQUEST', "CBSubs URL failed due to error: [error] ([code])\n\n[request]", array( '[code]' => $e->getCode(), '[error]' => $e->getMessage(), '[request]' => (string) $e->getRequest() ) ), null );

					return;
				}

				if ( $results && ( $result !== false ) && ( ! Application::Cms()->getClientId() ) ) {
					echo '<div class="CBSubsURL_Results_' . (int) $planId . '">' . (string) $result->getBody() . '</div>';
				}
			}
		}
	}

	/**
	 * Utility function for CBuser::replaceUserVars to urlencode() all CB substitutions
	 *
	 * @param  string  $str
	 * @return string
	 */
	public function _urlencode( $str )
	{
		return urlencode( $str );
	}

	/**
	 * Utility function for CBuser::replaceUserVars to htmlspecialchars() all CB substitutions
	 *
	 * @param  string  $str
	 * @return string
	 */
	public function _htmlspecialchars( $str )
	{
		return htmlspecialchars( $str, ENT_COMPAT, 'UTF-8' );
	}
}
