<?php
/**
* @version $Id: cbpaidsubscriptions.ccbill.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Registry\ParamsInterface;
use CBLib\Xml\SimpleXMLElement;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php');

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * This is the OEM version
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidccbilloem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param  cbpaidGatewayAccount        $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		$this->_gatewayUrls	=	array(	'psp+normal' => $this->getAccountParam( 'psp_normal_url' ) ); // 'bill.ccbill.com/jpost/signup.cgi.',
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams						=	$this->_getBasicRequstParams( $paymentBasket );

		// sign single payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	*/
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams									=	$this->_getBasicRequstParams( $paymentBasket );

		// CCBill doesn't allow for free trials.. fix the below to properly add subscription price and period
		// always use period and amount 3 for form price and period if amount 1 is zero

		// check for subscription or if single payment:
		if ( $paymentBasket->period3 ) {
			// is a subscription and duration of at least 3 days and maximum of 365 days; fill price and period variables:
			if ( $paymentBasket->period1 ) {
				$requestParams['formPrice']				=	sprintf( '%.2f', $paymentBasket->mc_amount1 );
				$requestParams['formPeriod']			=	$this->_convertToCCBillPeriod( explode( ' ', $paymentBasket->period1 ) );
			} else {
				$requestParams['formPrice']				=	sprintf( '%.2f', $paymentBasket->mc_amount3 );
				$requestParams['formPeriod']			=	$this->_convertToCCBillPeriod( explode( ' ', $paymentBasket->period3 ) );
			}

			// let's see if basket has been allowed to be autorecurring:
			$enable_processor							=	3; // we allow autorecurring but let user choose, as choice is at payment gateway and enforcement is only settable at processor in this processor...
			$subscription_possibilities					=	$this->_getPaySubscribePossibilities( $enable_processor, $paymentBasket );

			if ( $subscription_possibilities & 0x2 ) {
				// autorecurring allowed or mandatory:
				$requestParams['formRecurringPrice']	=	sprintf( '%.2f', $paymentBasket->mc_amount3 );
				$requestParams['formRecurringPeriod']	=	$this->_convertToCCBillPeriod( explode( ' ', $paymentBasket->period3 ) );

				// calculate number of rebills (reoccurances of a subscription):
				if ( $paymentBasket->recur_times ) {
					$rebills							=	$paymentBasket->recur_times;
				} else {
					$rebills							=	99; // 99 is used for unlimited until canceled
				}

				$requestParams['formRebills']			=	$rebills;
			}
		}

		// sign subscription payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	* The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	*/
	protected function handleReturn( $paymentBasket, $postdata )
	{
		$requestdata	=	array();

		// Our only way was to store into a browser cookie the basket params, as CCBill doesn't return anything custom: get it:
		$this->_basketFromCookie( $requestdata );

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	* The user cancelled his payment
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId						=	(int) $this->_getReqParam( 'basket' );
			$shared_secret							=	$this->_getReqParam( 'id' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$requestdata						=	array();

				// Our only way was to store into a browser cookie the basket params, as CCBill doesn't return anything custom: get it:
				$this->_basketFromCookie( $requestdata );

				$paymentBasketId					=	(int) cbGetParam( $requestdata, 'orderID', 0 );
				$shared_secret						=	cbGetParam( $requestdata, 'shared_secret', null );
			}

			$exists									=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $shared_secret == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status		=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}

		return false;
	}

	/**
	* The payment service provider server did a server-to-server notification: verify and handle it here:
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleNotification( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['orderID'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata	=	$postdata;
		} else {
			// but if customer needs GET, we will work with it too:
			$requestdata	=	$this->_getGetParams();
		}

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' );
	}

	/**
	 * Attempts to authorize and capture a credit card for a single payment of a payment basket using PSP DirectLink
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $returnText                  RETURN param
	 * @param  boolean              $transientErrorDoReschedule  RETURN param
	 * @return boolean              TRUE: succes, FALSE: failed or unknown result
	 */
	protected function processScheduledAutoRecurringPayment( $paymentBasket, &$returnText, &$transientErrorDoReschedule )
	{
		global $_CB_framework;

		// form XML request:
		$formvars								=	$this->_fillinAutoRecuringDataLinkRequstParams( $paymentBasket );
		$datalink_url							=	str_replace( '%bill\.ccbill\.com/jpost/signup\.cgi%i', 'datalink.ccbill.com/utils/subscriptionManagement.cgi', $this->pspUrl( $paymentBasket, true ) );

		//Send the XML via curl:
		$response								=	null;
		$status									=	null;
		$error									=	$this->_httpsRequest( $datalink_url, $formvars, 105, $response, $status, 'post', 'normal' );

		if ( $error || ( $status != 200 ) || ! $response ) {
			// An error occurred or got returned: Log it as $ipn notification with error:
			$now								=	$_CB_framework->now();
			$ipn								=	$this->_prepareIpn( 'B', $paymentBasket->payment_status, 'Autorecurring', 'PSP DataLink unavailable', $now, 'utf-8' );

			$ipn->bindBasket( $paymentBasket );

			$errorText							=	'HTTPS POST Connection to payment gateway server failed (check system information in CBSubs Settings): ERROR: ' . $error . ' (' . ($status == -100 ? 'Timeout' : $status ) . ')';

			$ipn->setRawResult( $errorText );
			$ipn->setRawData( '$response="' . addslashes( $response ) . '"' . "\n" . '$status=' . addslashes( $status ) . ';' . "\n" . '$xml=' . $datalink_url . ';' . "\n" . '$_POST=' . var_export( $formvars, true ) . ';' . "\n" );
			$this->_storeIpnResult( $ipn, 'DATALINKDOWN' );
			$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $errorText, CBTxt::T( "Sorry, the payment server did not reply." ) . ' ' . CBTxt::T( "Please contact site administrator to check payment status and error log." ) );

			$user								=	CBuser::getUserDataInstance( $paymentBasket->user_id );
			$username							=	( $user ? $user->username : '?' );
			$returnText							=	sprintf( CBTxt::T( "FAILED Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), 'HTTP error ' . ': ' . $error . ' ' . 'Status' . ': ' . $status );
			$transientErrorDoReschedule			=	true;
			$return								=	false;
		} else {
			// Parse the response XML results:
			$paymentResult						=	$this->_handleDataLinkPaymentResult( $paymentBasket, $response, 'A', array( 'formvars' => $formvars, 'xmlreply' => $response ) );

			$user								=	CBuser::getUserDataInstance( $paymentBasket->user_id );
			$username							=	( $user ? $user->username : '?' );

			if ( $paymentResult !== false ) {
				if ( ( $paymentResult === true ) && in_array( $paymentBasket->payment_status, array( 'Completed', 'Pending' ) ) ) {
					if ( $paymentBasket->payment_status == 'Completed') {
						$returnText				=	sprintf( CBTxt::T( "Completed Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s with txn_id %s and auth_id %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->txn_id, $paymentBasket->auth_id );
					} else {
						$returnText				=	sprintf( CBTxt::T( "Pending Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s with txn_id %s and auth_id %s for reason: %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->txn_id, $paymentBasket->auth_id, $paymentBasket->reason_code );
					}

					$transientErrorDoReschedule	=	false;
					$return						=	true;
				} else {
					$returnText					=	sprintf( CBTxt::T( "FAILED (%s) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s." ), $paymentBasket->translatedPaymentStatus(), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->reason_code );
					$transientErrorDoReschedule	=	true;
					$return						=	false;
				}
			} else {
				$returnText						=	sprintf( CBTxt::T( "FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->reason_code );
				$transientErrorDoReschedule		=	true;
				$return							=	false;
			}
		}

		return $return;
	}

	/**
	* Cancels a subscription
	*
	* @param  cbpaidPaymentBasket   $paymentBasket  paymentBasket object
	* @param  cbpaidPaymentItem[]   $paymentItems   redirect immediately instead of returning HTML for output
	* @return boolean                               true if unsubscription done successfully, false if error
	*/
	protected function handleStopPaymentSubscription( $paymentBasket, $paymentItems )
	{
		if ( $paymentBasket->mc_amount3 ) {
			// Recurring amount existing and if first amount existed it got payed OK:
			// Stop CBSubs scheduled auto-payments:
			$paymentBasket->unscheduleAutoRecurringPayments();
		}
		// As CBSubs controls timed payments in this case, return OK for stopping subscriptions:
		return true;
	}

	/**
	 * GATEWAY-INTERNAL SPECIFIC PRIVATE METHODS:
	 */

    /**
     * We got result of a DataLink payment request: handle that.
     *
     * @param  cbpaidPaymentBasket  $paymentBasket      New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
     * @param  string               $xmlreply           _POST data for saving edited tab content as generated with getEditTab
     * @param  string               $type               Type of return ('P' for DirectLink (first payment) 'A' for Autorecurring payment (DirectLink) )
     * @param  array                $additionalLogData  Additional strings that should be logged with the IPN
     * @return boolean|null                             TRUE if payment is Completed or Pending, FALSE if registration cancelled or ErrorMSG generated, or NULL if payment Denied or Refunded successfully
     */
	private function _handleDataLinkPaymentResult( $paymentBasket, $xmlreply, $type, $additionalLogData )
	{
		// Parse the XML reply into $xml object:
		$xml				=	@new SimpleXMLElement( $xmlreply, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) );

		if ( $xml && ( $xml->getName() == 'ncresponse' ) ) {
			// The reply has it all in the XML attributes, so convert it into array:
			$requestdata	=	$xml->attributes();

			// And handle the reply accordingly:
			return $this->_returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData );
		} else {
			// We had an error parsing the XML reply: Log the error:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': DataLink reply is not the expected XML record: ' . $xmlreply, CBTxt::T( "Sorry, an unexpected reply has been received from the payment processor." ) . ' ' . CBTxt::T( "Please contact site administrator to check error log." ) );

			return false;
		}
	}

	/**
	 * sign payment request $requestParams with validation code added to $requestParams array
	 *
	 * @param  array  $requestParams     OUTPUT: adds $requestParams['formDigest'] to it
	 */
	private function _signRequestParams( &$requestParams )
	{
		$listOfParams					=	array(
											'FORMPRICE',
											'FORMPERIOD',
											'FORMRECURRINGPRICE',
											'FORMRECURRINGPERIOD',
											'FORMREBILLS',
											'CURRENCYCODE',
											'$md5_salt'				// Concatenate security code generated at gateway: equivalent to: $this->getAccountParam( 'md5_salt' )
										);
		// Concatenate them using this payments concatenation function with $caseInsensitiveKeys = true:
		$string							=	$this->_concatVars( $requestParams, $listOfParams, null, '', '', false, false, true, false, false );

		// compute validation code doing md5 of the string without uppercasing:
		$requestParams['formDigest']	=	$this->_hashString( $string, 'md5', false );
	}

	/**
	 * Verifies the signature in $requestParams['responseDigest']
	 *
	 * @param  array    $requestParams  Checks $requestParams['responseDigest'] towards determining values
	 * @return boolean                  True: valid, False: not valid
	 */
	private function _pspVerifySignature( $requestParams )
	{
		// Here we cannot use $this->_concatVars because we need to concat a '1' or a '0' and to verify signature of only one parameter:
		$signParams				=	array();

		// Make keys case-insensitive:
		foreach ( $requestParams as $k => $v ) {
			if ( ( $v !== '' ) && ( $v !== null ) ) {
				$k				=	strtoupper( $k );
				$signParams[$k]	=	$v;
			}
		}

		// Concatenate parameter . 1/0 . salt:
		$string					=	'';

		if ( isset( $signParams['SUBSCRIPTION_ID'] ) ) {
			$string				.=	$signParams['SUBSCRIPTION_ID'] . 1;
		} elseif ( isset( $signParams['DENIALID'] ) ) {
			$string				.=	$signParams['DENIALID'] . 0;
		}

		$string					.=	$this->getAccountParam( 'md5_salt' );

		// Check if hash corresponds:
		return ( md5( $string ) === cbGetParam( $requestParams, 'responseDigest' ) );
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's params:
	 *
	 * STATUS:	Status of the payment:
		Statuses in 1 digit are 'normal' statuses:
		0 means the payment is invalid (e.g. data validation error) or the processing is not complete either because it is still underway, or because the transaction was interrupted. If the cause is a validation error, an additional error code (*) (NCERROR) identifies the error.
		1 means the customer cancelled the transaction.
		2 means the acquirer did not authorise the payment.
		5 means the acquirer autorised the payment.
		9 means the payment was captured.
		Statuses in 2 digits correspond either to 'intermediary' situations or to abnormal events. When the second digit is:
		1, this means the payment processing is on hold.
		2, this means an unrecoverable error occurred during the communication with the acquirer. The result is therefore not determined. You must therefore call the acquirer's helpdesk to find out the actual result of this transaction.
		3, this means the payment processing (capture or cancellation) was refused by the acquirer whilst the payment had been authorised beforehand. It can be due to a technical error or to the expiration of the authorisation. You must therefore call the acquirer's helpdesk to find out the actual result of this transaction.
		4, this means our system has been notified the transaction was rejected well after the transaction was sent to your acquirer.
		5, this means our system hasn't sent the requested transaction to the acquirer since the merchant will send the transaction to the acquirer himself, like he specified in his configuration.
	 *
	 * ACCEPTANCE:
	 * Acquirer's acceptance (authorization) code.
	 * The acquirer sends back this code to confirm the amount of the transaction has been blocked on the card of the customer. The acceptance code is not unique.
	 *
	 * @param  array                $postdata             raw POST data received from the payment gateway
	 * @param  string               $reason               OUT: reason_code
     * @return int|string
     */
	private function _paymentStatus( $postdata, &$reason )
	{
		$accept_payment_condition	=	$this->getAccountParam( 'accept_payment_condition' );
		$reason						=	cbGetParam( $postdata, 'reasonForDecline', null );
		$status						=	(int) cbGetParam( $postdata, 'reasonForDeclineCode', 0 );

		switch ( $status ) {
			case 0:
				$reason				=	null;
				$status				=	'Completed';
				break;
			case 1:
				$reason				=	'Website is not available for signup';
				$status				=	'Error';
				break;
			case 2:
				$reason				=	'Unable to determine website signup requirements';
				$status				=	'Error';
				break;
			case 4:
			case 9:
			case 22:
			case 25:
			case 34:
			case 36:
				$reason				=	'Bank system error';
				$status				=	'Error';
				break;
			case 10:
				$reason				=	'Website has invalid pricing';
				$status				=	'Error';
				break;
			case 16:
				$reason				=	'Subscription ID provided is invalid';
				$status				=	'Error';
				break;
			case 17:
				$reason				=	'Subscription ID does not exist in system';
				$status				=	'Error';
				break;
			case 26:
				$reason				=	'Card processing setup is incorrect';
				$status				=	'Error';
				break;
			case 27:
				$reason				=	'System error';
				$status				=	'Error';
				break;
			case 28:
				$reason				=	'Unable to process transaction at this time';
				$status				=	'Error';
				break;
			case 33:
				$reason				=	'Unable to determine transaction type';
				$status				=	'Error';
				break;
			case 37:
				$reason				=	'Currently do not process specified banks bin';
				$status				=	'Error';
				break;
			case 41:
				$reason				=	'Client inactive';
				$status				=	'Error';
				break;
			case 3:
			case 40:
				$reason				=	'Credit card used is not accepted';
				$status				=	'Denied';
				break;
			case 5:
			case 23:
				$reason				=	'Credit card entered is invalid';
				$status				=	'Denied';
				break;
			case 6:
				$reason				=	'Credit card expiration date is invalid';
				$status				=	'Denied';
				break;
			case 7:
				$reason				=	'Bank account number invalid';
				$status				=	'Denied';
				break;
			case 8:
				$reason				=	'Bank routing number invalid';
				$status				=	'Denied';
				break;
			case 11:
				$reason				=	'Transaction declined';
				$status				=	'Denied';
				break;
			case 12:
				$reason				=	'Subscription already active';
				$status				=	'Denied';
				break;
			case 13:
			case 21:
				$reason				=	'Free trial already active';
				$status				=	'Denied';
				break;
			case 14:
			case 32:
				$reason				=	'Credit card CVV2 not provided';
				$status				=	'Denied';
				break;
			case 18:
				$reason				=	'Previous transaction request denied';
				$status				=	'Denied';
				break;
			case 19:
				$reason				=	'Provided credentials not authorized for signup';
				$status				=	'Denied';
				break;
			case 20:
				$reason				=	'No decline';
				$status				=	'Denied';
				break;
			case 24:
				$reason				=	'Transaction denied by bank';
				$status				=	'Denied';
				break;
			case 29:
				$reason				=	'Credit card expired';
				$status				=	'Denied';
				break;
			case 30:
				$reason				=	'Telpehone number invalid';
				$status				=	'Denied';
				break;
			case 31:
				$reason				=	'Insufficient funds';
				$status				=	'Denied';
				break;
			case 35:
				$reason				=	'Credit card declined';
				$status				=	'Denied';
				break;
			case 38:
				$reason				=	'Transaction refused by bank';
				$status				=	'Denied';
				break;
			case 39:
				$reason				=	'Too many submissions';
				$status				=	'Denied';
				break;
			case 42:
				$reason				=	'Address invalid';
				$status				=	'Denied';
				break;
			case 15:
				if ( ( $accept_payment_condition == 'pending' ) || ( $accept_payment_condition == 'authorized' ) ) {
					$reason			=	null;
					$status			=	'Completed';
				} else {
					$reason			=	'Transaction being processed';
					$status			=	'Pending';
				}
				break;
		}

		return $status;
	}

	/**
	 * Compute the CBSubs payment_type based on gateway's reply $postdata:
	 *
	 * @param  array   $postdata raw POST data received from the payment gateway
	 * @return string  Human-readable string
	 */
	private function _getPaymentType( $postdata )
	{
		// Get Payment Method $pm:
		$pm			=	preg_replace( '/[^-+ a-zA-Z0-9_]/', '', stripslashes( cbGetParam( $postdata, 'PM', '' ) ) );

		if ( $pm == 'CreditCard' ) {
			// Get the Brand of the credit card:
			$brand	=	preg_replace( '/[^-+ a-zA-Z0-9_]/', '', stripslashes( cbGetParam( $postdata, 'BRAND', '' ) ) );
			$pm		=	( $brand ? $brand . ' ' : '' ) . CBTxt::T( 'Credit Card' );
		}

		return $pm;
	}

	/**
	 * Gets expiration time of items in basket, if they have one and have all the same.
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return int                   Unix-time of expiration, NULL if none is expiring, FALSE if different expiration times
	 */
	private function getItemsExpiryTime( $paymentBasket )
	{
		global $_CB_framework;

		$expiryTime				=	null;

		foreach ( $paymentBasket->loadPaymentItems() as $item ) {
			if ( ! $item->isLifetimeValidity() ) {
				$varName		=	( $item->autorecurring && $item->first_validity ? 'first_validity' : 'validity' );
				$subscription	=	$item->loadSubscription();
				$itemExpTime	=	$item->getExpiryTime( $_CB_framework->now(), $varName, 1, $item->reason, $subscription ? $subscription->status : 'A' );

				if ( $expiryTime === null ) {
					$expiryTime	=	$itemExpTime;
				} elseif ( $itemExpTime != $expiryTime) {
					$expiryTime	=	false;
					break;
				}
			}
		}

		return $expiryTime;
	}

	/**
	 * Converts 3-letters ISO 4217 currency code to ISO 3166-1 numeric code and outputs an array( 3-letters-currency-code, integer currency code
	 * Note this method is only for currencies supported by CCBill.
	 *
	 * @param  string  $currency   3-letters ISO 4217 currency-code (e.g. USD)
	 * @return array               array( $currency, int numeric-code ) (e.g. array( 'USD', 840 ).
	 */
	private function _currencyToCode( $currency )
	{
		switch ( $currency ){
			case 'EUR':
				return array( 'EUR', 978 );
			case 'AUD':
				return array( 'AUD', 036 );
			case 'CAD':
				return array( 'CAD', 124 );
			case 'GBP':
				return array( 'GBP', 826 );
			case 'JPY':
				return array( 'JPY', 392 );
			case 'USD':
			default:
				return array( 'USD', 840 );
		}
	}


	/**
	* Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	* @return array                                $requestParams
	*/
	private function _getBasicRequstParams( $paymentBasket )
	{
		global $_CB_framework;

		// maps current to supported code and name:
		$mapCurrency						=	$this->_currencyToCode( $paymentBasket->mc_currency );

		// mandatory parameters:
		$requestParams						=	array();
		$requestParams['orderID']			=	$paymentBasket->id;
		$requestParams['clientAccnum']		=	$this->getAccountParam( 'pspid' );
		$requestParams['clientSubacc']		=	$this->getAccountParam( 'pspsubid' );
		$requestParams['formName']			=	$this->getAccountParam( 'pspformname' );
		$requestParams['currency']			=	$mapCurrency[0];
		$requestParams['currencyCode']		=	$mapCurrency[1];
		$requestParams['language']			=	$this->getAccountParam( 'language', 'english' );

		// calculate expiration in days:
		$exiryDays							=	round( ( $this->getItemsExpiryTime( $paymentBasket ) -  $_CB_framework->now() ) / 86400 );

		// is a single payment; fill price and period variables:
		$requestParams['formPrice']			=	sprintf( '%.2f', $paymentBasket->mc_gross );
		$requestParams['formPeriod']		=	( $exiryDays > 365 ? 365 : ( ( $exiryDays < 3 ) ? 2 : $exiryDays ) );

		// courtesy fields (pre-filled but editable on credit card mask):
		$requestParams['customer_fname']	=	$paymentBasket->first_name;
		$requestParams['customer_lname']	=	$paymentBasket->last_name;
		$requestParams['username']			=	$paymentBasket->username;

		// recommended anti-fraud fields:
		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 50 ) ) {
			$requestParams['email']			=	$paymentBasket->payer_email;
		}

		if ( $this->getAccountParam( 'givehiddenaddress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields					=	array(	'address1' => array( $paymentBasket->address_street, 30 ),
														'zipcode' => array( $paymentBasket->address_zip, 10 ),
														'city' => array( $paymentBasket->address_city, 30 ),
														'country' => array( $paymentBasket->getInvoiceCountry( 2 ), 2 ),
														'state' => array( $paymentBasket->getInvoiceState(), 30 )
													);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField					=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]		=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddentelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['phone_number']	=	$paymentBasket->contact_phone;
		}

		// set basket cookie so when user returns we can handle the basket (16hr long cookie):
		setcookie( 'cbsubs_ccbill_basket', (int) $paymentBasket->id . '|' . $paymentBasket->shared_secret, ( time() + ( 3600 * 16 ) ) );

		return $requestParams;
	}

	/**
	* Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	* @return array                                $requestParams
	*/
	private function _fillinAutoRecuringDataLinkRequstParams( $paymentBasket )
	{
		// Fill-in basic params:
		$requestParams						=	$this->getSubscriptionRequstParams( $paymentBasket );

		// Fill-in login credentials:
		$requestParams['username']			=	$this->getAccountParam( 'datalink_username' );
		$requestParams['password']			=	$this->getAccountParam( 'datalink_pswd' );

		// Calculate recurring order ID
		$requestParams['orderID']			=	$paymentBasket->id . 'R' . ( intval( $paymentBasket->recur_times_used ) + 2 );

		// Add datalink required parameters
		$requestParams['subscriptionId']	=	$paymentBasket->id;
		$requestParams['action']			=	'viewSubscriptionStatus';
		$requestParams['returnXML']			=	1;

		return $requestParams;
	}

    /**
     * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
     *
     * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
     * @param  array                $requestdata         Data returned by gateway
     * @param  string               $type                Type of return ('I' for INS, 'R' for PDT, 'P' for DirectLink (first payment) 'A' for Autorecurring payment (DirectLink) )
     * @param  null                 $additionalLogData   Additional Texts to log into IPN raw data
     * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
     */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret													=	null;
		$paymentBasketId										=	(int) cbGetParam( $requestdata, 'orderID', 0 );

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( (int) $paymentBasketId );

			if ( isset( $requestdata['shared_secret'] ) ) {
				if ( $requestdata['shared_secret'] == $paymentBasket->shared_secret ) {
					$validSharedSecret							=	true;
				} else {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The shared_secret is missing in the return cookie.', CBTxt::T( "Please contact site administrator to check error log." ) );
					$validSharedSecret							=	false;
				}
			} else {
				$validSharedSecret								=	null;
			}

			// 1) Payment basket exists
			// 2) It's not a Returned-result to site with a missing or invalid shared secret in the cookie
			// 3) a) It's an automatic payment
			//    OR b) the returned 'id' matches the basket secret
			//		    but it's Not a Returned-result for an already completed basket:
			if ( $exists
			&& ( ! ( ( $type == 'R' ) && ( $validSharedSecret !== true ) ) )
			&&	( ( ( $type == 'P' ) || ( $type == 'A' ) )
				|| (	( cbGetParam( $requestdata, $this->_getPagingParamName( 'id' ), 0 ) == $paymentBasket->shared_secret )
						&& ( ! ( ( $type == 'R' ) && ( $paymentBasket->payment_status == 'Completed' ) ) ) ) ) )
			{
				// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason );
				$paymentType									=	$this->_getPaymentType( $requestdata );
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E', 'P' => 'V', 'A' => 'W' );

					if ( isset( $errorTypes[$type] ) ) {
						// Converts error status to CBSubs's error statuses:
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	0;
				$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : ( $type == 'P' ? 'DATALINK PAYMENT' : ( $type == 'A' ? 'AUTORECURRING DIRECTLINK PAYMENT' : 'UNKNOWN' ) ) ) ) . '";' . "\n";

				if ( $additionalLogData ) {
					foreach ( $additionalLogData as $k => $v ) {
						$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
					}
				}

				$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	$reason;

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( "Sorry, the payment server replied with an error." ) . ' ' . CBTxt::T( "Please contact site administrator to check payment status and error log." ) );

					$ret										=	false;
				} else {
					// Payment status is not Error:
					// Get important parameters from basket into the $ipn:
					$ipn->bindBasket( $paymentBasket );

					// Convert fields from the gateway INS to CB's $ipn Notification format:
					$insToIpn									=	array(
																			'mc_currency' => 'currency',
																			'sale_id' => 'orderID',
																			'txn_id' => 'subscription_id',
																			'first_name' => 'customer_fname',
																			'last_name' => 'customer_lname',
																			'address_street' => 'address1',
																			'address_zip' => 'zipcode',
																			'address_city' => 'city',
																			'address_country' => 'country',
																			'address_state' => 'state',
																			'payer_email' => 'email',
																			'contact_phone' => 'phone_number',
																			'ip_addresses' => 'ip_address'
																		);

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $requestdata, $v );
					}

					$ipn->mc_gross								=	sprintf( '%.2f', cbGetParam( $requestdata, 'initialPrice' ) );
					$ipn->user_id								=	(int) $paymentBasket->user_id;

					$recurring									=	( $type == 'A' ? true : false );

					if ( $recurring ) {
						if ( $type != 'A' ) {
							// in case of first payment of auto-recurring payments:
							$ipn->txn_type						=	'subscr_signup';
							$ipn->subscr_id						=	(int) $paymentBasket->id;
							$ipn->subscr_date					=	$ipn->payment_date;
						} else {
							if ( $paymentStatus == 'Denied' ) {
								// Auto-recurring payment got denied:
								if ( ( $paymentBasket->reattempts_tried + 1 ) <= cbpaidScheduler::getInstance( $this )->retries ) {
									// Schedule a new trial is possible: log it as simply a failed attempt:
									$ipn->txn_type				=	'subscr_failed';
								} else {
									// Schedule a new trial is not possible anymore: log it as a cancellation (?):
									$ipn->txn_type				=	'subscr_cancel';
								}
							} elseif ( in_array( $paymentStatus, array( 'Completed', 'Processed', 'Pending' ) ) ) {
								// Normal case: it's a recurring payment:
								$ipn->txn_type					=	'subscr_payment';
							}
						}
					} else {
						// Single payment:
						$ipn->txn_type							=	'web_accept';
					}

					// DirectLink Payments and Auto-recurring payments do not have a hash signature, other ones do have one that must be checked:
					if ( in_array( $type, array( 'P', 'A') ) || $this->_pspVerifySignature( $requestdata ) ) {
						// 1) Basket id matches orderId
						// 2) a) Amounts match
						//	  b) OR it's @author brunner
						// 3) Currency matches:
						if	( ( $paymentBasketId == cbGetParam( $requestdata, 'orderID' ) )
							&& (	( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross )
								||	( $ipn->payment_status == 'Refunded' ) )
							&& ( $paymentBasket->mc_currency == $ipn->mc_currency ) )
						{
							// Ok, we passed all basic consistancy tests:
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								// Status is also that of a successful (or on it's way to successful) transaction:
								// Stores the $ipn notification as a successful one:
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								// Binds to basket the relevant $ipn attributes:
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// Sets autorecur and autorenew types:
								$autorecurring_type				=	( $recurring ? 2 : 0 ); // 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date
								$autorenew_type					=	( $recurring ? 2 : 0 );	 // 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration

								if ( $recurring ) {
									// We want to reattempt auto-recurring payment in case of failure:
									$paymentBasket->reattempt	=	1;
								}

								// In case of refund we need to log the payment as it has same TnxId as first payment:
								$txnIdMultiplePaymentDates		=	( $paymentStatus == 'Refunded' );

								// Update payment basket, generates all basket payment events, generates a transaction record (payment):
								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, $txnIdMultiplePaymentDates );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									// ok, we had success in terms of a payment initiation or completion:
									$ret						=	true;

									if ( $recurring && ( $type != 'A' ) ) {
										// Now we need to schedule the automatic payments to be done by CBSubs using DirectLink, and which will trigger the auto-renewals (if it's not already the autorecurring payment (A)):
										$paymentBasket->scheduleAutoRecurringPayments();
									}
								}
							} else {
								// The payment didn't initiate or complete: We failed:
								// Store the notification $ipn as a failed one:
								$this->_storeIpnResult( $ipn, 'FAILED' );

								// Updates payment basket status with the notification status:
								$paymentBasket->payment_status	=	$ipn->payment_status;

								// Sets error to display to the user:
								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								// We will be re-displaying the basket to the user so he can retry payment:
								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							// We found a major mismatch between orderID, amount or currency:
							// Store it in notification:
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							// Log it as error and prepare display of error to user:
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': orderID, amount or currency missmatch.', CBTxt::T( "Sorry, the payment does not match the basket." ) . ' ' . CBTxt::T( "Please contact site administrator to check error log." ) );

							$ret								=	false;
						}
					} else {
						// MD5 hash mismatches:
						// Store it in notification:
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						// Log it as error and prepare display of error to user:
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': CCBill e-Commerce MD5 Salt hash does not match with gateway. Please check MD5 Salt setting.', CBTxt::T( "The MD5 signature is incorrect." ) . ' ' . CBTxt::T( "Please contact site administrator to check error log." ) );

						$ret									=	false;
					}
				}
			}
		} else {
			// No basket found:
			// Log it as error and prepare display of error to user:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The OrderId is missing in the return URL: Probably due to missing "cbsubs_ccbill_basket" cookie', CBTxt::T( "Please contact site administrator to check error log." ) );
		}

		return  $ret;
	}	// end function _returnParamsHandler

	/**
	 * Converts an array( period, timearray ), e.g. array( 7, 'D' ) to days
	 *
	 * @param  array  $periodTypeArray  array( period, timearray )
	 * @return int                      Days
	 */
	private function _convertToCCBillPeriod( $periodTypeArray )
	{
		$p		=	$periodTypeArray[0];
		$t		=	$periodTypeArray[1];

		if ( $t == 'W' ) {
			// Weeks have 7 days:
			$p	=	$p * 7;
		}

		if ( $t == 'M' ) {
			// CCBill months have 30 days:
			$p	=	$p * 30;
		}

		if ( $t == 'Y' ) {
			// CCBill years have 365 days:
			$p	=	$p * 365;
		}

		return $p;
	}

	/**
	 * Sets $requestdata['orderID'] from $_COOKIE
	 *
	 * @param  array  $requestdata
	 * @return void
	 */
	private function _basketFromCookie( &$requestdata )
	{
		$basket_cookie						=	cbGetParam( $_COOKIE, 'cbsubs_ccbill_basket', null );

		if ( $basket_cookie ) {
			$basket_cookie					=	explode( '|', $basket_cookie );
			$requestdata['orderID']			=	( isset( $basket_cookie[0] ) ? $basket_cookie[0] : null );
			$requestdata['shared_secret']	=	( isset( $basket_cookie[1] ) ? $basket_cookie[1] : null );

			// expire basket cookie as we don't need it anymore:
			setcookie( 'cbsubs_ccbill_basket', null, ( time() - ( 3600 * 16 ) ) );
		}
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * This is the OEM variant.
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountccbilloem extends cbpaidGatewayAccounthostedpage
{
	/**
	 * USED by XML interface ONLY !!!
	 * Renders URL for successful returns
	 * We are overriding this method, as CCBill accepts non-https URLs:
	 *
	 * @param  string            $value   Variable value ( 'successurl', 'cancelurl', 'notifyurl' )
	 * @param  ParamsInterface   $params
	 * @param  string            $name    The name of the form element
	 * @param  SimpleXMLElement  $node    The xml element for the parameter
	 * @return string                     HTML to display
	 */
	public function renderUrl( $value, $params, $name, $node )
	{
		return $this->getPayMean()->adminUrlRender( $node->attributes( 'value' ) );
	}
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * This is the gateway-specific version.
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidccbill extends cbpaidccbilloem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * This is the gateway-specific version.
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountccbill extends cbpaidGatewayAccountccbilloem
{
}
