<?php
/**
* @version $Id: cbpaidsubscriptions.eway.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Xml\SimpleXMLElement;
use CBLib\Language\CBTxt;

/** Ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM basis
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidewayoem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		switch( (int) $this->getAccountParam( 'gateway_psp_country' ) ) {
			case 2:
				$pspurl		=	str_replace( 'au.', 'payment.', $this->getAccountParam( 'psp_normal_url', '' ) );
				break;
			case 3:
				$pspurl		=	str_replace( 'au.', 'nz.', $this->getAccountParam( 'psp_normal_url', '' ) );
				break;
			case 1:
			default:
				$pspurl		=	$this->getAccountParam( 'psp_normal_url', '' );
				break;
		}

		$this->_gatewayUrls	=	array( 'psp+normal' => $pspurl );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		return array(	'cmd' => 'payment',
						'amount' => sprintf( '%0.2f', $paymentBasket->mc_gross ),
						'currency_code' => $paymentBasket->mc_currency,
						'custom' => $paymentBasket->id
					);
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		return $this->getSinglePaymentRequstParams( $paymentBasket );
	}

	/**
	* Handles the gateway-specific result of payments (redirects back to this site and gateway notifications). WARNING: unchecked access !
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @param  string               $result         result= get parameter, other than 'notify', 'success' or 'cancel'.
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleOtherResult( $paymentBasket, $postdata, $result )
	{
		$return						=	null;

		if ( $result == 'payment' ) {
			$paymentBasketId		=	(int) cbGetParam( $postdata, 'custom', 0 );
			$exists					=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( sprintf( '%.2f', $paymentBasket->mc_gross ) == cbGetParam( $postdata, 'amount' ) ) && ( $paymentBasket->mc_currency == cbGetParam( $postdata, 'currency_code' ) ) ) {
				$requestParams		=	$this->_getBasicRequstParams( $paymentBasket );

				$response			=	null;
				$status				=	null;
				$error				=	$this->_httpsRequest( $this->gatewayUrl( 'psp' ) . '/Request/?' . http_build_query( $requestParams ), null, 45, $response, $status, 'get/1.0' );

				if ( ( ! $error ) && ( $status == 200 ) && $response ) {
					$requestdata	=	$this->xmlTagValuesToArray( new SimpleXMLElement( $response, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) ) );

					if ( cbGetParam( $requestdata, 'Result' ) == 'True' ) {
						cbRedirect( cbGetParam( $requestdata, 'URI' ) );
					} else {
						cbRedirect( $this->getCancelUrl( $paymentBasket ), cbGetParam( $requestdata, 'Error' ), 'error' );
					}
				}
			} else {
				$this->_setErrorMSG( CBTxt::T( 'Payment basket does not login.' ) );

				$return				=	false;
			}
		}

		return $return;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['AccessPaymentCode'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata	=	$postdata;
		} else {
			// but if customer needs GET, we will work with it too (removing CMS/CB/CBSubs specific routing params):
			$requestdata	=	$this->_getGetParams();
		}
		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId				=	(int) cbGetParam( $postdata, 'MerchantReference', null );
			}

			$exists								=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status	=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}
		return false;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                              Text to return to gateway if notification, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		return null;
	}

	/**
	 * gives gateway button URL server name from gateway URL list
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  boolean              $autoRecurring   TRUE: autorecurring payment, FALSE: single payment
	 * @return string  server-name (with 'https://' )
	 */
	protected function pspUrl( $paymentBasket, $autoRecurring )
	{
		return $this->pspRedirectUrl( $paymentBasket, array(), $autoRecurring );
	}

	/**
	 * Returns https redirect URL for redirections to gateway for payment with $requestParams
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  array                $requestParams
	 * @param  boolean              $autoRecurring   TRUE: autorecurring payment, FALSE: single payment
	 * @return string                                Full URL for redirect including https:// in front
	 */
	protected function pspRedirectUrl( $paymentBasket, $requestParams, $autoRecurring )
	{
		return $this->cbsubsGatewayUrl( 'payment', null, $paymentBasket, $requestParams, false );
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's reply in $postdata:
	 *
	 * @param  array                $postdata             raw POST data received from the payment gateway
	 * @param  string               $reason               OUTPUT: text of the payment status
	 * @return array|float|int|null|string
	 * @internal param string $reasonCode OUT: reason_code
	 */
	private function _paymentStatus( $postdata, &$reason )
	{
		$status			=	cbGetParam( $postdata, 'ResponseCode' );

		switch ( strtoupper( $status ) ) {
			case 'CX':
				$reason	=	'Customer Cancelled Transaction';
				$status	=	'Denied';
				break;
			case '00':
			case '08':
			case '10':
			case '11':
			case '16':
				$reason	=	null;
				$status	=	'Completed';
				break;
			case '01':
				$reason	=	'Refer to Issuer';
				$status	=	'Denied';
				break;
			case '02':
				$reason	=	'Refer to Issuer, special';
				$status	=	'Denied';
				break;
			case '03':
				$reason	=	'No Merchant';
				$status	=	'Denied';
				break;
			case '04':
				$reason	=	'Pick Up Card';
				$status	=	'Denied';
				break;
			case '05':
				$reason	=	'Do Not Honour';
				$status	=	'Denied';
				break;
			case '06':
				$reason	=	'Error';
				$status	=	'Error';
				break;
			case '07':
				$reason	=	'Pick Up Card, Special';
				$status	=	'Denied';
				break;
			case '09':
				$reason	=	'Request In Progress';
				$status	=	'Denied';
				break;
			case '12':
				$reason	=	'Invalid Transaction';
				$status	=	'Denied';
				break;
			case '13':
				$reason	=	'Invalid Amount';
				$status	=	'Denied';
				break;
			case '14':
				$reason	=	'Invalid Card Number';
				$status	=	'Denied';
				break;
			case '15':
				$reason	=	'No Issuer';
				$status	=	'Denied';
				break;
			case '19':
				$reason	=	'Re-enter Last Transaction';
				$status	=	'Denied';
				break;
			case '21':
				$reason	=	'No Action Taken';
				$status	=	'Denied';
				break;
			case '22':
				$reason	=	'Suspected Malfunction';
				$status	=	'Denied';
				break;
			case '23':
				$reason	=	'Unacceptable Transaction Fee';
				$status	=	'Denied';
				break;
			case '25':
				$reason	=	'Unable to Locate Record On File';
				$status	=	'Denied';
				break;
			case '30':
				$reason	=	'Format Error';
				$status	=	'Error';
				break;
			case '31':
				$reason	=	'Bank Not Supported By Switch';
				$status	=	'Denied';
				break;
			case '33':
				$reason	=	'Expired Card, Capture';
				$status	=	'Denied';
				break;
			case '34':
				$reason	=	'Suspected Fraud, Retain Card';
				$status	=	'Denied';
				break;
			case '35':
				$reason	=	'Card Acceptor, Contact Acquirer, Retain Card';
				$status	=	'Denied';
				break;
			case '36':
				$reason	=	'Restricted Card, Retain Card';
				$status	=	'Denied';
				break;
			case '37':
				$reason	=	'Contact Acquirer Security Department, Retain Card';
				$status	=	'Denied';
				break;
			case '38':
				$reason	=	'PIN Tries Exceeded, Capture';
				$status	=	'Denied';
				break;
			case '39':
				$reason	=	'No Credit Account';
				$status	=	'Denied';
				break;
			case '40':
				$reason	=	'Function Not Supported';
				$status	=	'Denied';
				break;
			case '41':
				$reason	=	'Lost Card';
				$status	=	'Denied';
				break;
			case '42':
				$reason	=	'No Universal Account';
				$status	=	'Denied';
				break;
			case '43':
				$reason	=	'Stolen Card';
				$status	=	'Denied';
				break;
			case '44':
				$reason	=	'No Investment Account';
				$status	=	'Denied';
				break;
			case '51':
				$reason	=	'Insufficient Funds';
				$status	=	'Denied';
				break;
			case '52':
				$reason	=	'No Cheque Account';
				$status	=	'Denied';
				break;
			case '53':
				$reason	=	'No Savings Account';
				$status	=	'Denied';
				break;
			case '54':
				$reason	=	'Expired Card';
				$status	=	'Denied';
				break;
			case '55':
				$reason	=	'Incorrect PIN';
				$status	=	'Denied';
				break;
			case '56':
				$reason	=	'No Card Record';
				$status	=	'Denied';
				break;
			case '57':
				$reason	=	'Function Not Permitted to Cardholder';
				$status	=	'Denied';
				break;
			case '58':
				$reason	=	'Function Not Permitted to Terminal';
				$status	=	'Denied';
				break;
			case '59':
				$reason	=	'Suspected Fraud';
				$status	=	'Denied';
				break;
			case '60':
				$reason	=	'Acceptor Contact Acquirer';
				$status	=	'Denied';
				break;
			case '61':
				$reason	=	'Exceeds Withdrawal Limit';
				$status	=	'Denied';
				break;
			case '62':
				$reason	=	'Restricted Card';
				$status	=	'Denied';
				break;
			case '63':
				$reason	=	'Security Violation';
				$status	=	'Denied';
				break;
			case '64':
				$reason	=	'Original Amount Incorrect';
				$status	=	'Denied';
				break;
			case '66':
				$reason	=	'Acceptor Contact Acquirer, Security';
				$status	=	'Denied';
				break;
			case '67':
				$reason	=	'Capture Card';
				$status	=	'Denied';
				break;
			case '75':
				$reason	=	'PIN Tries Exceeded';
				$status	=	'Denied';
				break;
			case '82':
				$reason	=	'CVV Validation Error';
				$status	=	'Denied';
				break;
			case '90':
				$reason	=	'Cutoff In Progress';
				$status	=	'Denied';
				break;
			case '91':
				$reason	=	'Card Issuer Unavailable';
				$status	=	'Denied';
				break;
			case '92':
				$reason	=	'Unable To Route Transaction';
				$status	=	'Denied';
				break;
			case '93':
				$reason	=	'Cannot Complete, Violation Of The Law';
				$status	=	'Denied';
				break;
			case '94':
				$reason	=	'Duplicate Transaction';
				$status	=	'Denied';
				break;
			case '96':
				$reason	=	'System Error';
				$status	=	'Error';
				break;
			default:
				$reason	=	'Unknown response';
				$status	=	'Error';
				break;
		}

		return $status;
	}

	/**
	 * Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	private function _getBasicRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams									=	array();
		$requestParams['CustomerID']					=	$this->getAccountParam( 'pspid' );
		$requestParams['UserName']						=	$this->getAccountParam( 'pspusername' );
		$requestParams['Amount']						=	sprintf( '%.2f', $paymentBasket->mc_gross );
		$requestParams['Currency']						=	$paymentBasket->mc_currency;
		$requestParams['Language']						=	$this->getAccountParam( 'language' );
		$requestParams['MerchantReference']				=	$paymentBasket->id;
		$requestParams['MerchantInvoice']				=	$paymentBasket->invoice;
		$requestParams['InvoiceDescription']			=	$paymentBasket->item_name;
		$requestParams['MerchantOption1']				=	$paymentBasket->mc_currency;

		// courtesy fields (pre-filled but editable on credit card mask):
		$requestParams['ModifiableCustomerDetails']		=	'true';
		$requestParams['CustomerFirstName']				=	$paymentBasket->first_name;
		$requestParams['CustomerLastName']				=	$paymentBasket->last_name;

		// recommended anti-fraud fields:
		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 50 ) ) {
			$requestParams['CustomerEmail']				=	$paymentBasket->payer_email;
		}

		if ( $this->getAccountParam( 'givehiddenaddress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields								=	array(	'CustomerAddress' => array( $paymentBasket->address_street, 30 ),
																	'CustomerPostCode' => array( $paymentBasket->address_zip, 10 ),
																	'CustomerCity' => array( $paymentBasket->address_city, 30 ),
																	'CustomerCountry' => array( $paymentBasket->getInvoiceCountry( 3 ), 3 ),
																	'CustomerState' => array( $paymentBasket->getInvoiceState(), 30 )
																);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField								=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]					=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddentelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['CustomerPhone']				=	$paymentBasket->contact_phone;
		}

		// urls for return and cancel:
		$returnURL										=	$this->getSuccessUrl( $paymentBasket );
		$cancelURL										=	$this->getCancelUrl( $paymentBasket );
		if ( cbpaidApp::settingsParams()->get( 'https_posts', 0 ) ) {
			$returnURL									=	str_replace( 'http://', 'https://', $returnURL );
			$cancelURL									=	str_replace( 'http://', 'https://', $cancelURL );
		}

		$requestParams['ReturnURL']						=	$returnURL;
		$requestParams['CancelURL']						=	$cancelURL;

		// Static payment page styling:
		$page_title										=	$this->getAccountParam( 'eway_page_title' );

		if ( $page_title ) {
			$requestParams['PageTitle']					=	$page_title;
		}

		$page_descrtion									=	$this->getAccountParam( 'eway_page_description' );

		if ( $page_descrtion ) {
			$requestParams['PageDescription']			=	$page_descrtion;
		}

		$page_footer									=	$this->getAccountParam( 'eway_page_footer' );

		if ( $page_footer ) {
			$requestParams['PageFooter']				=	$page_footer;
		}

		$page_banner									=	$this->getAccountParam( 'eway_page_banner' );

		if ( $page_banner && ( strpos( $page_banner, 'https' ) !== false ) ) {
			$requestParams['PageBanner']				=	$page_banner;
		}

		$company_name									=	$this->getAccountParam( 'eway_company_name' );

		if ( $company_name ) {
			$requestParams['CompanyName']				=	$company_name;
		}

		$company_logo									=	$this->getAccountParam( 'eway_company_logo' );

		if ( $company_logo && ( strpos( $company_logo, 'https' ) !== false ) ) {
			$requestParams['CompanyLogo']				=	$company_logo;
		}

		return $requestParams;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata         Data returned by gateway
	 * @param  string               $type                Type of return ('R' for PDT, 'I' for INS, 'A' for Autorecurring payment (Vault) )
	 * @param  array                $additionalLogData   Additional data fields to log
	 * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret											=	null;
		$error											=	null;

		// PDT doesn't return transacton information; lets request for it:
		$transaction									=	cbGetParam( $requestdata, 'AccessPaymentCode', null );

		if ( $transaction ) {
			$requestParams								=	array();
			$requestParams['CustomerID']				=	$this->getAccountParam( 'pspid' );
			$requestParams['UserName']					=	$this->getAccountParam( 'pspusername' );
			$requestParams['AccessPaymentCode']			=	$transaction;

			$response									=	null;
			$status										=	null;
			$resultUrl									=	$this->gatewayUrl( 'psp' ) . '/Result/?' . http_build_query( $requestParams );
			$error										=	$this->_httpsRequest( $resultUrl, null, 45, $response, $status, 'get/1.0' );

			if ( ( ! $error ) && ( $status == 200 ) && $response ) {
				$xml_response							=	$this->xmlTagValuesToArray( new SimpleXMLElement( $response, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) ) );

				if ( $xml_response ) {
					$requestdata						=	$xml_response;
				} else {
					$error								=	sprintf( 'Malformed XML reply from transaction result check: Got response %s for request %s . Using Requestdata: %s', $response, $resultUrl, var_export( $requestdata, true ) );
				}
			} else {
				$error									=	sprintf( 'Failed accessing transaction result url: %s : Got HTTP Status: %s and reply: %s . Using Requestdata: %s', $resultUrl, $status, $response, var_export( $requestdata, true ) );
			}

			$exists										=	false;
			if ( ! $error ) {
				$paymentBasketId						=	(int) cbGetParam( $requestdata, 'MerchantReference', null );

				if ( $paymentBasketId ) {
					$exists								=	$paymentBasket->load( (int) $paymentBasketId );
					if ( ! $exists ) {
						$error							=	sprintf( 'Payment basket %s (MerchantReference) does not exist for transaction result %', $paymentBasketId, $response );
					}
				} else {
					$error								=	sprintf( 'MerchantReference is missing in the transaction result: %s . Parsed result: %s of request: %s', $response, $requestdata, $resultUrl );
				}
			} else {
				$paymentBasketId						=	null;
			}

			if ( ! $error ) {
				if ( $exists && ( ( cbGetParam( $requestdata, $this->_getPagingParamName( 'id' ), 0 ) == $paymentBasket->shared_secret ) && ( ! ( ( ( $type == 'R' ) || ( $type == 'I' ) ) && ( $paymentBasket->payment_status == 'Completed' ) ) ) ) ) {
					// Log the return record:
					$log_type									=	$type;
					$reason										=	null;
					$paymentStatus								=	$this->_paymentStatus( $requestdata, $reason );
					$paymentType								=	'Credit Card';
					$paymentTime								=	$_CB_framework->now();

					if ( $paymentStatus == 'Error' ) {
						$errorTypes								=	array( 'I' => 'D', 'R' => 'E' );

						if ( isset( $errorTypes[$type] ) ) {
							$log_type							=	$errorTypes[$type];
						}
					}

					$ipn										=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

					if ( $paymentStatus == 'Refunded' ) {
						// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
						$ipn->payment_date						=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
					}

					$ipn->test_ipn								=	( $this->getAccountParam( 'normal_gateway' ) == '0' ? 1 : 0 );		// Eway has no param telling if it is for test or real.
					$ipn->raw_data								=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : 'UNKNOWN' ) ) . '";' . "\n";

					if ( $additionalLogData ) {
						foreach ( $additionalLogData as $k => $v ) {
							$ipn->raw_data						.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
						}
					}

					$ipn->raw_data								.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

					if ( $paymentStatus == 'Error' ) {
						$paymentBasket->reason_code				=	$reason;

						$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
						$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

						$ret									=	false;
					} else {
						$ipn->bindBasket( $paymentBasket );

						$ipn->sale_id							=	$paymentBasketId;

						$insToIpn								=	array(	'txn_id' => 'MerchantReference',
																			'mc_currency' => 'MerchantOption1',
																		);

						foreach ( $insToIpn as $k => $v ) {
							$ipn->$k							=	cbGetParam( $requestdata, $v );
						}

						$ipn->mc_gross							=	sprintf( '%.2f', cbGetParam( $requestdata, 'ReturnAmount' ) );
						$ipn->user_id							=	(int) $paymentBasket->user_id;
						$ipn->txn_type							=	'web_accept';

						if ( ( $paymentBasketId == cbGetParam( $requestdata, 'MerchantReference', null ) ) && ( ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross ) || ( $ipn->payment_status == 'Refunded' ) ) && ( $paymentBasket->mc_currency == $ipn->mc_currency ) ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// add the gateway to the basket:
								$paymentBasket->payment_method	=	$this->getPayName();
								$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, false );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									$ret						=	true;
								}
							} else {
								// Status = 'Denied' or other:
								$this->_storeIpnResult( $ipn, ( $ipn->payment_status == 'Denied' ? 'SUCCESS' :'UKNOWN' ) );

								$paymentBasket->payment_status	=	$ipn->payment_status;

								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': amount or currency missmatch', CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					}
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': ' . $error , CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		}
		return  $ret;
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM basis
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountewayoem extends cbpaidGatewayAccounthostedpage
{
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific class
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaideway extends cbpaidewayoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific class
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccounteway extends cbpaidGatewayAccountewayoem
{
}
