<?php
/**
* @version $Id: cbpaidsubscriptions.payza.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php');

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM base
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidpayzaoem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		$this->_gatewayUrls	=	array(	'psp+normal'	 => $this->getAccountParam( 'psp_normal_url', '' ),
										'psp+test'		 => $this->getAccountParam( 'psp_test_url', '' ),
										'psp+ipn+normal' => $this->getAccountParam( 'psp_ipn_normal_url', '' ),
										'psp+ipn+test'	 => $this->getAccountParam( 'psp_ipn_test_url', '' ),
										'psp+api+normal' => $this->getAccountParam( 'psp_api_normal_url', '' ),
										'psp+api+test'	 => $this->getAccountParam( 'psp_api_test_url', '' ) );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams						=	$this->_getBasicRequstParams( $paymentBasket );

		// sign single payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams									=	$this->_getBasicRequstParams( $paymentBasket );

		// check for subscription or if single payment:
		if ( $paymentBasket->period3 ) {
			// calculate duration and type:
			list( $duration, $type )					=	$this->_periodsLimits( explode( ' ', $paymentBasket->period3 ) );

			// reset type to a subscription:
			$requestParams['ap_purchasetype']			=	'subscription';

			// add subscription specific parameters:
			$requestParams['ap_amount']					=	sprintf( '%.2f', $paymentBasket->mc_amount3 );
			$requestParams['ap_timeunit']				=	$type;
			$requestParams['ap_periodlength']			=	$duration;
			$requestParams['ap_periodcount']			=	$paymentBasket->recur_times;

			if ( $paymentBasket->period1 ) {
				// calculate duration and type for initial payment:
				list( $duration, $type )				=	$this->_periodsLimits( explode( ' ', $paymentBasket->period1 ) );

				// add initial subscription specific parameters:
				$requestParams['ap_trialamount']		=	sprintf( '%.2f', $paymentBasket->mc_amount1 );
				$requestParams['ap_trialtimeunit']		=	$type;
				$requestParams['ap_trialperiodlength']	=	$duration;
			}
		}

		// sign subscription payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	* The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	*/
	protected function handleReturn( $paymentBasket, $postdata )
	{
		// EPD encrypts the request data, we need to decode it:
		$requestdata	=	$this->_decodeEPD( $this->_getGetParams() );

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	* The user cancelled his payment
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );
			$exists								=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status	=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}

			$ret								=	false;
		} else {
			$this->_setErrorMSG( CBTxt::T( 'Invalid request.' ) );
			$ret								=	null;
		}

		return  $ret;
	}

	/**
	* The payment service provider server did a server-to-server notification: verify and handle it here:
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleNotification( $paymentBasket, $postdata )
	{
		// IPN v2 encrypts the request data, we need to decode it:
		$requestdata	=	$this->_decodeIPNV2( $_POST );

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' );
	}

	/**
	 * Cancels an existing recurring subscription
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  cbpaidPaymentItem[]  $paymentItems   payment items
	 * @return boolean|string                       TRUE if unsubscription done successfully, STRING if error
	 */
	protected function handleStopPaymentSubscription( $paymentBasket, $paymentItems )
	{
		global $_CB_framework;

		$return										=	false;

		// only for recurring subscriptions:
		if ( $paymentBasket->mc_amount3 ) {
			$subscription_id						=	$paymentBasket->subscr_id;

			// only if an actual subscription id exists:
			if ( $subscription_id ) {
				// curl is required to form the request:
				if ( function_exists( 'curl_init' ) ) {
					// send API request to cancel subscription:
					$response						=	null;
					$formvars						=	array(	'USER' => $this->getAccountParam( 'pspid' ),
																'PASSWORD' => $this->getAccountParam( 'psp_api_psw' ),
																'SUBSCRIPTIONREFERENCE' => $subscription_id,
																'NOTE' => CBTxt::T( 'Subscription cancellation on-site.' )
															);
					$request						=	curl_init();

					curl_setopt( $request, CURLOPT_URL, $this->_pspapiUrl() . '/CancelSubscription' );
					curl_setopt( $request, CURLOPT_POST, true );
					curl_setopt( $request, CURLOPT_POSTFIELDS, http_build_query( $formvars, null, '&' ) );
					curl_setopt( $request, CURLOPT_RETURNTRANSFER, true );
					curl_setopt( $request, CURLOPT_HEADER, false );
					curl_setopt( $request, CURLOPT_TIMEOUT, 30 );
					curl_setopt( $request, CURLOPT_SSL_VERIFYPEER, false );

					$response						=	curl_exec( $request );

					curl_close( $request );

					if ( $response ) {
						// parse from URL variables to array:
						parse_str( urldecode( $response ), $data );

						if ( $data ) {
							$error					=	(int) cbGetParam( $data, 'RETURNCODE', 0 );

							// successful cancellation; log response:
							if ( $error == '100' ) {
								$ipn				=	$this->_prepareIpn( 'R', $paymentBasket->payment_status, $paymentBasket->payment_type, 'Unsubscribe', $_CB_framework->now(), 'utf-8' );
								$ipn->test_ipn		=	( cbGetParam( $data, 'TESTMODE' ) == '1' ? 1 : 0 );
								$ipn->raw_result	=	'SUCCESS';
								$ipn->raw_data		=	'$message_type="STOP_PAYMENT_SUBSCRIPTION"' . ";\n"
													.	/* cbGetParam() not needed: we want raw info */ '$data=' . var_export( $data, true ) . ";\n"
													.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
													.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

								$ipn->bindBasket( $paymentBasket );

								$bskToIpn			=	array(	'sale_id' => 'sale_id',
																'txn_id' => 'txn_id',
																'subscr_id' => 'subscr_id'
															);

								foreach ( $bskToIpn as $k => $v ) {
									$ipn->$k		=	$paymentBasket->$v;
								}

								$ipn->txn_type		=	'subscr_cancel';

								if( ! $ipn->store() ) {
									$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe IPN failed to store', CBTxt::T( 'Submitted unsubscription failed on-site. Please cancel from your Payza account at My Account - Profile - Financial - Subscriptions.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
								} else {
									$return			=	true;
								}
							} else {
								$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API failed: ' . $this->_decodeAPIError( $error ), CBTxt::T( 'Submitted unsubscription failed on-site. Please cancel from your Payza account at My Account - Profile - Financial - Subscriptions.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
							}
						} else {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API reponse empty', CBTxt::T( 'Submitted unsubscription failed on-site. Please cancel from your Payza account at My Account - Profile - Financial - Subscriptions.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						}
					} else {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API response failed', CBTxt::T( 'Submitted unsubscription failed on-site. Please cancel from your Payza account at My Account - Profile - Financial - Subscriptions.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}
				} else {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe CURL requirement not met', CBTxt::T( 'Submitted unsubscription failed on-site. Please cancel from your Payza account at My Account - Profile - Financial - Subscriptions.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe failed from missing subscr_id in payment basket', CBTxt::T( 'Submitted unsubscription failed on-site. Please cancel from your Payza account at My Account - Profile - Financial - Subscriptions.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		}

		return $return;
	}

	/**
	 * GATEWAY-INTERNAL SPECIFIC PRIVATE METHODS:
	 */

	/**
	 * gives gateway IPN URL server name from gateway URL list
	 *
	 * @return string server-name (with 'https://' )
	 */
	private function _pspipnUrl( )
	{
		return $this->gatewayUrl( 'psp+ipn' );
	}

	/**
	 * gives gateway API URL server name from gateway URL list
	 *
	 * @return string server-name (with 'https://' )
	 */
	private function _pspapiUrl( )
	{
		return $this->gatewayUrl( 'psp+api' );
	}

	/**
	 * generate validation to check on return of PDT or IPN
	 *
	 * @param  array     $requestParams        The keyed parameters array to generate validation for
	 * @param  boolean   $rawParams            $requestParams are raw $_POST or $_GET input and should be sanitized and unescaped if needed
	 * @return string                          MD5 hash as specified by gateway
	 */
	private function _pspSignMD5( $requestParams, $rawParams )
	{
		$listOfParams				=	array(
										'AP_ITEMCODE',			// item code (basket id)
										'AP_PURCHASETYPE',		// purchase type (item or subscription)
										'AP_CURRENCY',			// currency code (USD, etc..)
										'AP_AMOUNT',			// purchase price (10.01, etc..)
										'AP_QUANTITY',			// purchase quantity (usually always 1)
										'$psp_security_code'	// Concatenate security code generated at gateway: equivalent to: $this->getAccountParam( 'pspsecret' )
									);
		// Concatenate them using this payments concatenation function with $caseInsensitiveKeys = true:
		$string						=	$this->_concatVars( $requestParams, $listOfParams, null, '', '', false, false, true, false, $rawParams );

		// compute validation code doing md5 of the string with uppercasing:
		return $this->_hashString( $string, 'md5', true );
	}

	/**
	 * sign payment request $requestParams with validation code added to $requestParams array
	 *
	 * @param  array     $requestParams        The keyed parameters array to add generate validation to $requestParams['apc_6']
	 */
	private function _signRequestParams( &$requestParams )
	{
		// add validation code:
		$requestParams['apc_6']	=	$this->_pspSignMD5( $requestParams, false );
	}

	/**
	 * validate PDT or IPN using signed validation code sent to gateway
	 *
	 * @param  array $requestParams     Checks $requestParams['apc_6'] towards determining values
	 * @return boolean                  True: valid, False: not valid
	 */
	private function _pspVerifySignature( $requestParams )
	{
		// generate validation code for data returned from gateway:
		$string					=	$this->_pspSignMD5( $requestParams, true );

		// confirm validation:
		return ( $string === cbGetParam( $requestParams, 'apc_6' ) );
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's params:
	 *
	 * STATUS:	Status of the payment:
		Statuses in 1 digit are 'normal' statuses:
		0 means the payment is invalid (e.g. data validation error) or the processing is not complete either because it is still underway, or because the transaction was interrupted. If the cause is a validation error, an additional error code (*) (NCERROR) identifies the error.
		1 means the customer cancelled the transaction.
		2 means the acquirer did not authorise the payment.
		5 means the acquirer autorised the payment.
		9 means the payment was captured.
		Statuses in 2 digits correspond either to 'intermediary' situations or to abnormal events. When the second digit is:
		1, this means the payment processing is on hold.
		2, this means an unrecoverable error occurred during the communication with the acquirer. The result is therefore not determined. You must therefore call the acquirer's helpdesk to find out the actual result of this transaction.
		3, this means the payment processing (capture or cancellation) was refused by the acquirer whilst the payment had been authorised beforehand. It can be due to a technical error or to the expiration of the authorisation. You must therefore call the acquirer's helpdesk to find out the actual result of this transaction.
		4, this means our system has been notified the transaction was rejected well after the transaction was sent to your acquirer.
		5, this means our system hasn't sent the requested transaction to the acquirer since the merchant will send the transaction to the acquirer himself, like he specified in his configuration.
	 *
	 * ACCEPTANCE:
	 * Acquirer's acceptance (authorization) code.
	 * The acquirer sends back this code to confirm the amount of the transaction has been blocked on the card of the customer. The acceptance code is not unique.
	 *
	 * @param  array                $postdata        raw POST data received from the payment gateway
	 * @param  string               $reason          OUT: reason_code
	 * @param  string               $previousStatus  previous CBSubs status
	 * @param  cbpaidPaymentBasket  $paymentBasket   (only for error logging purposes)
	 * @return string
	 */
	private function _paymentStatus( $postdata, &$reason, /** @noinspection PhpUnusedParameterInspection */ $previousStatus, /** @noinspection PhpUnusedParameterInspection */ &$paymentBasket )
	{
		$accept_payment_condition	=	$this->getAccountParam( 'accept_payment_condition', '' );
		$status						=	cbGetParam( $postdata, 'ap_status', null );

		switch ( $status ) {
			case 'Success':
			case 'Subscription-Payment-Success':
				$reason				=	null;
				$status				=	'Completed';
				break;
			case 'Subscription-Expired':
				$reason				=	'Means total amount for the product for all the subscriptions are paid successfully';
				$status				=	'Denied';
				break;
			case 'Subscription-Payment-Failed':
				$reason				=	'Buyer does not have enough funds for the subscription payment and run date will be rescheduled two times';
				$status				=	'Error';
				break;
			case 'Subscription-Payment-Rescheduled':
				if ( ( $accept_payment_condition == 'pending' ) || ( $accept_payment_condition == 'authorized' ) ) {
					$reason			=	null;
					$status			=	'Completed';
				} else {
					$reason			=	'Due to the failed attempt, the payment has been rescheduled';
					$status			=	'Pending';
				}
				break;
			case 'Subscription-Canceled':
				$reason				=	'The merchant or the buyer explicitly canceled the subscription or Payza cancelled it since the buyer didn’t have enough money after rescheduling two times';
				$status				=	'Denied';
				break;
		}

		return $status;
	}

	/**
	* Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	*
	* @param cbpaidPaymentBasket $paymentBasket paymentBasket object
	* @return array $requestParams
	*/
	private function _getBasicRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams								=	array();
		$requestParams['ap_merchant']				=	$this->getAccountParam( 'pspid' );
		$requestParams['ap_purchasetype']			=	'item';
		$requestParams['ap_itemcode']				=	$paymentBasket->id;
		$requestParams['ap_itemname']				=	$paymentBasket->item_name;
		$requestParams['ap_amount']					=	sprintf( '%.2f', $paymentBasket->mc_gross );
		$requestParams['ap_quantity']				=	$paymentBasket->quantity;
		$requestParams['ap_currency']				=	$paymentBasket->mc_currency;

		// courtesy fields (pre-filled but editable on credit card mask):
		$requestParams['ap_fname']					=	$paymentBasket->first_name;
		$requestParams['ap_lname']					=	$paymentBasket->last_name;

		// recommended anti-fraud fields:
		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 50 ) ) {
			$requestParams['ap_contactemail']		=	$paymentBasket->payer_email;
		}

		if ( $this->getAccountParam( 'givehiddenaddress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields							=	array(	'ap_addressline1' => array( $paymentBasket->address_street, 30 ),
																'ap_zippostalcode' => array( $paymentBasket->address_zip, 10 ),
																'ap_city' => array( $paymentBasket->address_city, 30 ),
																'ap_country' => array( $paymentBasket->getInvoiceCountry( 3 ), 3 ),
																'ap_stateprovince' => array( $paymentBasket->getInvoiceState(), 30 )
															);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField							=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]				=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddentelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['ap_contactphone']		=	$paymentBasket->contact_phone;
		}

		// urls for return, cancel, and IPNs:
		$requestParams['ap_returnurl']				=	$this->getSuccessUrl( $paymentBasket );
		$requestParams['ap_cancelurl']				=	$this->getCancelUrl( $paymentBasket );
		$requestParams['ap_alerturl']				=	$this->getNotifyUrl( $paymentBasket );		// This uses our account parameter $this->getAccountParam( 'notifications_host', '' )

		return $requestParams;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata         Data returned by gateway
	 * @param  string               $type                Type of return ('R' for PDT, 'I' for INS )
	 * @param  array|null           $additionalLogData   Additional data to log
	 * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret													=	null;
		$paymentBasketId										=	(int) cbGetParam( $requestdata, 'ap_itemcode', 0 );

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( ( ( cbGetParam( $requestdata, $this->_getPagingParamName( 'id' ), 0 ) == $paymentBasket->shared_secret ) && ( ! ( ( ( $type == 'R' ) || ( $type == 'I' ) ) && ( $paymentBasket->payment_status == 'Completed' ) ) ) ) ) ) {
				// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason, $paymentBasket->payment_status, $paymentBasket );
				$paymentType									=	'Credit Card';
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E' );

					if ( isset( $errorTypes[$type] ) ) {
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	( ( $this->getAccountParam( 'normal_gateway' ) == '0' ) || ( cbGetParam( $requestdata, 'ap_tes' ) == '1' ) ? 1 : 0 );
				$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : 'UNKNOWN' ) ) . '";' . "\n";

				if ( $additionalLogData ) {
					foreach ( $additionalLogData as $k => $v ) {
						$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
					}
				}

				$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	$reason;

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					$ret										=	false;
				} else {
					$ipn->bindBasket( $paymentBasket );

					$insToIpn									=	array(	'receiver_email' => 'ap_merchant',
																			'mc_currency' => 'ap_currency',
																			'quantity' => 'ap_quantity',
																			'sale_id' => 'ap_itemcode',
																			'txn_id' => 'ap_referencenumber',
																			'first_name' => 'ap_custfirstname',
																			'last_name' => 'ap_custlastname',
																			'address_street' => 'ap_custaddress',
																			'address_zip' => 'ap_custzip',
																			'address_city' => 'ap_custcity',
																			'address_country' => 'ap_custcountry',
																			'address_state' => 'ap_custstate',
																			'payer_email' => 'ap_custemailaddress'
																		);

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $requestdata, $v );
					}

					$ipn->mc_gross								=	sprintf( '%.2f', cbGetParam( $requestdata, 'ap_amount' ) );

					$psp_fee									=	sprintf( '%.2f', cbGetParam( $requestdata, 'ap_feeamount' ) );

					if ( $psp_fee && ( $psp_fee != '0.00' ) ) {
						$ipn->mc_fee							=	$psp_fee;
					}

					$ipn->user_id								=	(int) $paymentBasket->user_id;

					$recurring									=	( cbGetParam( $requestdata, 'ap_purchasetype' ) == 'subscription' || cbGetParam( $requestdata, 'ap_subscriptionreferencenumber' ) ? true : false );

					if ( $recurring ) {
						if ( ( $paymentStatus == 'Completed' ) && ( ! $paymentBasket->subscr_id ) ) {
							$ipn->txn_type						=	'subscr_signup';
							$ipn->subscr_id						=	cbGetParam( $requestdata, 'ap_subscriptionreferencenumber' );
							$ipn->subscr_date					=	$ipn->payment_date;
						} elseif ( $paymentStatus == 'Denied' ) {
							if ( ( $paymentBasket->reattempts_tried + 1 ) <= cbpaidScheduler::getInstance( $this )->retries ) {
								$ipn->txn_type					=	'subscr_failed';
							} else {
								$ipn->txn_type					=	'subscr_cancel';
							}
						} elseif ( in_array( $paymentStatus, array( 'Completed', 'Processed', 'Pending' ) ) ) {
							$ipn->txn_type						=	'subscr_payment';
						}
					} else {
						$ipn->txn_type							=	'web_accept';
					}

					// validate payment from PDT or IPN
					if ( $this->_pspVerifySignature( $requestdata ) ) {
						if ( ( $paymentBasketId == cbGetParam( $requestdata, 'ap_itemcode' ) ) && ( ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross ) || ( $ipn->payment_status == 'Refunded' ) ) && ( $paymentBasket->mc_currency == $ipn->mc_currency ) ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// add the gateway to the basket:
								$paymentBasket->payment_method	=	$this->getPayName();
								$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

								// 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date:
								$autorecurring_type				=	( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed' ) ) ? 2 : 0 );

								// 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration:
								$autorenew_type					=	( $autorecurring_type ? ( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 ) : 0 );

								if ( $recurring ) {
									$paymentBasket->reattempt	=	1; // we want to reattempt auto-recurring payment in case of failure
								}

								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, false );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									$ret						=	true;
								}
							} else {
								$this->_storeIpnResult( $ipn, 'FAILED' );

								$paymentBasket->payment_status	=	$ipn->payment_status;

								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': ap_itemcode, amount or currency missmatch', CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					} else {
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': apc_6 does not match with gateway. Please check IPN Security Code setting', CBTxt::T( 'The IPN Security Code signature is incorrect.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

						$ret									=	false;
					}
				}
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The ap_itemcode is missing in the return URL', CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return  $ret;
	}

	/**
	 * exchange token sent with IPN for IPN data
	 *
	 * @param array $requestParams
	 * @return array
	 */
	private function _decodeIPNV2( $requestParams )
	{
 		// check for IPN token variable:
		$token									=	cbGetParam( $requestParams, 'token' );

		// only request IPN if token exists:
		if ( $token ) {
			if ( function_exists( 'curl_init' ) ) {
				$response						=	null;
				$formvars						=	array( 'token' => $token );
				$request						=	curl_init();

				curl_setopt( $request, CURLOPT_URL, $this->_pspipnUrl() );
				curl_setopt( $request, CURLOPT_POST, true );
				curl_setopt( $request, CURLOPT_POSTFIELDS, http_build_query( $formvars, null, '&' ) );
				curl_setopt( $request, CURLOPT_RETURNTRANSFER, true );
				curl_setopt( $request, CURLOPT_HEADER, false );
				curl_setopt( $request, CURLOPT_TIMEOUT, 60 );
				curl_setopt( $request, CURLOPT_SSL_VERIFYPEER, false );

				$response						=	curl_exec( $request );

				curl_close( $request );

				if ( $response && ( $response != 'INVALID TOKEN' ) ) {
					// parse from URL variables to array:
					parse_str( urldecode( $response ), $ipn );

					if ( $ipn ) {
						// remove the IPN token variable:
						unset( $requestParams['token'] );

						// add payment information to request params:
						foreach ( $ipn as $k => $v ) {
							$requestParams[$k]	=	$v;
						}
					} else {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': IPNv2 request contained no data', CBTxt::T( 'No IPN data was found.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}
				} else {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': IPNv2 token failed to match', CBTxt::T( 'Sorry, the IPN token does not match.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': IPNv2 CURL requirement not met', CBTxt::T( 'CURL not installed.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': No IPNv2 token was found', CBTxt::T( 'No IPN token was found.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return $requestParams;
	}

	/**
	 * parse encoded data sent with return url
	 *
	 * @param array $requestParams
	 * @return array
	 */
	private function _decodeEPD( $requestParams )
	{
 		// check for EPD crypt variable:
		$crypt							=	cbGetParam( $requestParams, 'AP' );

		// only decrypt if anything actually exists:
		if ( $crypt ) {
			// decode the base64 encoded string:
			$encoded					=	base64_decode( $crypt );

			// complete validation key:
			$security_code				=	$this->getAccountParam( 'psp_security_code' );
			$key						=	$security_code . substr( $security_code, 0, ( 24 - strlen( $security_code ) ) );

			if ( is_callable( '' ) ) { // disabled on purpose as does not work
				// use mcrypt library for decryption if available:
				$decoded				=	mcrypt_decrypt( MCRYPT_3DES, $key, $encoded, MCRYPT_MODE_CBC, 'payza' );
			} else {
				// as mcrypt is not a prerequisite for CBSubs, use pure PHP implementation of 3DES if needed:
				cbpaidApp::import( 'processors.payza.payza_crypt' );
				$des					=	new CBPSAP_Crypt_TripleDES();
				$des->setKey( $key );
				$des->setIV( 'payza' );
				$decoded				=	$des->decrypt( $encoded );
			}

			// parse from URL variables to array:
			parse_str( trim( $decoded, "\x00..\x1F" ), $epd );

			if ( $epd ) {
				// removed the EPD crypt variable:
				unset( $requestParams['AP'] );

				// add decoded payment information to request params:
				foreach ( $epd as $k => $v ) {
					$requestParams[$k]	=	$v;
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': EPD failed to decrypt or find IPN data within encrypted string', CBTxt::T( 'No IPN data was found.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': No EPD crypt key was found', CBTxt::T( 'No IPN crypt was found.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return $requestParams;
	}

	/**
	 * parse API response error code
	 *
	 * @param int $code
	 * @return string
	 */
	private function _decodeAPIError( $code )
	{
		switch ( $code ) {
			case '100':
				$response	=	'Success';
				break;
			case '201':
				$response	=	'Missing parameter USER in the request';
				break;
			case '202':
				$response	=	'Missing parameter PASSWORD in the request';
				break;
			case '203':
				$response	=	'Missing parameter RECEIVEREMAIL in the request';
				break;
			case '204':
				$response	=	'Missing parameter AMOUNT in the request';
				break;
			case '206':
				$response	=	'Missing parameter CURRENCY in the request';
				break;
			case '206':
				$response	=	'Missing parameter PURCHASETYPE in the request';
				break;
			case '211':
				$response	=	'Invalid format for parameter USER. Value must be a valid e-mail address in the following format: username@example.com';
				break;
			case '212':
				$response	=	'Invalid format for parameter PASSWORD. Value must be a 16 character alpha-numeric string';
				break;
			case '213':
				$response	=	'Invalid format for parameter AMOUNT. Value must be numeric';
				break;
			case '214':
				$response	=	'Invalid value for parameter CURRENCY. Value must be a three character string representing an ISO-4217 currency code accepted by Payza';
				break;
			case '215':
				$response	=	'Invalid format for parameter RECEIVEREMAIL. Value must be a valid e-mail address in the following format: username@example.com';
				break;
			case '216':
				$response	=	'The format for parameter NOTE is invalid';
				break;
			case '217':
				$response	=	'Invalid value for parameter TESTMODE. Value must be either 0 or 1';
				break;
			case '218':
				$response	=	'Invalid value for parameter PURCHASETYPE. Value must be an integer number between 0 and 3';
				break;
			case '219':
				$response	=	'Invalid format for parameter SENDEREMAIL. Value must be a valid e-mail address in the following format: username@example.com';
				break;
			case '221':
				$response	=	'Cannot perform the request. Invalid USER and PASSWORD combination';
				break;
			case '222':
				$response	=	'Cannot perform the request. API Status is disabled for this account';
				break;
			case '223':
				$response	=	'Cannot perform the request. Action cannot be performed from this IP address';
				break;
			case '224':
				$response	=	'Cannot perform the request. USER account is not active';
				break;
			case '225':
				$response	=	'Cannot perform the request. USER account is locked';
				break;
			case '226':
				$response	=	'Cannot perform the request. Too many failed authentications. The API has been momentarily disabled for your account. Please try again later';
				break;
			case '231':
				$response	=	'Incomplete transaction. Amount to be sent must be positive and greater than 1.00';
				break;
			case '232':
				$response	=	'Incomplete transaction. Amount to be sent cannot be greater than the maximum amount';
				break;
			case '233':
				$response	=	'Incomplete transaction. You have insufficient funds in your account';
				break;
			case '234':
				$response	=	'Incomplete transaction. You are attempting to send more than your sending limit';
				break;
			case '235':
				$response	=	'Incomplete transaction. You are attempting to send more than your monthly sending limit';
				break;
			case '236':
				$response	=	'Incomplete transaction. You are attempting to send money to yourself';
				break;
			case '237':
				$response	=	'Incomplete transaction. You are attempting to send money to an account that cannot accept payments';
				break;
			case '238':
				$response	=	'Incomplete transaction. The recipient of the payment does not accept payments from unverified members';
				break;
			case '239':
				$response	=	'Invalid value for parameter NOTE. The field cannot exceed 1000 characters';
				break;
			case '240':
				$response	=	'Error with parameter SENDEREMAIL. The specified e-mail is not associated with your account';
				break;
			case '241':
				$response	=	'Error with parameter SENDEREMAIL. The specified e-mail has not been validated';
				break;
			case '242':
				$response	=	'Incomplete transaction. The recipient’s account is temporarily suspended and cannot receive money';
				break;
			case '243':
				$response	=	'Incomplete transaction. The recipient only accepts funds from members in the same country';
				break;
			case '244':
				$response	=	'Incomplete transaction. The recipient cannot receive funds at this time, please try again later';
				break;
			case '245':
				$response	=	'Incomplete transaction. The amount you are trying to send exceeds your transaction limit as an Unverified Member';
				break;
			case '246':
				$response	=	'Incomplete transaction. Your account must be Verified in order to transact money';
				break;
			case '247':
				$response	=	'Unsuccessful refund. Transaction does not belong to this account';
				break;
			case '248':
				$response	=	'Unsuccessful refund. Transaction does not exist in our system';
				break;
			case '249':
				$response	=	'Unsuccessful refund. Transaction is no longer refundable';
				break;
			case '250':
				$response	=	'Unsuccessful cancellation. Subscription does not belong to this account';
				break;
			case '251':
				$response	=	'Unsuccessful cancellation. Subscription does not exist in our system';
				break;
			case '252':
				$response	=	'Unsuccessful cancellation. Subscription is already canceled';
				break;
			case '260':
				$response	=	'Unsuccessful query. The specified CURRENCY balance is NOT open in your account';
				break;
			case '299':
				$response	=	'An unexpected error occurred';
				break;
			default:
				$response	=	'Failed';
				break;
		}

		return $response;
	}

	/**
	 * parse period into period and type then calculate new period limitations:
	 *
	 * @param array $periodTypeArray
	 * @return array
	 */
	private function _periodsLimits( $periodTypeArray )
	{
		// parse period into period and type:
		$p		=	$periodTypeArray[0];
		$t		=	$periodTypeArray[1];

		// change single letter type to full length:
		if ( $t == 'Y' ) {
			$t	=	'Year';
		} elseif ( $t == 'M' ) {
			$t	=	'Month';
		} elseif  ( $t == 'W' ) {
			$t	=	'Week';
		} elseif ( $t == 'D' ) {
			$t	=	'Day';
		}

		return array( $p, $t );
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountpayzaoem extends cbpaidGatewayAccounthostedpage
{
	/**
	 * USED by XML interface ONLY !!!
	 * Renders URL for successful returns
	 * We are overriding this method, as Payza accepts non-https URLs:
	 *
	 * @param  string              $value   Variable value ( 'successurl', 'cancelurl', 'notifyurl' )
	 * @param  ParamsInterface     $params
	 * @param  string              $name    The name of the form element
	 * @param  CBSimpleXMLElement  $node    The xml element for the parameter
	 * @return string                       HTML to display
	 */
	public function renderUrl( $value, $params, $name, $node )
	{
		return $this->getPayMean()->adminUrlRender( $node->attributes( 'value' ) );
	}
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidpayza extends cbpaidpayzaoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountpayza extends cbpaidGatewayAccountpayzaoem
{
}
