<?php
/**
* @version $Id: cbpaidsubscriptions.quickpay.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

/** Ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM base
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidquickpayoem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		$this->_gatewayUrls	=	array(	'psp+normal'		=>	$this->getAccountParam( 'psp_normal_url' ),
										'psp+test'			=>	$this->getAccountParam( 'psp_test_url' ),
										'psp+api+normal'	=>	$this->getAccountParam( 'psp_api_normal_url' ),
										'psp+api+test'		=>	$this->getAccountParam( 'psp_api_test_url' ) );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams		=	$this->_getBasicRequstParams( $paymentBasket );

		// sign single payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	 * NOT IMPLEMENTED FOR THIS GATEWAY:
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams						=	$this->_getBasicRequstParams( $paymentBasket );

		// check for subscription or if single payment:
		if ( $paymentBasket->period3 ) {
			$requestParams['type']			=	'subscription';

			if ( $paymentBasket->period1 ) {
				$requestParams['amount']	=	( sprintf( '%.2f', $paymentBasket->mc_amount1 ) * 100 );
			} else {
				$requestParams['amount']	=	( sprintf( '%.2f', $paymentBasket->mc_amount3 ) * 100 );
			}
		}

		// sign reocurring payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}


	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['order_id'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata					=	$postdata;
		} else {
			// but if customer needs GET, we will work with it too (removing CMS/CB/CBSubs specific routing params):
			$requestdata					=	$this->_getGetParams();
		}

		// check if order_id exists and if not add basket id from PDT success url:
		if ( ! isset( $requestdata['order_id'] ) ) {
			$requestdata['order_id']		=	(int) cbGetParam( $_GET, 'cbpbasket', null );
		}

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId				=	$this->_getBasketID( cbGetParam( $requestdata, 'order_id', null ) );
			}

			$exists								=	$paymentBasket->load( $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$this->_setCancelCount( $paymentBasket );

				$paymentBasket->payment_status	=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}

		return false;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                              Text to return to gateway if notification, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		$rawpostdata			=	file_get_contents( 'php://input' );

		if ( $rawpostdata && ( ( $rawpostdata[0] === '{' ) || ( $rawpostdata[0] === '[' ) ) ) {
			$postdata			=	json_decode( $rawpostdata, true );
		}

		if ( ( count( $postdata ) > 0 ) && isset( $postdata['order_id'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata		=	$postdata;
		} else {
			// but if gateway needs GET, we will work with it too:
			$requestdata		=	$this->_getGetParams();
		}

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' );
	}

	/**
	 * Attempts to authorize and capture a credit card for a single payment of a payment basket using PSP DirectLink
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $returnText                  RETURN param
	 * @param  boolean              $transientErrorDoReschedule  RETURN param
	 * @return boolean              TRUE: succes, FALSE: failed or unknown result
	 */
	protected function processScheduledAutoRecurringPayment( $paymentBasket, &$returnText, &$transientErrorDoReschedule )
	{
		global $_CB_framework;

		// https://learn.quickpay.net/tech-talk/api/services/#subscriptions
		$results								=	array();
		$response								=	null;
		$status									=	null;
		$error									=	null;

		if ( $paymentBasket->payment_status == 'Pending' ) {
			// Basket is pending a recurring payment so lets grab the existing recurring payment to check its status:
			$requestParams 						= 	array(	'order_id'		=>	$this->_getOrderID( $paymentBasket, ( $paymentBasket->recur_times_used + 1 ) ),
															'sort_by'		=>	'created_at',
															'sort_dir'		=>	'desc'
														);

			$error								=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/payments', $requestParams, 105, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_key' ) );

			if ( in_array( $status, array( 200, 202 ) ) && $response ) {
				$responseData					=	json_decode( $response, true );

				if ( $responseData && isset( $responseData[0] ) ) {
					$results					=	$responseData[0];
				}
			}
		}

		if ( ! $results ) {
			// Existing recurring payment doesn't exist so lets create it:
			$requestParams 						= 	array(	'order_id'		=>	$this->_getOrderID( $paymentBasket, ( $paymentBasket->recur_times_used + 1 ) ),
															'amount'		=>	( sprintf( '%2.f', $paymentBasket->mc_amount3 ) * 100 ),
															'auto_capture'	=>	true
														);

			$error								=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/subscriptions/' . $paymentBasket->subscr_id . '/recurring', $requestParams, 105, $response, $status, 'post', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_user_key' ) );

			if ( $response ) {
				$results						=	json_decode( $response, true );
			}

			if ( $status == 202 ) {
				// We're using instant capture and Quickpay can be a bit slow here so lets pause for a second then retry grabbing the payment:
				sleep( 1 );

				$error							=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/payments/' . cbGetParam( $results, 'id', null ), array(), 105, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_key' ) );

				if ( in_array( $status, array( 200, 202 ) ) && $response ) {
					$responseData				=	json_decode( $response, true );

					if ( $responseData ) {
						$results				=	$responseData;
					}
				}
			}
		}

		if ( ( ! in_array( $status, array( 200, 202 ) ) ) && ( $error || ( ! $response ) ) ) {
			$ipn								=	$this->_prepareIpn( 'B', $paymentBasket->payment_status, 'Autorecurring', null, $_CB_framework->now(), 'utf-8' );

			$ipn->bindBasket( $paymentBasket );

			$errorText							=	'HTTPS POST Connection to payment gateway server failed (check system information in CBSubs Settings): ERROR: ' . $error . ' (' . ( $status == -100 ? 'Timeout' : $status ) . ')';
			$rawData							=	'$response="' . preg_replace( '/([^\s]{100})/', '$1 ', $response ) . "\"\n"
												.	'$results=' . var_export( $results, true ) . ";\n"
												.	'$requestParams=' . var_export( $requestParams, true ) . ";\n";

			$ipn->setRawResult( $errorText );
			$ipn->setRawData( $rawData );

			$this->_storeIpnResult( $ipn, 'FAILED' );
			$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $errorText, CBTxt::T( "Sorry, the payment server did not reply." ) . ' ' . CBTxt::T( "Please contact site administrator to check payment status and error log." ) );

			$user								=	CBuser::getUserDataInstance( $paymentBasket->user_id );
			$username							=	( $user ? $user->username : '?' );
			$returnText							=	sprintf( CBTxt::T( "FAILED Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), 'HTTP error ' . ': ' . $error . ' ' . 'Status' . ': ' . $status );
			$transientErrorDoReschedule			=	true;
			$return								=	false;
		} else {
			$paymentResult						=	$this->_returnParamsHandler( $paymentBasket, $results, 'A', array( '$requestParams' => $requestParams ) );

			$user								=	CBuser::getUserDataInstance( $paymentBasket->user_id );
			$username							=	( $user ? $user->username : '?' );

			if ( $paymentResult !== false ) {
				if ( ( $paymentResult === true ) && in_array( $paymentBasket->payment_status, array( 'Completed', 'Pending' ) ) ) {
					if ( $paymentBasket->payment_status == 'Completed' ) {
						$returnText						=	sprintf( CBTxt::T( "Completed Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s with txn_id %s and auth_id %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->txn_id, $paymentBasket->auth_id );
						$transientErrorDoReschedule		=	false;
						$return							=	true;
					} else {
						$returnText						=	sprintf( CBTxt::T( "Pending Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s with txn_id %s and auth_id %s for reason: %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->txn_id, $paymentBasket->auth_id, $paymentBasket->reason_code );
						$transientErrorDoReschedule		=	true;
						$return							=	false;
					}
				} else {
					$returnText					=	sprintf( CBTxt::T( "FAILED (%s) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s." ), $paymentBasket->translatedPaymentStatus(), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->reason_code );
					$transientErrorDoReschedule	=	true;
					$return						=	false;
				}
			} else {
				$returnText						=	sprintf( CBTxt::T( "FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s." ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->reason_code );
				$transientErrorDoReschedule		=	true;
				$return							=	false;
			}
		}

		return $return;
	}

	/**
	* Cancels a subscription
	*
	* @param  cbpaidPaymentBasket   $paymentBasket  paymentBasket object
	* @param  cbpaidPaymentItem[]   $paymentItems   redirect immediately instead of returning HTML for output
	* @return boolean                               true if unsubscription done successfully, false if error
	*/
	protected function handleStopPaymentSubscription( $paymentBasket, $paymentItems )
	{
		if ( $paymentBasket->mc_amount3 ) {
			if ( $paymentBasket->subscr_id  ) {
				// Cancel the subscription at Quickpay:
				$response	=	null;
				$status		=	null;
				$error		=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/subscriptions/' . $paymentBasket->subscr_id . '/cancel', array(), 105, $response, $status, 'post', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_user_key' ) );

				if ( ( ! in_array( $status, array( 200, 202 ) ) ) && ( $error || ( ! $response ) ) ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription cancellation didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				}
			}

			// Stop CBSubs scheduled auto-payments:
			$paymentBasket->unscheduleAutoRecurringPayments();
		}

		// As CBSubs controls timed payments in this case, return OK for stopping subscriptions:
		return true;
	}

	/**
	 * GATEWAY-INTERNAL SPECIFIC PRIVATE METHODS:
	 */

	/**
	 * gives gateway API URL server name from gateway URL list
	 *
	 * @return string  server-name (with 'https://' )
	 */
	private function _pspApiUrl()
	{
		return $this->gatewayUrl( 'psp+api' );
	}

	/**
	 * sign payment request $requestParams with validation code added to $requestParams array
	 *
	 * @param  array  $requestParams
	 */
	private function _signRequestParams( &$requestParams )
	{
		// Concatenate request params using this payments concatenation function with $caseInsensitiveKeys = true:
		$string						=	$this->_concatVars( $requestParams, array(), null, ' ', '', true, true, false, true );

		// add validation code doing sha256 hash of the string:
		$requestParams['checksum']	=	hash_hmac( 'sha256', $string, $this->getAccountParam( 'psp_api_key' ) );
	}

	/**
	 * Validate a reply in $requestParams using checksum value, and then also through the API if it is available
	 *
	 * @param  array   $requestParams  $requestParams are raw $_POST or $_GET input and should be sanitized and unescaped if needed
	 * @return boolean                 TRUE: valid, FALSE: invalid
	 */
	private function _pspVerifySignature( $requestParams )
	{
		return ( $_SERVER["HTTP_QUICKPAY_CHECKSUM_SHA256"] == hash_hmac( 'sha256', file_get_contents("php://input"), $this->getAccountParam( 'psp_key' ) ) );
	}

	/**
	 * perform anti fraud checks on ipn values
	 *
	 * @param cbpaidPaymentNotification $ipn
	 * @param cbpaidPaymentBasket       $paymentBasket
	 * @return bool|string
	 */
	private function _pspVerifyPayment( $ipn, $paymentBasket )
	{
		global $_CB_database;

		$matching						=	true;

		if ( in_array( $ipn->getString( 'payment_status' ), array( 'Completed', 'Processed', 'Canceled_Reversal' ) ) ) {
			if ( in_array( $ipn->getString( 'txn_type' ), array( 'subscr_payment', 'subscr_signup' ) ) ) {
				$payments				=	$paymentBasket->getPaymentsTotals( $ipn->getString( 'txn_id' ) );

				if ( ( $paymentBasket->getFloat( 'mc_amount1', 0 ) != 0 ) && ( $payments->count == 0 ) ) {
					$amount				=	$paymentBasket->getFloat( 'mc_amount1', 0 );
				} else {
					$amount				=	$paymentBasket->getFloat( 'mc_amount3', 0 );
				}

				if ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) != sprintf( '%.2f', $amount ) ) {
					if ( ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) < sprintf( '%.2f', $amount ) ) || ( sprintf( '%.2f', ( $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ) ) != sprintf( '%.2f', $amount ) ) ) {
						if ( ( ! ( ( $paymentBasket->getFloat( 'mc_amount1', 0 ) != 0 ) && ( $payments->count == 0 ) ) ) && ( ( (float) sprintf( '%.2f', ( $ipn->getFloat( 'mc_gross', 0 ) - abs( $ipn->getFloat( 'tax', 0 ) ) ) ) ) < ( (float) sprintf( '%.2f', $amount ) ) ) ) {
							$matching	=	CBTxt::T( 'amount mismatch on recurring_payment: amount: [amount] != IPN mc_gross: [gross] or IPN mc_gross - IPN tax: [net] where IPN tax = [tax]', null, array( '[amount]' => $amount, '[net]' => ( $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ), '[gross]' => $ipn->getFloat( 'mc_gross', 0 ), '[tax]' => $ipn->getFloat( 'tax', 0 ) ) );
						}
					}
				}
			} else {
				if ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) != sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) ) {
					if ( ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) < sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) ) || ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ) != sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) ) ) {
						$matching		=	CBTxt::T( 'amount mismatch on webaccept: BASKET mc_gross: [basket_gross] != IPN mc_gross: [gross] or IPN mc_gross - IPN tax: [net] where IPN tax = [tax]', null, array( '[basket_gross]' => $paymentBasket->getFloat( 'mc_gross', 0 ), '[net]' => ( $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ), '[gross]' => $ipn->getFloat( 'mc_gross', 0 ), '[tax]' => $ipn->getFloat( 'tax', 0 ) ) );
					}
				}
			}
		}

		if ( in_array( $ipn->getString( 'txn_type' ), array( 'subscr_payment', 'subscr_signup', 'subscr_cancel', 'subscr_eot', 'subscr_failed' ) ) ) {
			if ( ! $paymentBasket->isAnyAutoRecurring() ) {
				$matching				=	CBTxt::T( 'Quickpay subscription IPN type [txn_type] for a basket without auto-recurring items', null, array( '[txn_type]' => $ipn->translatedTransactionType() ) );
			}
		}

		if ( ! in_array( $ipn->getString( 'txn_type' ), array( 'subscr_signup', 'subscr_cancel', 'subscr_eot', 'subscr_failed' ) ) ) {
			if ( ( $ipn->getString( 'txn_id' ) === '' ) || ( $ipn->getString( 'txn_id' ) === 0 ) || ( $ipn->getString( 'txn_id' ) === null ) ) {
				$matching				=	CBTxt::T( 'illegal transaction id' );
			} else {
				$countBaskets			=	$paymentBasket->countRows( "txn_id = '" . $_CB_database->getEscaped( $ipn->getString( 'txn_id' ) ) . "' AND payment_status = 'Completed'" );

				if ( ( ( $countBaskets == 1 ) && ( $paymentBasket->getString( 'txn_id' ) != $ipn->getString( 'txn_id' ) ) ) || ( $countBaskets > 1 ) ) {
					$matching			=	CBTxt::T( 'transaction already used for [count] other already completed payment(s)', null, array( '[count]' => $countBaskets ) );
				}
			}
		}

		return $matching;
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's reply in $postdata:
	 *
	 * @param  array   $postdata  raw POST data received from the payment gateway
	 * @param  string  $reason    OUT: reason_code
     * @return string             CBSubs status
	 */
	private function _paymentStatus( $postdata, &$reason )
	{
		$operations					=	cbGetParam( $postdata, 'operations', array() );

		if ( ! $operations ) {
			$reason					=	null;

			return 'Error';
		}

		$latest						=	( count( $operations ) - 1 );
		$status						=	cbGetParam( $operations[$latest], 'qp_status_code' );
		$acceptPaymentCondition		=	$this->getAccountParam( 'accept_payment_condition' );

		switch ( $status ) {
			case '30100':
				$reason				=	'3D Secure is required';
				$status				=	'Denied';
				break;
			case '40000':
				$reason				=	'Rejected By Acquirer';
				$status				=	'Denied';
				break;
			case '40001':
				$reason				=	'Request Data Error';
				$status				=	'Error';
				break;
			case '40002':
				$reason				=	'Authorization expired';
				$status				=	'Denied';
				break;
			case '40003':
				$reason				=	'Aborted';
				$status				=	'Denied';
				break;
			case '50000':
				$reason				=	'Gateway Error';
				$status				=	'Error';
				break;
			case '50300':
				$reason				=	'Communications Error (with Acquirer)';
				$status				=	'Error';
				break;
			case '20000':
			default: // API responses may not have a status code, but still have type and pending state
				switch ( cbGetParam( $operations[$latest], 'type' ) ) {
					case 'authorize':
					case 'recurring':
						if ( cbGetParam( $operations[$latest], 'pending' ) === true ) {
							if ( $acceptPaymentCondition == 'pending' ) {
								$reason		=	null;
								$status		=	'Completed';
							} else {
								$reason		=	'Transaction being processed';
								$status		=	'Pending';
							}
						} elseif ( cbGetParam( $operations[$latest], 'qp_status_msg' ) == 'Approved' ) {
							$reason			=	null;
							$status			=	'Completed';
						} else {
							$reason			=	'Transaction declined';
							$status			=	'Denied';
						}
						break;
					case 'capture':
						$reason				=	null;
						$status				=	'Completed';
						break;
					case 'refund':
						$reason				=	null;
						$status				=	'Refunded';
						break;
					default: // Unknown
						$reason				=	null;
						$status				=	null;
						break;
				}
				break;
		}

		return $status;
	}

	/**
	 * Compute the CBSubs payment_type based on gateway's reply $postdata:
	 *
	 * @param  array   $postdata raw POST data received from the payment gateway
	 * @return string  Human-readable string
	 */
	private function _getPaymentType( $postdata )
	{
		$metaData		=	cbGetParam( $postdata, 'metadata', array() );

		if ( ! $metaData ) {
			return null;
		}

		$type			=	strtolower( cbGetParam( $metaData, 'brand' ) );

		switch ( strtolower( $type ) ) {
			case 'american-express':
			case 'american-express-dk':
				$type	=	'American Express Credit Card';
				break;
			case 'dankort':
				$type	=	'Dankort Credit Card';
				break;
			case 'danske-dk':
				$type	=	'Danske Net Bank';
				break;
			case 'diners':
			case 'diners-dk':
				$type	=	'Diners Credit Card';
				break;
			case 'edankort':
				$type	=	'eDankort Credit Card';
				break;
			case 'fbg1886':
				$type	=	'Forbrugsforeningen af 1886';
				break;
			case 'jcb':
			case '3d-jcb':
				$type	=	'JCB Credit Card';
				break;
			case '3d-maestro':
			case '3d-maestro-dk':
				$type	=	'Maestro Credit Card';
				break;
			case 'mastercard':
			case 'mastercard-dk':
			case '3d-mastercard':
			case '3d-mastercard-dk':
				$type	=	'Mastercard Credit Card';
				break;
			case 'mastercard-debet-dk':
			case '3d-mastercard-debet-dk':
				$type	=	'Mastercard Debit Card';
				break;
			case 'nordea-dk':
				$type	=	'Nordea Net Bank';
				break;
			case 'visa':
			case 'visa-dk':
			case '3d-visa':
			case '3d-visa-dk':
				$type	=	'Visa Credit Card';
				break;
			case 'visa-electron':
			case 'visa-electron-dk':
			case '3d-visa-electron':
			case '3d-visa-electron-dk':
				$type	=	'Visa Debit Card';
				break;
			case 'paypal':
				$type	=	'PayPal';
				break;
			case 'creditcard':
			case '3d-creditcard':
				$type	=	'Credit Card';
                break;
			default:
				break;
		}

		return $type;
	}

	/**
	 * Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	private function _getBasicRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams										=	array();
		$requestParams['version']							=	'v10';
		$requestParams['merchant_id']						=	$this->getAccountParam( 'psp_id' );
		$requestParams['agreement_id']						=	$this->getAccountParam( 'psp_api_id' );
		$requestParams['order_id']							=	$this->_getOrderID( $paymentBasket );

		// Lets try to do a quick check to see if the order id has already been used as quickpay doesn't allow reuse of initialized order ids
		$exists												=	false;

		try {
			$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/payments', array(	'order_id'		=>	$requestParams['order_id'],
																					'sort_by'		=>	'created_at',
																					'sort_dir'		=>	'desc'
																				), 30, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_key' ) );

			$exists											=	false;

			if ( in_array( $status, array( 200, 202 ) ) && $response ) {
				$responseData								=	json_decode( $response, true );

				if ( $responseData && isset( $responseData[0] ) ) {
					$exists									=	true;
				}
			}
		} catch( Exception $e ) {}

		if ( $exists ) {
			// Order id does exist so lets just use our existing cancellation incrementing behavior to increment the order id
			$this->_setCancelCount( $paymentBasket );

			$requestParams['order_id']						=	$this->_getOrderID( $paymentBasket );
		}

		$requestParams['amount']							=	( sprintf( '%.2f', $paymentBasket->mc_gross ) * 100 );
		$requestParams['currency']							=	$paymentBasket->mc_currency;
		$requestParams['type']								=	'payment';
		$requestParams['language']							=	$this->getAccountParam( 'language' );
		$requestParams['autocapture']						=	1;

		// urls for return, cancel, and IPNs:
		$requestParams['continueurl']						=	$this->getSuccessUrl( $paymentBasket );
		$requestParams['cancelurl']							=	$this->getCancelUrl( $paymentBasket );
		$requestParams['callbackurl']						=	$this->getNotifyUrl( $paymentBasket );

		// optional parameters:
		$requestParams['description']						=	$paymentBasket->item_name;
		$requestParams['variables[user_id]']				=	$paymentBasket->user_id;
		$requestParams['variables[basket_id]']				=	$paymentBasket->id;

		// recommended anti-fraud fields:
		$requestParams['invoice_address_selection']			=	1;
		$requestParams['invoice_address[name]']				=	$paymentBasket->address_name;

		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 50 ) ) {
			$requestParams['invoice_address[email]']		=	$paymentBasket->payer_email;
			$requestParams['customer_email']				=	$paymentBasket->payer_email;
		}

		if ( $this->getAccountParam( 'givehiddenddress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields									=	array(	'invoice_address[street]'			=>	array( $paymentBasket->address_street, 30 ),
																		'invoice_address[zip_code]'			=>	array( $paymentBasket->address_zip, 10 ),
																		'invoice_address[city]'				=>	array( $paymentBasket->address_city, 30 ),
																		'invoice_address[country_code]'		=>	array( $paymentBasket->getInvoiceCountry( 3 ), 3 ),
																		'invoice_address[region]'			=>	array( $paymentBasket->getInvoiceState(), 30 )
																	);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField									=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]						=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddentelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['invoice_address[phone_number]']	=	$paymentBasket->contact_phone;
		}

		$paymentMethods										=	$this->getAccountParam( 'psp_payment_methods' );

		if ( $this->getAccountParam( 'psp_3d_secure' ) && ( ! $paymentMethods ) ) {
			$paymentMethods									=	'3d-creditcard';
		}

		if ( $paymentMethods ) {
			$requestParams['payment_methods']				=	$paymentMethods;
		}

		return $requestParams;
	}


	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata         Data returned by gateway
	 * @param  string               $type                Type of return ('R' for PDT, 'I' for INS, 'A' for Autorecurring payment (Vault) )
     * @param  array                $additionalLogData   Additional strings to log with IPN
     * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		if ( $type === 'R' ) {
			// Delay PDT to give priority to IPNs
			sleep( 3 );
		} elseif ( $type === 'I' ) {
			if ( cbGetParam( $requestdata, 'state' ) === 'pending' ) {
				// Delay pending to give priority to processed as QuickPay will send both at the exact same time
				sleep( 2 );
			}
		}

		$ret													=	null;
		$paymentBasketId										=	cbGetParam( $requestdata, 'order_id', null );

		if ( ! is_int( $paymentBasketId ) ) {
			// Looks like we've the order id instead of basket id so lets convert it to basket id:
			$paymentBasketId									=	$this->_getBasketID( $paymentBasketId );
		}

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( $paymentBasketId );
			$sharedSecret										=	cbGetParam( $requestdata, $this->_getPagingParamName( 'id' ) );

			if ( ! $sharedSecret ) {
				$sharedSecret									=	$this->_getReqParam( 'id' );
			}

			if ( $exists && ( ( $type == 'A' ) || ( ( $sharedSecret == $paymentBasket->shared_secret ) && ( ! ( ( ( $type == 'R' ) || ( $type == 'I' ) ) && ( $paymentBasket->payment_status == 'Completed' ) ) ) ) ) ) {
				// PDT doesn't return transacton information; lets request for it:
				$error											=	null;

				if ( $type == 'R' ) {
					$requestParams 								= 	array(	'order_id'		=>	$this->_getOrderID( $paymentBasket ),
																			'sort_by'		=>	'created_at',
																			'sort_dir'		=>	'desc'
																		);

					$requestdata								=	array();
					$response									=	null;
					$status										=	null;

					if ( $paymentBasket->period3 ) {
						// https://learn.quickpay.net/tech-talk/api/services/#subscriptions
						$error									=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/subscriptions', $requestParams, 105, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_user_key' ) );

						if ( in_array( $status, array( 200, 202 ) ) && $response ) {
							$responseData						=	json_decode( $response, true );

							if ( $responseData && isset( $responseData[0] ) ) {
								$requestdata					=	$responseData[0];
							}
						}
						// If this faills lets roll into checking if a payment for the matching order id exists
					}

					if ( ! $requestdata ) {
						// https://learn.quickpay.net/tech-talk/api/services/#payments
						$error									=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/payments', $requestParams, 105, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_key' ) );

						if ( in_array( $status, array( 200, 202 ) ) && $response ) {
							$responseData						=	json_decode( $response, true );

							if ( $responseData && isset( $responseData[0] ) ) {
								$requestdata					=	$responseData[0];
							}
						}
					}

					// Recheck the basket as the IPN may have paid it while the above HTTP requests were going on
					if ( $paymentBasket->load( $paymentBasketId ) && ( $paymentBasket->payment_status === 'Completed' ) ) {
						return null;
					}
				}

				// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason );

				if ( $error ) {
					$paymentStatus								=	'Error';
				}

				$paymentType									=	$this->_getPaymentType( $requestdata );
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E', 'A' => 'W' );

					if ( isset( $errorTypes[$type] ) ) {
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	( (int) cbGetParam( $requestdata, 'test_mode' ) ? 1 : 0 );
				$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : ( $type == 'A' ? 'AUTORECURRING VAULT PAYMENT' : 'UNKNOWN' ) ) ) . '";' . "\n";

				if ( $additionalLogData ) {
					foreach ( $additionalLogData as $k => $v ) {
						$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
					}
				}

				$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	( $error ? $error : $reason );

					if ( $type == 'R' ) {
						$this->_setCancelCount( $paymentBasket );
					}

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					$ret										=	false;
				} else {
					$ipn->bindBasket( $paymentBasket );

					$operation									=	cbGetParam( $requestdata, 'operations', array() );

					if ( $operation ) {
						$operation								=	$operation[( count( $operation ) - 1 )];
					}

					if ( cbGetParam( $operation, 'type' ) === 'cancel' ) {
						// The last operation was a cancellation so just re-render the basket if we're a PDT
						if ( $type == 'R' ) {
							$paymentBasket->payment_status		=	'RedisplayOriginalBasket';
						}

						return false;
					}

					$insToIpn									=	array(	'address_name'		=>	'name',
																			'address_street'	=>	'street',
																			'address_zip'		=>	'zip_code',
																			'address_city'		=>	'city',
																			'address_country'	=>	'country_code',
																			'address_state'		=>	'region',
																			'contact_phone'		=>	'phone_number',
																			'payer_email'		=>	'email'
																		);

					$invoiceAddress								=	cbGetParam( $requestdata, 'invoice_address', array() );

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $invoiceAddress, $v );
					}

					$ipn->txn_id								=	cbGetParam( $requestdata, 'id' );
					$ipn->mc_gross								=	sprintf( '%.2f', ( cbGetParam( $operation, 'amount' ) / 100 ) );
					$ipn->mc_currency							=	cbGetParam( $requestdata, 'currency' );
					$ipn->user_id								=	(int) $paymentBasket->user_id;
					$ipn->sale_id								=	cbGetParam( $requestdata, 'order_id', null );

					// check what type of purchase this is:
					$recurring									=	( ( strtolower( cbGetParam( $requestdata, 'type' ) ) == 'subscription' ) || ( $type == 'A' ) ? true : false );

					// handle recurring subscriptions properly or default to single payment:
					if ( $recurring ) {
						if ( ( $paymentStatus == 'Completed' ) && ( ! $paymentBasket->subscr_id ) ) {
							$ipn->txn_type						=	'subscr_signup';
							$ipn->subscr_id						=	cbGetParam( $requestdata, 'subscription_id' );

							if ( ! $ipn->subscr_id ) {
								$ipn->subscr_id					=	cbGetParam( $requestdata, 'id' );
							}

							$ipn->subscr_date					=	$ipn->payment_date;

							// Check if an initial payment already exists for the subscription:
							$requestParams 						= 	array(	'order_id'		=>	$this->_getOrderID( $paymentBasket, 0 ),
																			'sort_by'		=>	'created_at',
																			'sort_dir'		=>	'desc'
																		);

							$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/payments', $requestParams, 105, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_key' ) );

							$results							=	array();

							if ( in_array( $status, array( 200, 202 ) ) && $response ) {
								$responseData					=	json_decode( $response, true );

								if ( $responseData && isset( $responseData[0] ) ) {
									$results					=	$responseData[0];
								}
							}

							// Trigger initial payment for the subscription since QuickPay does not pay initial payment for a subscription:
							if ( ! $results ) {
								$requestParams 					= 	array(	'order_id'		=>	$this->_getOrderID( $paymentBasket, 0 ),
																			'auto_capture'	=>	true
																		);

								if ( $paymentBasket->period1 ) {
									$requestParams['amount']	=	( sprintf( '%.2f', $paymentBasket->mc_amount1 ) * 100 );
								} else {
									$requestParams['amount']	=	( sprintf( '%.2f', $paymentBasket->mc_amount3 ) * 100 );
								}

								$error							=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/subscriptions/' . $ipn->subscr_id . '/recurring', $requestParams, 105, $response, $status, 'post', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_user_key' ) );

								if ( $response ) {
									$results					=	json_decode( $response, true );
								}

								if ( $status == 202 ) {
									// We're using instant capture and QuickPay can be a bit slow here so lets pause for a second then retry grabbing the payment:
									sleep( 1 );

									$error						=	$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/payments/' . cbGetParam( $results, 'id', null ), array(), 105, $response, $status, 'get', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_key' ) );

									if ( in_array( $status, array( 200, 202 ) ) && $response ) {
										$responseData			=	json_decode( $response, true );

										if ( $responseData ) {
											$results			=	$responseData;
										}
									}
								}

								if ( ( ! in_array( $status, array( 200, 202 ) ) ) && ( $error || ( ! $response ) ) ) {
									$paymentBasket->reason_code	=	$error;

									if ( $type == 'R' ) {
										$this->_setCancelCount( $paymentBasket );
									}

									$ipn->payment_status		=	'Error';
									$ipn->txn_type				=	'subscr_failed';

									$rawData					=	'$response="' . preg_replace( '/([^\s]{100})/', '$1 ', $response ) . "\"\n"
																.	'$results=' . var_export( $results, true ) . ";\n"
																.	'$requestParams=' . var_export( $requestParams, true ) . ";\n";

									$ipn->setRawData( $rawData );

									$this->_storeIpnResult( $ipn, 'FAILED' );
									$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $error, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

									// Cancel the subscription at QuickPay since it was created, but can't be paid:
									$response					=	null;
									$status						=	null;

									$this->httpsRequestGuzzle( $this->_pspApiUrl() . '/subscriptions/' . $ipn->subscr_id . '/cancel', array(), 105, $response, $status, 'post', 'normal', '*/*/v10', true, 443, '', $this->getAccountParam( 'psp_api_user_key' ) );

									return false;
								} else {
									$ipn->txn_id				=	cbGetParam( $results, 'id' );
								}
							} else {
								$ipn->txn_id					=	cbGetParam( $results, 'id' );
							}
						} elseif ( $paymentStatus == 'Denied' ) {
							if ( ( $paymentBasket->reattempts_tried + 1 ) <= cbpaidScheduler::getInstance( $this )->retries ) {
								$ipn->txn_type					=	'subscr_failed';
							} else {
								$ipn->txn_type					=	'subscr_cancel';
							}
						} elseif ( in_array( $paymentStatus, array( 'Completed', 'Processed', 'Pending' ) ) ) {
							$ipn->txn_type						=	'subscr_payment';
						}
					} else {
						$ipn->txn_type							=	'web_accept';
					}

					// PDT and recurring payments are automatically validated as we have to use API to request its information to begin with so only verify IPNs:
					if ( in_array( $type, array( 'R', 'A' ) ) || $this->_pspVerifySignature( $requestdata ) ) {
						// Verify the payment for PDT and IPNs; skip for scheduled payments as we don't need to verify what we've directly triggered
						$valid									=	( $type === 'A' ? true : $this->_pspVerifyPayment( $ipn, $paymentBasket ) );

						if ( $valid === true ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// add the gateway to the basket:
								$paymentBasket->payment_method	=	$this->getPayName();
								$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

								// 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date:
								$autorecurring_type				=	( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed' ) ) ? 2 : 0 );

								// 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration:
								$autorenew_type					=	( $autorecurring_type ? ( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 ) : 0 );

								if ( $recurring ) {
									$paymentBasket->reattempt	=	1; // we want to reattempt auto-recurring payment in case of failure
								}

								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, false );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									$ret						=	true;

									if ( $recurring && ( $type != 'A' ) ) {
										$paymentBasket->scheduleAutoRecurringPayments();
									}
								}
							} else {
								$this->_storeIpnResult( $ipn, 'FAILED' );

								$paymentBasket->payment_status		=	$ipn->payment_status;

								if ( $type == 'R' ) {
									$this->_setCancelCount( $paymentBasket );

									$this->_setErrorMSG( $this->getTxtNextStep( $paymentBasket ) );

									$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								}

								$ret								=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': payment missmatch. Error: ' . $valid, CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					} else {
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': checksum does not match with gateway.', CBTxt::T( 'The Secret Key signature is incorrect.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

						$ret									=	false;
					}
				}
			}
		} else {
			$returndata											=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': order_id is missing: ' . $returndata, CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return  $ret;
	}

	/**
	 * Increments the number of times the basket has been cancelled so we can safely suffix the basket
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 */
	private function _setCancelCount( $paymentBasket )
	{
		$previousCancels	=	$paymentBasket->getParam( 'quickpay.cancel', 0, 'integrations' );

		$paymentBasket->setParam( 'quickpay.cancel', ( $previousCancels + 1 ), 'integrations' );

		$paymentBasket->storeParams( 'integrations' );

		$paymentBasket->store();
	}

	/**
	 * Gets the properly formatted order id from basket
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param int                 $recurring
	 * @return string
	 */
	private function _getOrderID( $paymentBasket, $recurring = null )
	{
		$cancels	=	(int) $paymentBasket->getParam( 'quickpay.cancel', 0, 'integrations' );

		return preg_replace( '/[^-a-zA-Z0-9]/', '', $this->getAccountParam( 'psp_prefix' ) . $paymentBasket->id . ( $cancels ? '-C' . $cancels : null ) . ( $recurring !== null ? ( $recurring ? '-R' . $recurring : '-I' ) : null ) );
	}

	/**
	 * Gets the basket id from order id
	 *
	 * @param int $orderId
	 * @return int
	 */
	private function _getBasketID( $orderId )
	{
		return (int) preg_replace( '/(-C\d+)?(-R\d+)?(-I)?$/', '', str_replace( $this->getAccountParam( 'psp_prefix' ), '', $orderId ) );
	}

	/**
	 * PRIVATE BACKEND FUNCTION USED BY CLASS cbpaidGatewayAccountccbilloem below
	 */

	/**
	 * USED by XML interface ONLY !!! Renders URL to set in the QuickPay interface for notifications:
	 * Called by cbpaidGatewayAccountquickpayoem::renderNotifyUrl() just below in next class
	 *
	 * @param  string  $gatewayId
	 * @return string  HTML to display
	 */
	public function renderNotifyUrl( /** @noinspection PhpUnusedParameterInspection */ $gatewayId )
	{
		//FIXME : in such cases, the notification URL should not contain the gateway ID and be independant of the gateway ID but only of the gateway TYPE: That way multiple gateways for same PSP account can be setup (e.g. for conditions on promos)
		return $this->getNotifyUrl( null );
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountquickpayoem extends cbpaidGatewayAccounthostedpage		//  extends cbpaidGatewayAccount
{

	/**
	 * USED by XML interface ONLY !!! Renders URL for notifications
	 *
	 * @param  string           $gatewayId  Id of gateway
	 * @param  ParamsInterface  $params     Params of gateway
	 * @return string                       HTML to display
	 */
	public function renderNotifyUrl( $gatewayId, /** @noinspection PhpUnusedParameterInspection */ $params )
	{
		$payClass	=	$this->getPayMean();

		return str_replace( 'http://', 'https://', $payClass->renderNotifyUrl( $gatewayId ) );
	}

	/**
	 * USED by XML interface ONLY !!! Renders URL for site returns
	 *
     * @param  string           $gatewayId  Id of gateway
     * @param  ParamsInterface  $params     Params of gateway
     * @return string                       HTML to display
     */
	public function renderSiteUrl( $gatewayId, $params )
	{
		global $_CB_framework;

		return $_CB_framework->getCfg( 'live_site' );
	}

	/**
	* Gets payment mean handler : Overridde to phpdocument return of correct class
	*
	* @param  string             $methodCheck
	* @return cbpaidquickpayoem
	*/
	public function getPayMean( $methodCheck = null )
	{
		return parent::getPayMean( $methodCheck );
	}
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidquickpay extends cbpaidquickpayoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountquickpay extends cbpaidGatewayAccountquickpayoem
{
}
