<?php
/**
* CBSubs (TM): Community Builder Paid Subscriptions Plugin: Skrill
* @version 1.0
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @author Tobias Satzger
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Registry\ParamsInterface;
use CBLib\Xml\SimpleXMLElement;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php');

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * This is the OEM version
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidskrilloem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param  cbpaidGatewayAccount        $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		$this->_gatewayUrls	= array(
			'psp+normal'	=> $this->getAccountParam( 'psp_normal_url', '' )
			);
	}

    /**
     * Refunds a payment
     *
     * @param  cbpaidPaymentBasket       $paymentBasket  paymentBasket object
     * @param  cbpaidPayment             $payment        payment object
     * @param  cbpaidPaymentItem[]|null  $paymentItems   Array of payment items to refund completely (if no $amount)
     * @param  boolean                   $lastRefund     Last refund for $payment ?
     * @param  float                     $amount         Amount in currency of the payment
     * @param  string                    $reasonText     Refund reason comment text for gateway
     * @param  string                    $returnText     RETURN param : Text of success message or of error message
     * @return boolean                                   true if refund done successfully, false if error
     */
    public function refundPayment( $paymentBasket, $payment, $paymentItems, $lastRefund, $amount, $reasonText, &$returnText )
	{
        // Get the status of this transaction
        $refundVars                     =   array();
        $refundVars['email']            =   $this->getAccountParam( 'pspemail' );
        $refundVars['password']         =   strtolower( md5( $this->getAccountParam( 'psppassword', '' ) ) );
        $refundVars['action']           =   'prepare';
        $refundVars['transaction_id']   =   $paymentBasket->id;
        $refundResult                   =	null;
        $refundStatus                   =	null;
        $refundError                    =   $this->_httpsRequest( $this->getRefundUrl(), $refundVars, 60, $refundResult, $refundStatus );

        if( ( $refundError != 0 ) || ( $refundStatus != 200) ) {
            $this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The refund request failed: ' . $refundResult, CBTxt::T( "The refund request does not work. This payment gateway might not be setup correctly. The merchant might not have enabled this service in 'Merchant tools' or might not have provided the correct API/MQI password." ) );
            return false;
        }

        return $this->handleRefundResult($refundResult);
    }

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams						=	$this->_getBasicRequstParams( $paymentBasket );

		return $requestParams;
	}
	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	*/
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		return $this->getSinglePaymentRequstParams( $paymentBasket );
	}

	/**
	* The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	*/
	protected function handleReturn( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['transaction_id'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata						=	$postdata;
		} else {
			// but if customer needs GET, we will work with it too (removing CMS/CB/CBSubs specific routing params):
			$requestdata						=	$this->_getGetParams();
		}

		// check if ordernumber exists and if not add basket id from PDT success url:
		if ( ! isset( $requestdata['transaction_id'] ) ) {
			$requestdata['transaction_id']		=	cbGetParam( $_GET, 'cbpbasket', null );
		}

		$pspPassword							=	$this->getAccountParam( 'psppassword' );
		if( ! empty( $pspPassword ) ) {

			// Get the status of this transaction
			$queryVars								=	array();
			$queryVars['email']						=	$this->getAccountParam( 'pspemail' );
			$queryVars['password']					=	strtolower( md5( $pspPassword ) );
			$queryVars['action']					=	'status_trn';
			$queryVars['trn_id']					=	$requestdata['transaction_id'];
			$queryResult							=	null;
			$queryStatus							=	null;
			$queryError								=	$this->_httpsRequest( $this->getQueryUrl(), $queryVars, 60, $queryResult, $queryStatus );
			$matches								=	array();
			preg_match( '/(.+)\s*(.+)/', $queryResult, $matches );
			$statusResult							=	$matches[1];

			// Check for error
			if ( ( $queryError != 0 ) || ( $queryStatus != 200 ) || ( substr( $statusResult, 0, 3 ) != 200 ) ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The query request failed: ' . $queryResult, CBTxt::T( 'The merchant query interface does not work. This payment gateway might not be setup correctly. The merchant might not have enabled this service in "Merchant tools" or might not have provided the correct API/MQI password.' ) );
				return false;
			}

			// Get content of result
			$contentResultString					=	$matches[2];
			$contentResultArray						=	array();
			parse_str( $contentResultString, $contentResultArray );

			// Check MD5 sum of PDT
			if ( array_key_exists( 'msid', $requestdata ) ) {
				$msid								=   strtoupper( $requestdata['msid'] );
				if ( ! empty( $msid ) && array_key_exists( 'merchant_id', $contentResultArray ) ) {
					$merchant_id					=   $contentResultArray['merchant_id'];
					if ( empty( $merchant_id ) ) {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The query response does not contain the merchant_id.', CBTxt::T( "Please contact site administrator to check error log." ) );
						return false;
					}

					$calculatedChecksum				=	strtoupper( md5(
						$merchant_id .
							$contentResultArray['transaction_id'] .
							strtoupper( md5( $this->getAccountParam( 'pspsecret', '' ) ) )
					) );

					if ( $calculatedChecksum != $msid ) {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The checksum of the PDT response does not match the calculated one.' . $queryResult, CBTxt::T( "Please contact site administrator to check error log." ) );
						return false;
					}
				}
			}

			$requestdata = $contentResultArray;
		}

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	* The user cancelled his payment
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId						=	(int) $this->_getReqParam( 'basket' );
			$shared_secret							=	$this->_getReqParam( 'id' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId					=	(int) cbGetParam( $postdata, 'transaction_id', null );
			}

			$exists									=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $shared_secret == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status		=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}

		return false;
	}

	/**
	* The payment service provider server did a server-to-server notification: verify and handle it here:
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	* @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	* @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	*/
	protected function handleNotification( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['transaction_id'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata					=	$postdata;
		} else {
			// but if customer needs GET, we will work with it too:
			$requestdata					=	$this->_getGetParams();
		}

		// Get the transaction_id from the PDT url
		$requestdata['transaction_id']		=	cbGetParam( $_GET, 'cbpbasket', null );

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' );
	}

	/**
	 * GATEWAY-INTERNAL SPECIFIC PRIVATE METHODS:
	 */

	/**
	 * Verifies the signature in $requestParams['md5sig']
	 *
	 * @param  array    $requestParams  Checks $requestParams['responseDigest'] towards determining values
	 * @return boolean                  True: valid, False: not valid
	 */
	private function _pspVerifySignature( $requestParams )
	{
		// Calculate signature
		$secret					=	$this->getAccountParam( 'pspsecret', '' );
		$secretMD5				=	strtoupper(md5($secret));
		$signatureString		=	cbGetParam( $requestParams, 'merchant_id' )
										. cbGetParam( $requestParams, 'transaction_id' )
										. $secretMD5
										. cbGetParam( $requestParams, 'mb_amount' )
										. cbGetParam( $requestParams, 'mb_currency' )
										. cbGetParam( $requestParams, 'status' );
		$calculatedSig			=	strtoupper(md5($signatureString));

		// Response signature
		$responseSig			=	strtoupper($requestParams['md5sig']);

		// Does it match?
		return ( $calculatedSig == $responseSig );
	}

    /**
     * Returns the URL used to send queries query the Skrill.
     *
     * @return string
     */
	private function getQueryUrl( )
	{
		return 'https://www.moneybookers.com/app/query.pl';
	}

    /**
     * Returns the URL used to send refunds to Skrill.
     *
     * @return string
     */
	private function getRefundUrl( )
	{
		return 'https://www.moneybookers.com/app/refund.pl';
	}

    /**
     * Handle the XML reply of the refund request.
     *
     * @param  string               $xmlreply           XML reply of the refund request.
     * @return boolean                                  TRUE if the refund request was successful; otherwise FALSE.
     */
	private function handleRefundResult( $xmlreply )
	{
		$xml								=	@new SimpleXMLElement( $xmlreply, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) );
		$sidElem							=	$xml->getElementByPath( '/response/sid' );
		if( $sidElem != false ) {
			$sid							=	$sidElem->data();
			if( is_string( $sid ) ) {
				// Success: The reply contains a $sid, so the refund was successful.
				return true;
			} else {
				// Invalid sid
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The XML reply of the refund request does not contain a valid sid: ' . $xmlreply, CBTxt::T("Sorry, an unexpected reply has been received from the payment processor.") . ' ' . CBTxt::T("Please contact site administrator to check error log.") );
				return false;
			}
		}
		$errorElem							=	$xml->getElementByPath( '/response/error/error_msg' );
		if( $errorElem != false ) {
			$errorMessage					=	$sidElem->data();
			if( ! empty( $errorMessage ) ) {
				// Problem with error message
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The XML reply of the refund request indicates the following error: ' . $errorMessage, CBTxt::T("Sorry, an unexpected reply has been received from the payment processor.") . ' ' . CBTxt::T("Please contact site administrator to check error log.") );
				return false;
			}
		}
		// Unknown problem
		$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The refund request was not successfull and the XML reply is invalid: ' . $xmlreply, CBTxt::T("Sorry, an unexpected reply has been received from the payment processor.") . ' ' . CBTxt::T("Please contact site administrator to check error log.") );
		return false;
	}

    /**
	 * Computes payment status
	 *
	 * @param  array                $postdata             raw POST data received from the payment gateway
	 * @return null|string
     */
	private function _paymentStatus( $postdata )
	{
		$status						=	(int) cbGetParam( $postdata, 'status', -10 );

		switch ( $status ) {
			case 2:
				return 'Completed';
			case 0:
				return 'Pending';
			case -1:
				return 'Denied';
			case -2:
				return 'Error';
			case -3:
				return 'Refunded';
		}

		return null;
	}

	/**
	* Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	* @return array                                $requestParams
	*/
	private function _getBasicRequstParams( $paymentBasket )
	{
		$requestParams								=	array();

		// Basic params
		$requestParams['transaction_id']			=	$paymentBasket->id;
		$requestParams['pay_to_email']				=	$this->getAccountParam( 'pspemail' );
		$requestParams['language']					=	$this->getAccountParam( 'language', 'EN' );
		$requestParams['return_url']				=	$this->getSuccessUrl( $paymentBasket );
		$requestParams['cancel_url']				=	$this->getCancelUrl( $paymentBasket );
		$requestParams['status_url']				=	$this->getNotifyUrl( $paymentBasket );

		// Customer params
		$requestParams['firstname']					=	$paymentBasket->first_name;
		$requestParams['lastname']					=	$paymentBasket->last_name;
		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 100 ) ) {
			$requestParams['pay_from_email']		=	$paymentBasket->payer_email;
		}
		if ( $this->getAccountParam( 'givehiddenaddress' ) ) {
			$requestParams['address']				=	$paymentBasket->address_street;
			$requestParams['city']					=	$paymentBasket->address_city;
			$requestParams['postal_code']			=	$paymentBasket->address_zip;
			$requestParams['state']					=	$paymentBasket->getInvoiceState();
			$requestParams['country']				=	$paymentBasket->getInvoiceCountry( 3 );
		}
		if ( $this->getAccountParam( 'givehiddentelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 20 ) ) {
			$requestParams['phone_number']			=	$paymentBasket->contact_phone;
		}

		// Item params
		$requestParams['amount']					=	sprintf( '%.2f', $paymentBasket->mc_gross );
		$requestParams['currency']					=	$paymentBasket->mc_currency;
		if ( $paymentBasket->tax && ( $paymentBasket->tax > 0 ) ) {
			// Treat roundings particularly well, so it matches gross amount always:
			$mc_gross_2f							=	sprintf( '%.2f', $paymentBasket->mc_gross );
			$requestParams['amount2_description']	=	CBTxt::T( "Net value" );
			$requestParams['amount2']				=	sprintf( '%.2f', $mc_gross_2f - $paymentBasket->tax );
			$requestParams['amount3_description']	=	CBTxt::T( "Taxes" );
			$requestParams['amount3']				=	sprintf( '%.2f', $paymentBasket->tax );
		}
		$requestParams['detail1_description']		=	CBTxt::T( "Subscription" );
		$requestParams['detail1_text']				=	$paymentBasket->item_name;

		// Optional params
		$company									=	$this->getAccountParam( 'company', '' );
		if( ! empty( $company ) ) {
			$requestParams['recipient_description']	=	substr($company, 0, 30); // Max length 30 character
		}
		$note										=	$this->getAccountParam( 'note', '' );
		if( ! empty( $note ) ) {
			$requestParams['confirmation_note']		=	substr($company, 0, 240); // Max length 240 character;
		}
		$logo										=	$this->getAccountParam( 'logo', '' );
		if( ! empty( $logo ) && ( strlen( $logo ) <= 240 ) ) {
			$requestParams['logo_url']				=	$logo; // Max length 240 character;
		}

		return $requestParams;
	}

    /**
     * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
     *
     * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
     * @param  array                $requestdata         Data returned by gateway
     * @param  string               $type                Type of return ('I' for INS, 'R' for PDT, 'P' for DirectLink (first payment) 'A' for Autorecurring payment (DirectLink) )
     * @param  null                 $additionalLogData   Additional Texts to log into IPN raw data
     * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
     */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		/* Check basket ID */
		$paymentBasketId								=	(int) cbGetParam( $requestdata, 'transaction_id' );
		if ( ! $paymentBasketId ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The response does not contain a transaction_id.', CBTxt::T( "Please contact site administrator to check error log." ) );
			return false;
		}

		/* Check basket */
		if ( ! $paymentBasket->load( (int) $paymentBasketId ) ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The response does not contain a valid transaction_id.', CBTxt::T( "Please contact site administrator to check error log." ) );
			return false;
		}

		/* Only proceed if this function is called by a PDT or IPN callback */
		if ( ! in_array($type, array('I', 'R') ) ) {
			return null;
		}

		/* Don't proceed with the PDT if the request data doesn't contain more then the 'transaction_id' */
		if ( ( $type == 'R' ) && ! ( sizeof( $requestdata ) > 1 ) ) {
			return null;
		}

		/* Check state */
		if ( ( $paymentBasket->payment_status == 'Completed' ) ) {
			return null;
		}

		/* Check payment recipient */
		if ( ( $this->getAccountParam( 'pspemail' ) ) != ( cbGetParam( $requestdata, 'pay_to_email', null ) ) ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ": The value for pay_to_email does not match with the merchant's one.", CBTxt::T( "Please contact site administrator to check error log." ) );
			return false;
		}

		/* Log the return record */
		$log_type										=	$type;
		$reason											=	cbGetParam( $requestdata, 'failed_reason_code', null );
		$paymentStatus									=	$this->_paymentStatus( $requestdata );
		$paymentType									=	cbGetParam( $requestdata, 'payment_type', null );
		$paymentTime									=	$_CB_framework->now();
		if ( $paymentStatus == 'Error' ) {
			$errorTypes									=	array( 'I' => 'D', 'R' => 'E' );

			if ( isset( $errorTypes[$type] ) ) {
				// Converts error status to CBSubs's error statuses:
				$log_type								=	$errorTypes[$type];
			}
		}
		$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

		/* Check signature */
		if ( isset( $requestdata['md5sig'] ) && ! $this->_pspVerifySignature( $requestdata ) ) {
			$this->_storeIpnResult( $ipn, 'SIGNERROR' );
			$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ": The MD5 signature of the response does not match the calculated one. Please make sure that you enter the the correct 'secret word' in the plugin's settings.", CBTxt::T( "The MD5 signature is incorrect." ) . ' ' . CBTxt::T( "Please contact site administrator to check error log." ) );
			return false;
		}

		/* Process payment */
		if ( $paymentStatus == 'Refunded' ) {
			// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
			$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
		}
		$ipn->test_ipn									=	( ( $this->getAccountParam( 'normal_gateway' ) == '0' ) || (int) cbGetParam( $requestdata, 'testmode' ) ? 1 : 0 );
		$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : ( $type == 'P' ? 'DATALINK PAYMENT' : ( $type == 'A' ? 'AUTORECURRING DIRECTLINK PAYMENT' : 'UNKNOWN' ) ) ) ) . '";' . "\n";
		if ( $additionalLogData ) {
			foreach ( $additionalLogData as $k => $v ) {
				$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
			}
		}
		$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
														.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
														.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

		if ( $paymentStatus == 'Error' ) {
			$paymentBasket->reason_code					=	$reason;

			$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
			$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( "Sorry, the payment server replied with an error." ) . ' ' . CBTxt::T( "Please contact site administrator to check payment status and error log." ) );

			return false;
		} else {
			// Get important parameters from basket into the $ipn:
			$ipn->bindBasket( $paymentBasket );

			$ipn->mc_gross								=	sprintf( '%.2f', cbGetParam( $requestdata, 'amount' ) );
			$ipn->mc_currency							=	cbGetParam( $requestdata, 'currency' );
			$ipn->user_id								=	(int) $paymentBasket->user_id;
			$ipn->txn_type								=	'web_accept';
			$ipn->payment_status						=	$paymentStatus;
			$autorecurring_type							=	0;
			$autorenew_type								=	0;

			// 1) Basket id matches transaction_id
			// 2) Amounts match
			// 3) Currency matches:
			if	( ( $paymentBasketId == cbGetParam( $requestdata, 'transaction_id' ) )
				&& ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross )
				&& ( $paymentBasket->mc_currency == $ipn->mc_currency ) )
			{
				// Ok, we passed all basic consistancy tests:
				if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
					// Status is also that of a successful (or on it's way to successful) transaction:
					// Stores the $ipn notification as a successful one:
					$this->_storeIpnResult( $ipn, 'SUCCESS' );
					// Binds to basket the relevant $ipn attributes:
					$this->_bindIpnToBasket( $ipn, $paymentBasket );

					// In case of refund we need to log the payment as it has same TnxId as first payment:
					$txnIdMultiplePaymentDates			=	( $paymentStatus == 'Refunded' );

					// Update payment basket, generates all basket payment events, generates a transaction record (payment):
					$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, $txnIdMultiplePaymentDates );

					if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
						// ok, we had success in terms of a payment initiation or completion:
						return true;
					}
				} else {
					// The payment didn't initiate or complete: We failed:
					// Store the notification $ipn as a failed one:
					$this->_storeIpnResult( $ipn, 'FAILED' );

					// Updates payment basket status with the notification status:
					$paymentBasket->payment_status		=	$ipn->payment_status;

					// Sets error to display to the user:
					$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

					// We will be re-displaying the basket to the user so he can retry payment:
					$paymentBasket->payment_status		=	'RedisplayOriginalBasket';
					return false;
				}
			} else {
				// We found a major mismatch between orderID, amount or currency:
				// Store it in notification:
				$this->_storeIpnResult( $ipn, 'MISMATCH' );
				// Log it as error and prepare display of error to user:
				$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': transaction_id, amount or currency missmatch.', CBTxt::T( "Sorry, the payment does not match the basket." ) . ' ' . CBTxt::T( "Please contact site administrator to check error log." ) );

				return false;
			}
		}

		return  null;
	}	// end function _returnParamsHandler
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * This is the OEM variant.
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountskrilloem extends cbpaidGatewayAccounthostedpage
{
	/**
	 * USED by XML interface ONLY !!!
	 * Renders URL for successful returns
	 * We are overriding this method, as CCBill accepts non-https URLs:
	 *
	 * @param  string              $value   Variable value ( 'successurl', 'cancelurl', 'notifyurl' )
	 * @param  ParamsInterface     $params
	 * @param  string              $name    The name of the form element
	 * @param  CBSimpleXMLElement  $node    The xml element for the parameter
	 * @return string                       HTML to display
	 */
	public function renderUrl( $value, $params, $name, $node )
	{
		return $this->getPayMean()->adminUrlRender( $node->attributes( 'value' ) );
	}
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * This is the gateway-specific version.
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidskrill extends cbpaidskrilloem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * This is the gateway-specific version.
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountskrill extends cbpaidGatewayAccountskrilloem
{
}
