<?php
/**
* @version $Id: cbpaidsubscriptions.stripe.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CB\Database\Table\UserTable;
use CBLib\Input\Get;
use CBLib\Language\CBTxt;
use CBLib\Registry\Registry;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

class cbpaidstripeoem extends cbpaidCreditCardsPayHandler
{
	/**
	 * Overrides base class with 1:
	 * Hash type: 1 = only if there is a basket id (default), 2 = always, 0 = never
	 * @var int
	 */
	protected $_urlHashType				=	1;

	/**
	 * Gateway API version used
	 * @var string
	 */
	public $gatewayApiVersion			=	'1.3.0';

	/**
	 * Stripe API version used
	 * @var string
	 */
	private string $stripeApiVersion	=	'2022-11-15';

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		$this->_gatewayUrls				=	[	'psp+normal'	=>	$this->getAccountParam( 'psp_normal_url', '' ) . '/v1',
												'psp+test'		=>	$this->getAccountParam( 'psp_test_url', '' ) . '/v1',
											];

		if ( $this->getAccountParam( 'card_button_method', '' ) === 'server' ) {
			// For B/C force the old server usage to elements:
			$account->set( 'card_button_method', 'elements' );
		} elseif ( $this->getAccountParam( 'card_button_method', '' ) === 'button' ) {
			// For B/C force the old button usage to checkout:
			$account->set( 'card_button_method', 'checkout' );
		}

		$availableMethods				=	[];

		if ( $this->getAccountParam( 'card_button_method', '' ) === 'checkout' ) {
			if ( (int) $this->getAccountParam( 'enabled', 0 ) === 1 ) {
				$availableMethods[]		=	'stripe_single_checkout';
			} elseif ( (int) $this->getAccountParam( 'enabled', 0 ) === 2 ) {
				$availableMethods[]		=	'stripe_recurring_checkout';
			} elseif ( (int) $this->getAccountParam( 'enabled', 0 ) === 3 ) {
				$availableMethods[]		=	'stripe_single_checkout';
				$availableMethods[]		=	'stripe_recurring_checkout';
			}
		} elseif ( (int) $this->getAccountParam( 'enabled', 0 ) === 1 ) {
			$availableMethods[]			=	'stripe_single';

			if ( (int) $this->getAccountParam( 'card_element_checkout', 0 ) ) {
				$availableMethods[]		=	'stripe_single_checkout';
			}
		} elseif ( (int) $this->getAccountParam( 'enabled', 0 ) === 2 ) {
			$availableMethods[]			=	'stripe_recurring';

			if ( (int) $this->getAccountParam( 'card_element_checkout', 0 ) ) {
				$availableMethods[]		=	'stripe_recurring_checkout';
			}
		} elseif ( (int) $this->getAccountParam( 'enabled', 0 ) === 3 ) {
			$availableMethods[]			=	'stripe_single';
			$availableMethods[]			=	'stripe_recurring';

			if ( (int) $this->getAccountParam( 'card_element_checkout', 0 ) ) {
				$availableMethods[]		=	'stripe_single_checkout';
				$availableMethods[]		=	'stripe_recurring_checkout';
			}
		}

		$account->set( 'cardtypes', $availableMethods );
	}

	/**
	 * CBSUBS PAY HANLDER API METHODS:
	 */

	/**
	 * Draws the credit-card form
	 *
	 * @param  UserTable            $user                 User
	 * @param  cbpaidPaymentBasket  $paymentBasket        paymentBasket object
	 * @param  string               $cardType             CC-brand if no choice
	 * @param  string               $postUrl              URL for the <form>
	 * @param  string               $payButtonText        Text for payment text (if basket allows single-payments)
	 * @param  string               $subscribeButtonText  Text for subscribe button (if basket allows auto-recurring subscriptions)
	 * @param  string|null          $chosenCard
	 * @return string
	 */
	protected function _drawCCform( $user, $paymentBasket, $cardType, $postUrl, $payButtonText, $subscribeButtonText, $chosenCard = null ): string
	{
		global $_CB_framework;

		if ( in_array( $chosenCard, [ 'stripe_single', 'stripe_recurring', 'stripe_single_checkout', 'stripe_recurring_checkout' ], true ) ) {
			// Redisplay shows standard CC form, which doesn't work with custom payment methods so lets override it and enforce our method:
			$cardType			=	$chosenCard;
		}

		if ( ! $cardType ) {
			// If there's an error detecting card type we won't have a card type to check against so lets try to fallback to one based on what we know:
			$cardTypes			=	$this->getAccountParam( 'cardtypes', [] );

			if ( $subscribeButtonText ) {
				if ( \in_array( 'stripe_recurring', $cardTypes, true ) ) {
					$cardType	=	'stripe_recurring';
				} elseif ( \in_array( 'stripe_recurring_checkout', $cardTypes, true ) ) {
					$cardType	=	'stripe_recurring_checkout';
				}
			} elseif ( \in_array( 'stripe_single', $cardTypes, true ) ) {
				$cardType		=	'stripe_single';
			} elseif ( \in_array( 'stripe_single_checkout', $cardTypes, true ) ) {
				$cardType		=	'stripe_single_checkout';
			}
		}

		if ( in_array( $cardType, [ 'stripe_single_checkout', 'stripe_recurring_checkout' ], true ) ) {
			// https://stripe.com/docs/checkout
			$session			=	$this->getStripeCheckoutSessionURL( $paymentBasket, ( $cardType === 'stripe_recurring_checkout' ) );

			if ( ! $session ) {
				cbRedirect( $paymentBasket->getShowBasketUrl( false ), CBTxt::T( "Submitted checkout request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ), 'error' );
			}

			$_CB_framework->redirect( $session ); // Direct redirect as we don't want the redirect intercept behavior done during debug mode
		}

		$recurring				=	( $cardType === 'stripe_recurring' );

		if ( $recurring && $this->hasStripeTestClocks() ) {
			// Render the test form that will directly post to the payment processing and bypass payment confirmation
			ob_start();
			$this->renderTestClockForm( $paymentBasket, $postUrl, $subscribeButtonText );
			return ob_get_clean();
		}

		return $this->renderElements( $paymentBasket, $postUrl, $payButtonText, $subscribeButtonText, $recurring );
	}

	/**
	 * Returns the JS necessary to process an elements payment
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $postUrl
	 * @param null|string         $payButtonText
	 * @param null|string         $subscribeButtonText
	 * @param bool                $recurring
	 * @return string
	 */
	protected function renderElements( cbpaidPaymentBasket $paymentBasket, string $postUrl, ?string $payButtonText, ?string $subscribeButtonText, bool $recurring ): string
	{
		global $_CB_framework;

		// https://stripe.com/docs/js/element
		// https://stripe.com/docs/elements
		// https://stripe.com/docs/payments/quickstart#init-stripe-html
		$intent				=	$this->getStripePaymentIntentSecret( $paymentBasket, $recurring );

		if ( ! $intent ) {
			cbRedirect( $paymentBasket->getShowBasketUrl( false ), CBTxt::T( "Submitted payment request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ), 'error' );
		}

		$returnURL			=	$this->getStripeReturnURL( $paymentBasket, $recurring );

		// https://stripe.com/docs/stripe.js
		$_CB_framework->document->addHeadScriptUrl( 'https://js.stripe.com/v3/' );

		$js					=	"const stripe = Stripe( " . json_encode( $this->getAccountParam( 'stripe_pk', '' ), JSON_HEX_TAG ) . " );"
							.	"const elements = stripe.elements({"
							.		"clientSecret: " . json_encode( $intent, JSON_HEX_TAG  ) . ","
							.		"loader: 'always'"
							.	"});"
							.	"const payment = elements.create( 'payment', {"
							.		"layout: " . json_encode( $this->getAccountParam( 'card_element_layout', 'accordion' ), JSON_HEX_TAG ) . ","
							.		"defaultValues: {"
							.			"billingDetails: {"
							.				"name: " . json_encode( $paymentBasket->getString( 'address_name', '' ), JSON_HEX_TAG ) . ","
							.				"email: " . json_encode( $paymentBasket->getString( 'payer_email', '' ), JSON_HEX_TAG ) . ","
							.				"phone: " . json_encode( $paymentBasket->getString( 'contact_phone', '' ), JSON_HEX_TAG ) . ","
							.				"address: {"
							.					"line1: " . json_encode( $paymentBasket->getString( 'address_street', '' ), JSON_HEX_TAG ) . ","
							.					"city: " . json_encode( $paymentBasket->getString( 'address_city', '' ), JSON_HEX_TAG ) . ","
							.					"state: " . json_encode( ( $paymentBasket->getInvoiceState() ?: '' ), JSON_HEX_TAG ) . ","
							.					"country: " . json_encode( ( $paymentBasket->getInvoiceCountry( 2 ) ?: '' ), JSON_HEX_TAG ) . ","
							.					"postal_code: " . json_encode( $paymentBasket->getString( 'address_zip', '' ), JSON_HEX_TAG ) . ","
							.				"}"
							.			"}"
							.		"}"
							.	"});"
							.	"payment.on( 'ready', function() {"
							.		"$( '.cbsubsStripeButton' ).removeClass( 'hidden' );"
							.	"});"
							.	"payment.mount( '#stripe-element' );"
							.	"const form = $( '.cbsubsStripeForm' );"
							.	"let formReady = true;"
							.	"payment.on( 'change', function( e ) {"
							.		"formReady = e.complete;"
							.	"});"
							.	"form.on( 'submit', function( e ) {"
							.		"if ( e.isTrigger ) {"
							.			"return;"
							.		"}"
							.		"e.preventDefault();"
							.		"if ( ! formReady ) {"
							.			"return;"
							.		"}"
							.		"$( '.cbsubsStripePayment' ).css( 'opacity', 0.5 );"
							.		"$( '.cbsubsStripeLoading' ).removeClass( 'hidden' );"
							.		"$( '.cbsubsStripeError' ).addClass( 'hidden' );"
							.		"stripe." . ( $recurring ? 'confirmSetup' : 'confirmPayment' ) . "({"
							.			"elements,"
							.			"confirmParams: {"
							.				"return_url: " . json_encode( $returnURL, JSON_HEX_TAG ) . ","
							.			"},"
							.		"}).then( function( result ) {"
							.			"if ( result.error ) {"
							.				"$( '.cbsubsStripePayment' ).css( 'opacity', 1 );"
							.				"$( '.cbsubsStripeLoading' ).addClass( 'hidden' );"
							.				"if ( result.error.type != 'validation_error' ) {"
							.					"$( '.cbsubsStripeError' ).removeClass( 'hidden' );"
							.					"$( '.cbsubsStripeError' ).html( result.error.message );"
							.				"}"
							.			"} else {"
							.				"if ( typeof result.setupIntent != 'undefined' ) {"
							.					"$( '#stripe-token' ).val( result.setupIntent.id );"
							.					"const setupIntentAction = result.setupIntent.next_action;"
							.					"if ( setupIntentAction && ( setupIntentAction.type === 'redirect_to_url' ) ) {"
							.						"window.location = setupIntentAction.redirect_to_url.url;"
							.					"}"
							.				"} else {"
							.					"$( '#stripe-token' ).val( result.paymentIntent.id );"
							.					"const paymentIntentAction = result.paymentIntent.next_action;"
							.					"if ( paymentIntentAction && ( paymentIntentAction.type === 'redirect_to_url' ) ) {"
							.						"window.location = paymentIntentAction.redirect_to_url.url;"
							.					"}"
							.				"}"
							.				"form.submit();"
							.			"}"
							.		"});"
							.	"});";

		$_CB_framework->outputCbJQuery( $js );

		ob_start();
		$this->renderElementForm( $paymentBasket, $postUrl, $payButtonText, $subscribeButtonText, $recurring );
		return ob_get_clean();
	}

	/**
	 * Renders the Stripe element form
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $postUrl
	 * @param null|string         $payButtonText
	 * @param null|string         $subscribeButtonText
	 * @param bool                $recurring
	 */
	protected function renderElementForm( cbpaidPaymentBasket $paymentBasket, string $postUrl, ?string $payButtonText, ?string $subscribeButtonText, bool $recurring ): void
	{
		$legend		=	CBTxt::Th( 'PAYMENT_CONTAINER_TITLE', 'Payment' );
		?>
		<form action="<?php echo $postUrl; ?>" method="post" autocomplete="off" name="cbsubsStripeForm" class="form-auto m-0 mt-3 mb-3 cb_form cbValidation cbsubsStripeForm">
			<fieldset class="position-relative d-block w-100 border border-info m-0 p-0 cbFieldset cbregPaymentForm">
				<?php if ( $legend ) { ?>
				<legend class="border-0 w-auto m-0 ml-2 mr-2 pl-1 pr-1 pb-1 cbFieldsetLegend">
					<?php echo $legend; ?>
				</legend>
				<?php } ?>
				<div class="cbregCCnumexp cbsubsStripePayment">
					<div class="p-2">
						<div class="mb-2 alert alert-danger cbsubsStripeError hidden"></div>
						<div id="stripe-element">
							<div class="text-center text-muted cbsubsStripeLoading">
								<div class="spinner-border"></div>
							</div>
						</div>
						<div class="mt-2 text-center cbregCCbutton cbsubsStripeButton hidden">
							<button type="submit" class="btn btn-primary btn-block text-wrap">
								<?php echo ( $recurring ? $subscribeButtonText : $payButtonText ); ?>
							</button>
						</div>
					</div>
				</div>
				<div class="p-2 position-absolute d-flex flex-column justify-content-center align-items-center w-100 h-100 text-muted cbsubsStripeLoading hidden" style="top: 0; left: 0; z-index: 999;">
					<div class="spinner-border"></div>
				</div>
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'basket' ); ?>" value="<?php echo $paymentBasket->getInt( 'id', 0 ); ?>">
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'shopuser' ); ?>" value="<?php echo $this->shopuserParam( $paymentBasket ); ?>">
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'paymenttype' ); ?>" value="<?php echo ( $recurring ? 2 : 1 ); ?>">
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'cardtype' ); ?>" value="<?php echo ( $recurring ? 'stripe_recurring' : 'stripe_single' ); ?>">
				<input type="hidden" id="stripe-token" name="cbptoken" value="">
			</fieldset>
		</form>
		<?php
	}

	/**
	 * Renders the Stripe test clock form
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $postUrl
	 * @param null|string         $subscribeButtonText
	 */
	protected function renderTestClockForm( cbpaidPaymentBasket $paymentBasket, string $postUrl, ?string $subscribeButtonText ): void
	{
		?>
		<form action="<?php echo $postUrl; ?>" method="post" autocomplete="off" name="cbsubsStripeForm" class="form-auto m-0 mt-3 mb-3 cb_form cbValidation cbsubsStripeForm">
			<fieldset class="position-relative d-block w-100 border border-info m-0 p-0 cbFieldset cbregPaymentForm">
				<legend class="border-0 w-auto m-0 ml-2 mr-2 pl-1 pr-1 pb-1 cbFieldsetLegend">
					<?php echo CBTxt::Th( 'Test Payment' ); ?>
				</legend>
				<div class="cbregCCnumexp cbsubsStripePayment">
					<div class="p-2">
						<div>
							<?php echo CBTxt::Th( 'This is for testing purposes only. A test clock will be created and attached to a test customer. A subscription will then be created for the test customer. You can advance time on the subscription within your Stripe dashboard to test subscription handling on your site.' ); ?>
						</div>
						<div class="mt-2 text-center cbregCCbutton cbsubsStripeButton">
							<button type="submit" class="btn btn-primary btn-block text-wrap">
								<?php echo $subscribeButtonText; ?>
							</button>
						</div>
					</div>
				</div>
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'basket' ); ?>" value="<?php echo $paymentBasket->getInt( 'id', 0 ); ?>">
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'shopuser' ); ?>" value="<?php echo $this->shopuserParam( $paymentBasket ); ?>">
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'paymenttype' ); ?>" value="2">
				<input type="hidden" name="<?php echo $this->_getPagingParamName( 'cardtype' ); ?>" value="stripe_recurring">
			</fieldset>
		</form>
		<?php
	}

	/**
	 * This is the main method to initiate a payment, depending on $redirectNow, it will either:
	 * 'redirect' : Redirect directly to the payment page of the payment gateway
	 * 'radios'   : Return all elements needed to display a list of selection radios, each with list of cards accepted, label of the radio, and a description that will be displayed when the radio is selected.
	 * 'buttons'  : Returns all elements needed to display a list of buttons with hidden form elements
	 * Note: this method can be called 2 times in case the radio is selected, to also get what's needed to display the payment buttons.
	 *
	 * $redirectNow Expected return array:
	 * ------------ ----------------------
	 * 'redirect' : return array( 'url_to_which_to_redirect' )
	 * 'radios'   : return array( array( account_id, submethod, paymentMethod:'single'|'subscribe', array(cardtypes), 'label for radio', 'description for radio' ), ... )
	 * 'buttons'  : return array( array( post_url, requestParams, customImage, altText, titleText, payNameForCssClass, butId ), ... )
	 *
	 * @param  UserTable            $user           object reflecting the user being registered (it can have id 0 or be NULL in future)
	 * @param  cbpaidPaymentBasket  $paymentBasket  Order Basket to be paid
	 * @param  string               $redirectNow    'redirect', 'radios', 'buttons', other: return null (see above)
	 * @return string|array                         array: See above, OR string: HTML to display in buttons area
	 */
	public function getPaymentBasketProcess( $user, $paymentBasket, $redirectNow )
	{
		$availableMethods				=	$this->getAccountParam( 'cardtypes', [] );
		$currency						=	strtoupper( $paymentBasket->getString( 'mc_currency', '' ) );

		foreach ( $availableMethods as $k => $v ) {
			if ( ( ( $v === 'stripe_single' ) || ( $v === 'stripe_single_checkout' ) ) && ( $paymentBasket->isAnyAutoRecurring() === 1 ) ) {
				// Recurring payments for this basket are required so we can't accept single payments
				unset( $availableMethods[$k] );
				continue;
			}

			$recurring					=	( ( $v === 'stripe_recurring' ) || ( $v === 'stripe_recurring_checkout' ) );

			if ( $recurring && ( ! $paymentBasket->isAnyAutoRecurring() ) ) {
				// Recurring payments for this basket are not allowed so we can't accept recurring payments
				unset( $availableMethods[$k] );
				continue;
			}

			$checkout					=	( ( $v === 'stripe_single_checkout' ) || ( $v === 'stripe_recurring_checkout' ) );
			$paymentMethods				=	$this->getAccountParam( 'card_' . ( $checkout ? 'checkout' : 'element' ) . ( $recurring ? '_recurring' : '' ) . '_filter_methods', '' );

			if ( $paymentMethods ) {
				if ( ! \is_array( $paymentMethods ) ) {
					$paymentMethods		=	explode( '|*|', $paymentMethods );
				}

				if ( ! $this->validatePaymentMethods( $paymentMethods, $currency, $recurring, $checkout ) ) {
					// Payment method filtering is being applied and there are no available payment methods for the payment type being used so remove it
					unset( $availableMethods[$k] );
				}
			}
		}

		$this->account->set( 'cardtypes', $availableMethods );

		return parent::getPaymentBasketProcess( $user, $paymentBasket, $redirectNow );
	}

	/**
	 * Returns gateway specific basket details for display
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @return string|null
	 */
	public function getPaymentBasketDetails( $paymentBasket ): string
	{
		if ( ( ! $this->getAccountParam( 'customer_portal', 0 ) )
			 || ( Application::MyUser()->getUserId() !== $paymentBasket->getInt( 'user_id', 0 ) )
			 || ( ! $paymentBasket->isAnyAutoRecurring() )
		) {
			return '';
		}

		return '<a href="' . $this->cbsubsGatewayUrl( 'change', null, $paymentBasket ) . '" class="btn btn-sm btn-light border">' . CBTxt::Th( 'Change Payment Method' ) . '</a>';
	}

	/**
	 * Handles the gateway-specific result of payments (redirects back to this site and gateway notifications). WARNING: unchecked access !
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket         New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata              _POST data for saving edited tab content as generated with getEditTab
	 * @param  bool                 $allowHumanHtmlOutput  Input+Output: set to FALSE if it's an IPN, and if it is already false, keep quiet
	 * @return null|string|bool                            HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	public function resultNotification( $paymentBasket, $postdata, &$allowHumanHtmlOutput )
	{
		global $_CB_framework;

		$now								=	$_CB_framework->now();
		$enableProcessor					=	$this->getAccountParam( 'enabled', 0 );
		$authorizedCardTypes				=	$this->getAccountParam( 'cardtypes', [] );
		$result								=	Application::Input()->getString( 'result', '' );

		switch ( $result ) {
			case 'paynow':
				$return						=	'';

				if ( $this->_checkIfHttpS( 'params' ) ) {
					$paymentBasketId		=	(int) $this->_getReqParam( 'basket' );
					$shopUser				=	$this->_getReqParam( 'shopuser' );

					$card					=	[];
					$card['type']			=	$this->_getReqParam( 'cardtype', '' );
					$card['number']			=	'';
					$card['cvv']			=	'';
					$card['name']			=	'';
					$card['firstname']		=	'';
					$card['lastname']		=	'';
					$card['expmonth']		=	'';
					$card['expyear']		=	'';
					$card['paymentType']	=	(int) $this->_getReqParam( 'paymenttype' );

					if ( $this->getAccountParam( 'show_cc_avs', 0 ) ) {
						$card['address']	=	'';
						$card['zip']		=	'';
						$card['country']	=	'';
					}

					if ( $this->getAccountParam( 'show_cc_avs', 0 ) >= 3 ) {
						$card['city']	 	=	'';
					}

					if ( $this->getAccountParam( 'show_cc_avs', 0 ) >= 4 ) {
						$card['state']	 	=	'';
					}

					$exists					=	$paymentBasket->load( $paymentBasketId );
					$user					=	CBuser::getUserDataInstance( $paymentBasket->getInt( 'user_id', 0 ) );

					if ( $exists && ( $paymentBasket->getString( 'payment_status', '' ) !== 'Completed' ) && ( $shopUser === $this->shopuserParam( $paymentBasket ) ) ) {
						if ( ( ! Application::MyUser()->getUserId() ) || $paymentBasket->authoriseAction( 'pay' ) ) {
							if ( ! in_array( $card['type'], $authorizedCardTypes, true ) ) {
								$this->_setErrorMSG( CBTxt::T( 'Bad credit card type: please check credit-card type again.' ) );
							} elseif ( ! in_array( $card['paymentType'], [ 1, 2 ], true ) ) {
								$this->_setErrorMSG( CBTxt::T( 'Please click on button to pay.' ) );
							} else {
								$subscribePossibilities			=	$this->_getPaySubscribePossibilities( $enableProcessor, $paymentBasket );
								$subscribeAvailability			=	( $card['paymentType'] & $subscribePossibilities );		// logical AND

								if ( $subscribeAvailability === 1 ) {
									// single payment:
									/** @var cbpaidPaymentNotification|null $ipn */
									$ipn						=	null;

									$transactionId				=	$this->_attemptSinglePayment( $card, $paymentBasket, $now, $ipn, $subscribeAvailability );

									if ( $transactionId !== false ) {
										$this->updatePaymentStatus( $paymentBasket, 'web_accept', ( $ipn ? $ipn->getString( 'payment_status', 'Completed' ) : 'Completed' ), $ipn, 1, 0, 0, false );
									}
								} elseif ( $subscribeAvailability === 2 ) {
									// recuring payment:
									/** @var cbpaidPaymentNotification|null $ipn */
									$ipn						=	null;

									$occurrences				=	0;
									$autorecurring_type			=	0;
									$autorenew_type				=	0;

									$subscriptionId				=	$this->processSubscriptionPayment( $card, $paymentBasket, $now, $ipn, $occurrences, $autorecurring_type, $autorenew_type );

									if ( is_string( $subscriptionId ) ) {
										$this->updatePaymentStatus( $paymentBasket, 'subscr_signup', ( $ipn ? $ipn->getString( 'payment_status', 'Completed' ) : 'Completed' ), $ipn, $occurrences, $autorecurring_type, $autorenew_type, false );
									}
								} else {
									$this->_setErrorMSG( CBTxt::T( 'Submitted payment without pressing pay or subscribe button.' ) );
								}
							}
						} else {
							$exists			=	false;

							$this->_setLogErrorMSG( 3, null, $this->getPayName(), CBTxt::T( 'Unauthorized payment action.' ) );
						}
					} else {
						if ( ! ( $exists && in_array( $paymentBasket->getString( 'payment_status', '' ), [ 'Completed', 'Pending' ], true ) ) ) {
							$this->_setErrorMSG( CBTxt::T( 'Payment basket does not exist.' ) );
						}

						$exists				=	false;
					}

					$errorMsg				=	$this->getErrorMSG( '<br />' );

					if ( ( $errorMsg ) && $exists && ( $shopUser === $this->shopuserParam( $paymentBasket ) ) && ( ! in_array( $paymentBasket->getString( 'payment_status', '' ), [ 'Completed', 'Pending' ], true ) ) ) {
						$return				.=	$this->displayPayForm( $user, $paymentBasket, $card['type'], $errorMsg, $enableProcessor, $authorizedCardTypes );
					}
				} else {
					$this->_setLogErrorMSG( 3, null, $this->getPayName(), CBTxt::T( 'Unauthorized access without https.' ) );
				}

				return $return;
			case 'cancel':
				// We cancelled at stripe so lets clear checkout sessions and intents, but we need to do so without changing the new payment basket object for redisplay so load in a new object:
				$previousBasket			=	new cbpaidPaymentBasket();

				if ( $previousBasket->load( (int) $this->_getReqParam( 'basket' ) ) ) {
					$cleared			=	false;

					foreach ( [ 'checkout_session.single', 'payment_intent.single', 'setup_intent.single', 'checkout_session.recurring', 'payment_intent.recurring', 'setup_intent.recurring' ] as $param ) {
						$paramValue		=	$previousBasket->getParam( $param, '', 'integrations', GetterInterface::STRING );

						if ( ! $paramValue ) {
							continue;
						}

						switch ( $paramValue ) {
							case 'checkout_session.single':
							case 'checkout_session.recurring':
								$this->cancelStripeCheckoutSession( $paramValue );
								break;
							case 'payment_intent.single':
							case 'payment_intent.recurring':
								$this->cancelStripePaymentIntent( $paramValue );
								break;
							case 'setup_intent.single':
							case 'setup_intent.recurring':
								$this->cancelStripeSetupIntent( $paramValue );
								break;
						}

						$previousBasket->setParam( $param, null, 'integrations' );

						$cleared		=	true;
					}

					if ( $cleared ) {
						$previousBasket->storeParams( 'integrations' );

						$previousBasket->store();
					}
				}
				break;
			case 'change':
				$allowHumanHtmlOutput	=	false; // This is a redirect so no need to output anything
				$paymentBasketId		=	(int) $this->_getReqParam( 'basket' );

				if ( ! $paymentBasketId ) {
					$this->_setLogErrorMSG( 5, $paymentBasket, $this->getPayName() . ' change notice: basket id not supplied', CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					return null;
				}

				$exists					=	$paymentBasket->load( $paymentBasketId );

				if ( ! $exists ) {
					$this->_setLogErrorMSG( 5, $paymentBasket, $this->getPayName() . ' change notice: basket does not exist', CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					return null;
				}

				if ( Application::MyUser()->getUserId() !== $paymentBasket->getInt( 'user_id', 0 ) ) {
					$this->_setLogErrorMSG( 5, $paymentBasket, $this->getPayName() . ' change notice: basket does not belong to user', CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					return null;
				}

				if ( ( ! $this->getAccountParam( 'customer_portal', 0 ) ) || ( ! $paymentBasket->isAnyAutoRecurring() ) ) {
					$this->_setLogErrorMSG( 5, $paymentBasket, $this->getPayName() . ' change notice: customer portal not enabled or basket not recurring', CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					return null;
				}

				$customerId				=	$this->getStripeCustomerID( $paymentBasket );

				if ( ! $customerId ) {
					$this->_setLogErrorMSG( 5, $paymentBasket, $this->getPayName() . ' change notice: missing customer id', CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					return null;
				}

				$portalResults			=	new Registry();
				$portalURL				=	$this->getStripeCustomerPortalURL( $customerId, $portalResults );

				if ( ! $portalURL ) {
					$error				=	$portalResults->getString( 'error.message', '' );

					if ( ! $error ) {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Billing Portal API HTTPS POST request to payment gateway server failed.', CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					} else {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Billing Portal API error returned. ERROR: ' . $error . ' CODE: ' . $portalResults->getString( 'error.code', '' ), CBTxt::T( 'Payment method for this subscription cannot be changed.' ) );
					}

					return null;
				}

				cbRedirect( $portalURL );
				break;
		}

		return parent::resultNotification( $paymentBasket, $postdata, $allowHumanHtmlOutput );
	}

	/**
	 * Returns a list of available payment methods, their features, and their currency limits
	 *
	 * @return array
	 */
	private function getPaymentFeatures(): array
	{
		return	[	'us_bank_account'	=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'USD' ],
											],
					'bacs_debit'		=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => true ],
												'currencies'	=> [ 'GBP' ],
											],
					'au_becs_debit'		=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'AUD' ],
											],
					'acss_debit'		=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => false ],
												'currencies'	=> [ 'CAD' ],
											],
					'sepa_debit'		=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'EUR' ],
											],
					'bancontact'		=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'EUR' ],
											],
					'blik'				=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'PLN' ],
											],
					'eps'				=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'EUR' ],
											],
					'fpx'				=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'MYR' ],
											],
					'giropay'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'EUR' ],
											],
					'ideal'				=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'EUR' ],
											],
					'p24'				=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'EUR', 'PLN' ],
											],
					'sofort'			=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'EUR' ],
											],
					'affirm'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'USD' ],
											],
					'afterpay_clearpay'	=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'USD', 'CAD', 'GBP', 'AUD', 'NZD', 'EUR' ],
											],
					'klarna'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'EUR', 'USD', 'GBP', 'DKK', 'SEK', 'NOK' ],
											],
					'pix'				=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'BRL' ],
											],
					'paynow'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'SGD' ],
											],
					'promptpay'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'THB' ],
											],
					'boleto'			=>	[	'recurring'		=> [ 'elements' => true, 'checkout' => true ],
												'currencies'	=> [ 'BRL' ],
											],
					'konbini'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'JPY' ],
											],
					'oxxo'				=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'MXN' ],
											],
					'alipay'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'CNY', 'AUD', 'CAD', 'EUR', 'GBP', 'HKD', 'JPY', 'SGD', 'MYR', 'NZD', 'USD' ],
											],
					'grabpay'			=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'SGD', 'MYR' ],
											],
					'wechat_pay'		=>	[	'recurring'		=> [ 'elements' => false, 'checkout' => false ],
												'currencies'	=> [ 'CNY', 'AUD', 'CAD', 'EUR', 'GBP', 'HKD', 'JPY', 'SGD', 'USD', 'DKK', 'NOK', 'SEK', 'CHF' ],
											],
				];
	}

	/**
	 * Validates supplied payment methods to ensure they can be used for the supplied currency and for single or recurring payments
	 *
	 * @param array  $paymentMethods
	 * @param string $currency
	 * @param bool   $recurring
	 * @param bool   $checkout
	 * @return array
	 */
	private function validatePaymentMethods( array $paymentMethods, string $currency, bool $recurring = false, bool $checkout = false ): array
	{
		// https://stripe.com/docs/payments/payment-methods/integration-options
		if ( ! $currency ) {
			return array_values( array_unique( $paymentMethods ) );
		}

		$paymentFeatures		=	$this->getPaymentFeatures();
		$currency				=	strtoupper( $currency );

		foreach ( $paymentMethods as $k => $v ) {
			if ( in_array( $v, [ 'card', 'link' ], true ) ) {
				continue;
			}

			$methodFeatures		=	( $paymentFeatures[$v] ?? [] );

			if ( ! $methodFeatures ) {
				unset( $paymentMethods[$k] );
				continue;
			}

			if ( $recurring && ( ! $methodFeatures['recurring'][( $checkout ? 'checkout' : 'elements' )] ) ) {
				unset( $paymentMethods[$k] );
				continue;
			}

			if ( ! in_array( $currency, $methodFeatures['currencies'], true ) ) {
				unset( $paymentMethods[$k] );
			}
		}

		return array_values( array_unique( $paymentMethods ) );
	}

	/**
	 * Returns a cbpaidGatewaySelectorButton object parameters for rendering an HTML form with a visible button and hidden fields for the gateway
	 * Or a string with HTML content instead (not recommended)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  string               $subMethod      'single', 'subscribe' or gateway-specific string (e.g. credit-card brand)
	 * @param  string               $paymentType    'single' or 'subscribe' or for subscriptions 'cancel'
	 * @return cbpaidGatewaySelectorButton|string                  or string with HTML
	 */
	protected function getPayButtonRecepie( $paymentBasket, $subMethod, $paymentType )
	{
		$altText				=	sprintf( CBTxt::T( "Pay with %s" ), CBTxt::T( 'Stripe' ) );
		$titleText				=	$altText;

		if ( $paymentType === 'stripe_single' ) {
			$paymentImage		=	$this->getAccountParam( 'card_element_image', 'stripe' );
			$customImgParam		=	'card_element_custom_image';
			$customCssParam		=	'card_element_custom_css';
		} elseif ( $paymentType === 'stripe_recurring' ) {
			$paymentImage		=	$this->getAccountParam( 'card_element_recurring_image', 'stripe' );
			$customImgParam		=	'card_element_recurring_custom_image';
			$customCssParam		=	'card_element_recurring_custom_css';
		} elseif ( $paymentType === 'stripe_single_checkout' ) {
			$paymentImage		=	$this->getAccountParam( 'card_checkout_image', 'stripe' );
			$customImgParam		=	'card_checkout_custom_image';
			$customCssParam		=	'card_checkout_custom_css';
		} elseif ( $paymentType === 'stripe_recurring_checkout' ) {
			$paymentImage		=	$this->getAccountParam( 'card_checkout_recurring_image', 'stripe' );
			$customImgParam		=	'card_checkout_recurring_custom_image';
			$customCssParam		=	'card_checkout_recurring_custom_css';
		} else {
			return parent::getPayButtonRecepie( $paymentBasket, $subMethod, $paymentType );
		}

		$requestParams			=	$this->fillinCCFormRequstParams( $paymentBasket, $paymentType );
		$buttonId				=	'cbpaidButt' . strtolower( $paymentType ?: $this->getPayName() );
		$customImage			=	trim( $this->getAccountParam( $customImgParam, '' ) );

		if ( $customImage === '' ) {
			$customImage		=	$this->_renderCCimg( $paymentImage, true );
		}

		$payNameForCssClass		=	( $paymentType ?: $this->getPayName() );
		$pspUrl					=	$this->_getPayFormUrl( $paymentBasket );

		return cbpaidGatewaySelectorButton::getPaymentButton( $this->getAccountParam( 'id', 0 ), $subMethod, $paymentType, $pspUrl, $requestParams, $customImage, $this->getAccountParam( $customCssParam, '' ), $altText, $titleText, $payNameForCssClass, $buttonId );
	}

	/**
	 * Refunds a payment
	 *
	 * @param  cbpaidPaymentBasket       $paymentBasket  paymentBasket object
	 * @param  cbpaidPayment             $payment        payment object
	 * @param  cbpaidPaymentItem[]|null  $paymentItems   Array of payment items to refund completely (if no $amount)
	 * @param  bool                      $lastRefund     Last refund for $payment ?
	 * @param  float                     $amount         Amount in currency of the payment
	 * @param  string                    $reasonText     Refund reason comment text for gateway
	 * @param  string                    $returnText     RETURN param : Text of success message or of error message
	 * @return bool                                      true if refund done successfully, false if error
	 */
	public function refundPayment( $paymentBasket, $payment, $paymentItems, $lastRefund, $amount, $reasonText, &$returnText ): bool
	{
		global $_CB_framework;

		// https://stripe.com/docs/api#refunds
		// https://stripe.com/docs/refunds
		if ( ! $this->hasStripeApi() ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Stripe API Publishable Key and Secret Key not set.', CBTxt::T( 'Needed Stripe API Publishable Key and Secret Key not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return false;
		}

		if ( $amount !== $payment->getFloat( 'mc_gross', 0 ) ) {
			$amount							=	sprintf( '%.02f', $amount );
			$logType						=	'4';
			$paymentStatus					=	'Partially-Refunded';
			$refundType						=	'Partial';
		} else {
			$logType						=	'3';
			$paymentStatus					=	'Refunded';
			$refundType						=	'Full';
		}

		$chargeId							=	$paymentBasket->getString( 'txn_id', '' );

		if ( ( ( strpos( $chargeId, 'ch_' ) === 0 ) && ( strpos( $chargeId, 'py_' ) === 0 ) ) ) {
			// Already partially refunded or disputed so get the charge id from parent transaction
			$chargeId						=	$paymentBasket->getString( 'parent_txn_id', '' );
		}

		$requestParams						=	[	'charge'				=>	$chargeId,
													'metadata[user_id]'		=>	$paymentBasket->getInt( 'user_id', 0 ),
													'metadata[order_id]'	=>	$paymentBasket->getInt( 'id', 0 ),
													'metadata[invoice]'		=>	$paymentBasket->getString( 'invoice', '' ),
													'metadata[payment_id]'	=>	$payment->getInt( 'id', 0 ),
													'metadata[description]'	=>	$reasonText,
												];

		if ( $refundType !== 'Full' ) {
			$requestParams['amount']		=	( sprintf( '%.02f', $amount ) * 100 );
		}

		$results							=	$this->httpsRequestStripe( 'post', '/refunds', $requestParams );

		if ( $results->has( 'error' ) ) {
			$error							=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Refunds API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted refund request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Refunds API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}

			return false;
		}

		if ( ! in_array( $results->getString( 'status', '' ), [ 'succeeded', 'pending' ], true ) ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Refunds API error returned. ERROR: ' . $results->getString( 'failure_reason', '' ), $results->getString( 'failure_reason', '' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return false;
		}

		$ipn								=	$this->_prepareIpn( $logType, $paymentStatus, 'instant', null, $_CB_framework->now(), 'utf-8' );

		$ipn->bindBasket( $paymentBasket );

		$ipn->set( 'user_id', $paymentBasket->getInt( 'user_id', 0 ) );

		$ipn->setRawResult( 'SUCCESS' );

		$rawData							=	'$results=' . var_export( $results, true ) . ";\n"
											.	'$requestParams=' . var_export( $requestParams, true ) . ";\n";

		$ipn->setRawData( $rawData );

		$ipn->set( 'mc_currency', strtoupper( $results->getString( 'currency', '' ) ) );
		$ipn->set( 'mc_gross', ( - ( $results->getFloat( 'amount', 0 ) / 100 ) ) );

		$ipn->computeRefundedTax( $payment );

		$ipn->set( 'txn_id', $results->getString( 'id', '' ) );
		$ipn->set( 'parent_txn_id', $results->getString( 'charge', '' ) );

		$this->updatePaymentStatus( $paymentBasket, $ipn->getString( 'txn_type', '' ), $ipn->getString( 'payment_status', '' ), $ipn, 1, 0, 0, false );

		$ipn->store();

		// If this is a recurring subscription and the last refund for it then we need to cancel the subscription since refund API does not cancel subscriptions:
		if ( $paymentBasket->getString( 'subscr_id', '' ) && $lastRefund ) {
			$this->stopPaymentSubscription( $paymentBasket, $paymentItems );
		}

		return true;
	}

	/**
	 * CBSUBS ON-SITE CREDIT-CARDS PAGES PAYMENT API METHODS:
	 */

	/**
	 * Returns text "using your xxxx account no...."
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return string
	 */
	public function getTxtUsingAccount( $paymentBasket ): string
	{
		switch ( strtolower( $paymentBasket->getString( 'payment_type', '' ) ) ) {
			case 'acss_debit':
				return ' ' . CBTxt::T( 'using ACSS Direct Debit (CA)' );
			case 'affirm':
				return ' ' . CBTxt::T( 'using Affirm' );
			case 'afterpay_clearpay':
				return ' ' . CBTxt::T( 'using Afterpay / Clearpay' );
			case 'afterpay':
				return ' ' . CBTxt::T( 'using Afterpay' );
			case 'clearpay':
				return ' ' . CBTxt::T( 'using Clearpay' );
			case 'alipay':
				return ' ' . CBTxt::T( 'using Alipay' );
			case 'au_becs_debit':
				return ' ' . CBTxt::T( 'using BECS Direct Debit (AU)' );
			case 'bacs_debit':
				return ' ' . CBTxt::T( 'using BACS Direct Debit (UK)' );
			case 'bancontact':
				return ' ' . CBTxt::T( 'using Bancontact' );
			case 'blik':
				return ' ' . CBTxt::T( 'using BLIK' );
			case 'boleto':
				return ' ' . CBTxt::T( 'using Boleto' );
			case 'eps':
				return ' ' . CBTxt::T( 'using EPS' );
			case 'fpx':
				return ' ' . CBTxt::T( 'using FPX' );
			case 'giropay':
				return ' ' . CBTxt::T( 'using Giropay' );
			case 'ideal':
				return ' ' . CBTxt::T( 'using iDEAL' );
			case 'klarna':
				return ' ' . CBTxt::T( 'using Klarna' );
			case 'konbini':
				return ' ' . CBTxt::T( 'using Konbini' );
			case 'link':
				return ' ' . CBTxt::T( 'using Link' );
			case 'oxxo':
				return ' ' . CBTxt::T( 'using OXXO' );
			case 'p24':
				return ' ' . CBTxt::T( 'using Przelewy24' );
			case 'paynow':
				return ' ' . CBTxt::T( 'using PayNow' );
			case 'pix':
				return ' ' . CBTxt::T( 'using Pix' );
			case 'promptpay':
				return ' ' . CBTxt::T( 'using PromptPay' );
			case 'sepa_debit':
				return ' ' . CBTxt::T( 'using SEPA Direct Debit (EU)' );
			case 'sofort':
				return ' ' . CBTxt::T( 'using SOFORT' );
			case 'us_bank_account':
				return ' ' . CBTxt::T( 'using ACH Direct Debit (US)' );
			case 'wechat_pay':
				return ' ' . CBTxt::T( 'using WeChat Pay' );
			case '':
			case 'stripe':
			case 'stripe_single':
			case 'stripe_recurring':
			case 'stripe_single_checkout':
			case 'stripe_recurring_checkout':
				return ' ' . CBTxt::T( 'using Stripe' );
			default:
				return parent::getTxtUsingAccount( $paymentBasket );
		}
	}

	/**
	 * @param null|string $level
	 * @param null|string $errorText
	 * @param null|string $errorCode
	 * @return string[]
	 */
	protected function returnErrorResponse( ?string $level = 'spurious', ?string $errorText = '', ?string $errorCode = '8888' ): array
	{
		if ( ! $errorText ) {
			$errorText	=	CBTxt::T( "Submitted payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' );
		} else {
			$errorText	.=	' ' . CBTxt::T( 'Please contact site administrator to check error log.' );
		}

		return [	'level'		=>	$level,
					'errorText'	=>	$errorText,
					'errorCode'	=>	$errorCode,
				];
	}

	/**
	 * Attempts to authorize and capture a credit card for a single payment of a payment basket
	 *
	 * @param array                           $card                contains type, number, firstname, lastname, expmonth, expyear, and optionally: address, zip, country
	 * @param cbpaidPaymentBasket             $paymentBasket
	 * @param int                             $now                 unix timestamp of now
	 * @param cbpaidsubscriptionsNotification $ipn                 returns the stored notification
	 * @param bool                            $authnetSubscription true if it is a subscription and amount is in mc_amount1 and not in mc_gross
	 * @return array|bool                     subscriptionId       if subscription request succeeded, otherwise ARRAY( 'level' => 'spurious' or 'fatal', 'errorText', 'errorCode' => string ) of error to display
	 */
	protected function processSinglePayment( $card, $paymentBasket, $now, &$ipn, $authnetSubscription )
	{
		if ( ! $this->hasStripeApi() ) {
			$this->_setLogErrorMSG( 3, null, CBTxt::T( 'Needed Stripe API Publishable Key and Secret Key not set.' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return $this->returnErrorResponse( 'fatal', CBTxt::T( 'Needed Stripe API Publishable Key and Secret Key not set.' ) );
		}

		// Lets see if we can find the source from direct or receiver methods:
		$source				=	Application::Input()->getString( 'post/cbptoken', '' );

		if ( ! $source ) {
			// Couldn't find a source so lets see if this is a redirect method payment intent:
			$source			=	Application::Input()->getString( 'get/payment_intent', '' );
		}

		if ( ! $source ) {
			// Couldn't find a source so lets see if this is a redirect method setup intent:
			$source			=	Application::Input()->getString( 'get/setup_intent', '' );
		}

		$logType			=	'';
		$requestParams		=	[];
		$results			=	new Registry();

		if ( $paymentBasket->getString( 'payment_status', '' ) === 'Pending' ) {
			// Basket is already in a pending state so lets check on its charge status:
			$return			=	$this->processSinglePaymentPending( $source, $paymentBasket, $logType, $results );

			if ( $results->get( 'charge_status', '' ) === 'pending' ) {
				// Just do nothing since we aren't changing the status
				return false;
			}
		} elseif ( ( ! $source ) && $paymentBasket->getParam( 'checkout_session.single', '', 'integrations', GetterInterface::STRING ) ) {
			// This is a checkout session basket so lets verify the checkout session charge:
			$return			=	$this->processSinglePaymentCheckout( $source, $paymentBasket, $logType, $results );
		} elseif ( ( strpos( $source, 'pi_' ) === 0 ) || ( strpos( $source, 'seti_' ) === 0 ) ) {
			// We need to check the status of this payment intent to ensure it has been completed:
			$return			=	$this->processSinglePaymentIntent( $source, $paymentBasket, $logType, $results );
		} else {
			$this->_setLogErrorMSG( 3, null, CBTxt::T( 'Invalid payment source provided. Sources and Tokens are no longer supported.' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return $this->returnErrorResponse( 'fatal', CBTxt::T( 'Invalid payment source provided.' ) );
		}

		$ipn				=	$this->_logNotification( $logType, $now, $paymentBasket, $card, $requestParams, $results, $return );

		return $return;
	}

	/**
	 * Handles processing a single payments pending basket
	 * Checks if the pending charge has completed or not
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @return string|array
	 */
	private function processSinglePaymentPending( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results )
	{
		global $_CB_framework;

		$chargeId				=	$paymentBasket->getString( 'txn_id', '' );

		// If we already have a charge id then just check its status instead of getting latest charge from the payment intent
		if ( $chargeId && ( ( strpos( $chargeId, 'ch_' ) === 0 ) || ( strpos( $chargeId, 'py_' ) === 0 ) ) ) {
			$results			=	$this->getStripeCharge( $chargeId );

			if ( $results->has( 'error' ) ) {
				$error			=	$results->getString( 'error.message', '' );

				if ( ! $error ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

					$logType	=	'B';

					return $this->returnErrorResponse();
				}

				$status			=	$results->getString( 'error.code', '' );

				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'V';

				return $this->returnErrorResponse( 'fatal', $error, $status );
			}

			$chargeId			=	$results->getString( 'id', '' );
			$chargeStatus		=	$results->getString( 'status', '' );

			if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ] ) ) {
				$error			=	$results->getString( 'failure_message', '' );
				$status			=	$results->getString( 'failure_code', '' );

				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'V';

				return $this->returnErrorResponse( 'fatal', $error, $status );
			}

			$logType			=	'P';

			$results->set( 'charge_id', $chargeId );
			$results->set( 'charge_status', $chargeStatus );

			return $chargeId;
		}

		$results				=	$this->getStripePaymentIntent( $paymentBasket->getParam( 'payment_intent.single', '', 'integrations', GetterInterface::STRING ) );

		if ( $results->has( 'error' ) ) {
			$error				=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intents API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'B';

				return $this->returnErrorResponse();
			}

			$status				=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intents API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		if ( ( $results->getString( 'status', '' ) === 'requires_action' ) && ( $results->getString( 'next_action.type', '' ) === 'redirect_to_url' ) ) {
			// Stripe JS failed to confirm the payment and there's a redirect required so goahead and do it:
			$_CB_framework->redirect( $results->getRaw( 'next_action.redirect_to_url.url', '' ) );
		}

		if ( ! in_array( $results->getString( 'status', '' ), [ 'processing', 'succeeded' ], true ) ) {
			// We do not perform manual confirmation for payment intents so if it isn't processing or succeeded then fail the payment:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intent in an invalid status. STATUS: ' . $results->getString( 'status', '' ), CBTxt::T( "Submitted payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'B';

			return $this->returnErrorResponse();
		}

		$chargeId				=	$results->getString( 'latest_charge.id', '' );
		$chargeStatus			=	$results->getString( 'latest_charge.status', '' );

		if ( ! $chargeId ) {
			$logType			=	'P';

			$results->set( 'charge_status', 'pending' );

			return $chargeId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error				=	$results->getString( 'latest_charge.failure_message', '' );
			$status				=	$results->getString( 'latest_charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intent API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType				=	'P';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $chargeId;
	}

	/**
	 * Handles processing a single payments checkout session
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @return string|array
	 */
	private function processSinglePaymentCheckout( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results )
	{
		$results			=	$this->getStripeCheckoutSession( $paymentBasket->getParam( 'checkout_session.single', '', 'integrations', GetterInterface::STRING ), [ 'payment_intent.latest_charge' ] );

		if ( $results->has( 'error' ) ) {
			$error			=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType	=	'B';

				return $this->returnErrorResponse();
			}

			$status			=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType		=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		if ( ! in_array( $results->getString( 'payment_intent.status', '' ), [ 'processing', 'succeeded' ], true ) ) {
			// We do not perform manual confirmation for payment intents so if it isn't processing or succeeded then fail the payment:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout in an invalid status. STATUS: ' . $results->getString( 'status', '' ), CBTxt::T( "Submitted payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType		=	'V';

			return $this->returnErrorResponse();
		}

		$chargeId			=	$results->getString( 'payment_intent.latest_charge.id', '' );
		$chargeStatus		=	$results->getString( 'payment_intent.latest_charge.status', '' );

		if ( ! $chargeId ) {
			$logType		=	'P';

			$results->set( 'charge_status', 'pending' );

			return $chargeId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error			=	$results->getString( 'payment_intent.latest_charge.failure_message', '' );
			$status			=	$results->getString( 'payment_intent.latest_charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType		=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType			=	'P';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $chargeId;
	}

	/**
	 * Handles processing a single payments setup intents and payment intents
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @return string|array
	 */
	private function processSinglePaymentIntent( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results )
	{
		global $_CB_framework;

		if ( strpos( $source, 'seti_' ) === 0 ) {
			$this->_setLogErrorMSG( 3, null, CBTxt::T( 'SetupIntent attempted on Single Payment. SetupIntent is only permitted for Recurring Payments.' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'B';

			return $this->returnErrorResponse();
		}

		$results				=	$this->getStripePaymentIntent( $source );

		if ( $results->has( 'error' ) ) {
			$error				=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intents API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'B';

				return $this->returnErrorResponse();
			}

			$status				=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intents API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		if ( ( $results->getString( 'status', '' ) === 'requires_action' ) && ( $results->getString( 'next_action.type', '' ) === 'redirect_to_url' ) ) {
			// Stripe JS failed to confirm the payment and there's a redirect required so goahead and do it:
			$_CB_framework->redirect( $results->getRaw( 'next_action.redirect_to_url.url', '' ) );
		}

		if ( ! in_array( $results->getString( 'status', '' ), [ 'processing', 'succeeded' ], true ) ) {
			// We do not perform manual confirmation for payment intents so if it isn't processing or succeeded then fail the payment:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intent in an invalid status. STATUS: ' . $results->getString( 'status', '' ), CBTxt::T( "Submitted payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'B';

			return $this->returnErrorResponse();
		}

		$chargeId				=	$results->getString( 'latest_charge.id', '' );
		$chargeStatus			=	$results->getString( 'latest_charge.status', '' );

		if ( ! $chargeId ) {
			$logType			=	'P';

			$results->set( 'charge_status', 'pending' );

			return $chargeId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error				=	$results->getString( 'latest_charge.failure_message', '' );
			$status				=	$results->getString( 'latest_charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intent API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType				=	'P';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $chargeId;
	}

	/**
	 * Attempts to subscribe a credit card for AIM + ARB subscription of a payment basket.
	 * ARB are subscriptions to a cron script running at payment service server each day.
	 *
	 * @param  array                           $card            : $card['type'], $card['number'], $card['name'], $card['firstname'], $card['lastname'], $card['expmonth'], $card['expyear'], and optionally: $card['address'], $card['zip'], $card['country']
	 * @param  cbpaidPaymentBasket             $paymentBasket
	 * @param  int                             $now              unix timestamp of now
	 * @param  cbpaidPaymentNotification|null  $ipn              returns the stored notification
	 * @param  int                             $occurrences      returns the number of occurences pay-subscribed firmly
	 * @param  int                             $autorecurring_type  returns:  0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date
	 * @param  int                             $autorenew_type      returns:  0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration
	 * @return mixed   STRING subscriptionId   if subscription request succeeded, otherwise ARRAY( 'level' => 'inform', 'spurious' or 'fatal', 'errorText', 'errorCode' => string )
	 *  of error to display
	 */
	protected function processSubscriptionPayment( $card, $paymentBasket, $now, &$ipn, &$occurrences, &$autorecurring_type, &$autorenew_type )
	{
		// https://stripe.com/docs/api#subscriptions
		// https://stripe.com/docs/subscriptions/quickstart
		if ( ! $this->hasStripeApi() ) {
			$this->_setLogErrorMSG( 3, null, CBTxt::T( 'Needed Stripe API Publishable Key and Secret Key not set.' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return $this->returnErrorResponse( 'fatal', CBTxt::T( 'Needed Stripe API Publishable Key and Secret Key not set.' ) );
		}

		// Lets see if we can find the source from direct or receiver methods:
		$source					=	Application::Input()->getString( 'post/cbptoken', '' );

		if ( ! $source ) {
			// Couldn't find a source so lets see if this is a redirect method payment intent:
			$source				=	Application::Input()->getString( 'get/payment_intent', '' );
		}

		if ( ! $source ) {
			// Couldn't find a source so lets see if this is a redirect method setup intent:
			$source				=	Application::Input()->getString( 'get/setup_intent', '' );
		}

		$logType				=	'';
		$requestParams			=	[];
		$results				=	new Registry();

		$autorecurring_type		=	2;
		$autorenew_type			=	( ( (int) $this->getAccountParam( 'enabled', 0 ) === 3 ) && ( $paymentBasket->isAnyAutoRecurring() === 2 ) ? 1 : 2 );

		if ( $paymentBasket->getString( 'payment_status', '' ) === 'Pending' ) {
			// Basket is already in a pending state so lets check on its charge status:
			$return				=	$this->processSubscriptionPaymentPending( $source, $paymentBasket, $logType, $results );

			if ( $results->get( 'charge_status', '' ) === 'pending' ) {
				// Just do nothing since we aren't changing the status
				return false;
			}
		} elseif ( ( ! $source ) && $paymentBasket->getParam( 'checkout_session.recurring', '', 'integrations', GetterInterface::STRING ) ) {
			// This is a checkout session basket so lets verify the checkout session charge:
			$return				=	$this->processSubscriptionPaymentCheckout( $source, $paymentBasket, $logType, $results );
		} elseif ( ( strpos( $source, 'pi_' ) === 0 ) || ( strpos( $source, 'seti_' ) === 0 ) || $this->hasStripeTestClocks() ) {
			// We need to check the status of this payment intent to ensure it has been completed:
			$return				=	$this->processSubscriptionPaymentIntent( $source, $paymentBasket, $logType, $results, $requestParams );
		} else {
			$this->_setLogErrorMSG( 3, null, CBTxt::T( 'Invalid payment source provided. Sources and Tokens are no longer supported.' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return $this->returnErrorResponse( 'fatal', CBTxt::T( 'Invalid payment source provided.' ) );
		}

		$ipn					=	$this->_logNotification( $logType, $now, $paymentBasket, $card, $requestParams, $results, $return );

		return $return;
	}

	/**
	 * Handles processing a subscription payments pending basket
	 * Checks if the pending charge has completed or not
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @return string|array
	 */
	private function processSubscriptionPaymentPending( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results )
	{
		global $_CB_framework;

		$subscriptionId			=	$paymentBasket->getString( 'subscr_id', '' );
		$chargeId				=	$paymentBasket->getString( 'txn_id', '' );

		// If we already have a charge id then just check its status instead of getting latest charge from the subscription
		if ( $chargeId && ( ( strpos( $chargeId, 'ch_' ) === 0 ) || ( strpos( $chargeId, 'py_' ) === 0 ) ) ) {
			$results			=	$this->getStripeCharge( $chargeId );

			if ( $results->has( 'error' ) ) {
				$error			=	$results->getString( 'error.message', '' );

				if ( ! $error ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

					$logType	=	'C';

					return $this->returnErrorResponse();
				}

				$status			=	$results->getString( 'error.code', '' );

				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'W';

				return $this->returnErrorResponse( 'fatal', $error, $status );
			}

			$chargeId			=	$results->getString( 'id', '' );
			$chargeStatus		=	$results->getString( 'status', '' );

			if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
				$error			=	$results->getString( 'failure_message', '' );
				$status			=	$results->getString( 'failure_code', '' );

				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'W';

				return $this->returnErrorResponse( 'fatal', $error, $status );
			}

			$logType			=	'A';

			$results->set( 'charge_id', $chargeId );
			$results->set( 'charge_status', $chargeStatus );

			return $subscriptionId;
		}

		// Check if the setup intent requires further action (e.g. 3DS authentication)
		$results				=	$this->getStripeSetupIntent( $paymentBasket->getParam( 'setup_intent.recurring', '', 'integrations', GetterInterface::STRING ) );

		if ( $results->has( 'error' ) ) {
			$error				=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intents API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'C';

				return $this->returnErrorResponse();
			}

			$status				=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intents API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		if ( ( $results->getString( 'status', '' ) === 'requires_action' ) && ( $results->getString( 'next_action.type', '' ) === 'redirect_to_url' ) ) {
			// Stripe JS failed to confirm the payment and there's a redirect required so goahead and do it:
			$_CB_framework->redirect( $results->getRaw( 'next_action.redirect_to_url.url', '' ) );
		}

		// Now we can check if the subscription is ready
		$results				=	$this->getStripeSubscription( $subscriptionId, [ 'latest_invoice.charge' ] );

		if ( $results->has( 'error' ) ) {
			$error				=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'C';

				return $this->returnErrorResponse();
			}

			$status				=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$chargeId				=	$results->getString( 'latest_invoice.charge.id', '' );
		$chargeStatus			=	$results->getString( 'latest_invoice.charge.status', '' );

		if ( ! $chargeId ) {
			$logType			=	'A';

			$results->set( 'charge_status', 'pending' );

			return $subscriptionId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error				=	$results->getString( 'latest_invoice.charge.failure_message', '' );
			$status				=	$results->getString( 'latest_invoice.charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType				=	'A';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $subscriptionId;
	}

	/**
	 * Handles processing a subscription payments checkout session
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @return string|array
	 */
	private function processSubscriptionPaymentCheckout( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results )
	{
		$results				=	$this->getStripeCheckoutSession( $paymentBasket->getParam( 'checkout_session.recurring', '', 'integrations', GetterInterface::STRING ), [ 'subscription.latest_invoice.charge' ] );

		if ( $results->has( 'error' ) ) {
			$error				=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType		=	'C';

				return $this->returnErrorResponse();
			}

			$status				=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$subscriptionId			=	$results->getString( 'subscription.id', '' );
		$chargeId				=	$results->getString( 'subscription.latest_invoice.charge.id', '' );
		$chargeStatus			=	$results->getString( 'subscription.latest_invoice.charge.status', '' );

		if ( ! $chargeId ) {
			$logType			=	'A';

			$results->set( 'charge_status', 'pending' );

			return $subscriptionId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error				=	$results->getString( 'subscription.latest_invoice.charge.failure_message', '' );
			$status				=	$results->getString( 'subscription.latest_invoice.charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType			=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType				=	'A';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $subscriptionId;
	}

	/**
	 * Handles processing a subscription payments setup intents and payment intents
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @param array               $requestParams
	 * @return string|array
	 */
	private function processSubscriptionPaymentIntent( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results, array &$requestParams )
	{
		global $_CB_framework;

		if ( $this->hasStripeTestClocks() ) {
			$clockId					=	$this->processSubscriptionTestClock( $paymentBasket, $logType, $results, $requestParams );

			if ( is_array( $clockId ) ) {
				// Test clock or its test customer failed so just return whatever error happened
				return $clockId;
			}
		} else {
			if ( strpos( $source, 'pi_' ) === 0 ) {
				$this->_setLogErrorMSG( 3, null, CBTxt::T( 'PaymentIntent attempted on Recurring Payment. PaymentIntent is only permitted for Single Payments.' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType				=	'C';

				return $this->returnErrorResponse();
			}

			$results					=	$this->getStripeSetupIntent( $source );

			if ( $results->has( 'error' ) ) {
				$error					=	$results->getString( 'error.message', '' );

				if ( ! $error ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intents API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

					$logType			=	'C';

					return $this->returnErrorResponse();
				}

				$status					=	$results->getString( 'error.code', '' );

				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intents API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType				=	'W';

				return $this->returnErrorResponse( 'fatal', $error, $status );
			}

			if ( ( $results->getString( 'status', '' ) === 'requires_action' ) && ( $results->getString( 'next_action.type', '' ) === 'redirect_to_url' ) ) {
				// Stripe JS failed to confirm the payment and there's a redirect required so goahead and do it:
				$_CB_framework->redirect( $results->getRaw( 'next_action.redirect_to_url.url', '' ) );
			}

			if ( ! in_array( $results->getString( 'status', '' ), [ 'processing', 'succeeded' ], true ) ) {
				// We do not perform manual confirmation for setup intents so if it isn't processing or succeeded then fail the payment:
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intent in an invalid status. STATUS: ' . $results->getString( 'status', '' ), CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType				=	'C';

				return $this->returnErrorResponse();
			}
		}

		$recurringPeriod				=	$paymentBasket->getString( 'period3', '' );
		$recurringPeriodLimits			=	$this->getStripePeriodLimits( explode( ' ', $recurringPeriod ) );
		$recurringAmount				=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount3', 0 ) );

		if ( $recurringPeriodLimits['error'] ) {
			// Period limits are not valid for a recurring subscription at stripe so abort creating the subscription:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Invalid plans duration', $recurringPeriodLimits['error'] );

			return $this->returnErrorResponse();
		}

		$initialPeriod					=	$paymentBasket->getString( 'period1', '' );
		$amountDiff						=	0;
		$trialEnd						=	'';

		if ( $initialPeriod ) {
			$initialAmount				=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount1', 0 ) );

			if ( ( $recurringAmount !== $initialAmount ) || ( $recurringPeriod !== $initialPeriod ) ) {
				if ( $initialAmount > 0 ) {
					if ( ( $initialAmount > $recurringAmount ) && ( $recurringPeriod === $initialPeriod ) ) {
						// The initial price is greater than the recurring price, but the duration is the same so lets take the difference between recurring and initial price and add it as an initial charge
						$amountDiff		=	( $initialAmount - $recurringAmount );
					} else {
						// The initial price is lower than recurring or duration are different so the only option is a subscription schedule to cover the two different price durations
						return $this->processSubscriptionPaymentIntentSchedule( $source, $paymentBasket, $logType, $results, $requestParams );
					}
				} else {
					// If the initial price is $0 then it's just a free trial so we can skip scheduling and utilize trial_end instead
					$trialEnd			=	$recurringPeriodLimits['start'];
				}
			}
		}

		$productId						=	$this->getStripeProductID( $paymentBasket );

		if ( ! $productId ) {
			// Product create failed; lets abort:
			return $this->returnErrorResponse();
		}

		// Create Subscription:
		// https://stripe.com/docs/api#subscriptions
		$requestParams					=	[	'customer'											=>	$results->getString( 'customer', '' ),
												'default_payment_method'							=>	$results->getString( 'payment_method', '' ),
												'description'										=>	$paymentBasket->getString( 'item_name', '' ),
												'payment_behavior'									=>	( $this->getAccountParam( 'stripe_sca_subscriptions', 'reject' ) === 'incomplete' ? 'allow_incomplete' : 'error_if_incomplete' ),
												'off_session'										=>	'true',
												'items[0][price_data][currency]'					=>	strtolower( $paymentBasket->getString( 'mc_currency', '' ) ),
												'items[0][price_data][product]'						=>	$productId,
												'items[0][price_data][recurring][interval]'			=>	$recurringPeriodLimits['interval'],
												'items[0][price_data][recurring][interval_count]'	=>	$recurringPeriodLimits['interval_count'],
												'items[0][price_data][unit_amount]'					=>	( $recurringAmount * 100 ),
												'metadata[user_id]'									=>	$paymentBasket->getInt( 'user_id', 0 ),
												'metadata[order_id]'								=>	$paymentBasket->getInt( 'id', 0 ),
												'metadata[invoice]'									=>	$paymentBasket->getString( 'invoice', '' ),
												'metadata[gateway]'									=>	(int) $this->getAccountParam( 'id', 0 ),
												'expand'											=>	[ 'latest_invoice.charge' ],
											];

		if ( $trialEnd ) {
			$requestParams['trial_end']																=	$trialEnd;
		}

		if ( $amountDiff ) {
			$requestParams['add_invoice_items[0][price_data][currency]']							=	strtolower( $paymentBasket->getString( 'mc_currency', '' ) );
			$requestParams['add_invoice_items[0][price_data][product]']								=	$productId;
			$requestParams['add_invoice_items[0][price_data][unit_amount]']							=	( $amountDiff * 100 );
		}

		$results						=	$this->httpsRequestStripe( 'post', '/subscriptions', $requestParams );

		if ( $results->has( 'error' ) ) {
			$error						=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType				=	'C';

				return $this->returnErrorResponse();
			}

			$status						=	$results->getString( 'error.code', '' );

			if ( $status === 'subscription_payment_intent_requires_action' ) {
				$userError				=	CBTxt::T( 'Your payment method requires authenticate for every payment, including recurring payments, and cannot be used for subscriptions.' );
			} else {
				$userError				=	$error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' );
			}

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API error returned. ERROR: ' . $error . ' CODE: ' . $status, $userError );

			$logType					=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$paymentIntent					=	$results->getString( 'latest_invoice.payment_intent', '' );

		if ( $paymentIntent ) {
			$paymentBasket->setParam( 'payment_intent.recurring', $paymentIntent, 'integrations' );

			$paymentBasket->storeParams( 'integrations' );
		}

		$subscriptionId					=	$results->getString( 'id', '' );
		$chargeId						=	$results->getString( 'latest_invoice.charge.id', '' );
		$chargeStatus					=	$results->getString( 'latest_invoice.charge.status', '' );

		if ( ! $chargeId ) {
			// Charge doesn't exist yet for this subscription so lets mark it pending and let the webhook handle it:
			$logType					=	'A';

			$results->set( 'charge_status', 'pending' );

			return $subscriptionId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error						=	$results->getString( 'latest_invoice.charge.failure_message', '' );
			$status						=	$results->getString( 'latest_invoice.charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Charges API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType					=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType						=	'A';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $subscriptionId;
	}

	/**
	 * Handles processing a subscription payment using a test clock
	 * This will set acceptable values into $registry to process remaining subscription behavior
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @param array               $requestParams
	 * @return string|array
	 */
	private function processSubscriptionTestClock( cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results, array &$requestParams )
	{
		// Create Test Clock:
		// https://stripe.com/docs/api/test_clocks/create
		$requestParams		=	[	'frozen_time'	=>	Application::Application()->getStartTime(),
									'name'			=>	$paymentBasket->getString( 'item_name', '' ),
								];

		$clockResults		=	$this->httpsRequestStripe( 'post', '/test_helpers/test_clocks', $requestParams );

		if ( $clockResults->has( 'error' ) ) {
			$error			=	$clockResults->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Test Clock API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType	=	'C';

				return $this->returnErrorResponse();
			}

			$status			=	$clockResults->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Test Clock API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType		=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		// Create Test Customer:
		// https://stripe.com/docs/api/customers/create
		$requestParams		=	[	'email'										=>	$paymentBasket->getString( 'payer_email', '' ),
									'test_clock'								=>	$clockResults->getString( 'id', '' ),
									'payment_method'							=>	'pm_card_visa',
									'invoice_settings[default_payment_method]'	=>	'pm_card_visa',
								];

		$customerResults	=	$this->httpsRequestStripe( 'post', '/customers', $requestParams );

		if ( $customerResults->has( 'error' ) ) {
			$error			=	$customerResults->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Test Customer API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType	=	'C';

				return $this->returnErrorResponse();
			}

			$status			=	$customerResults->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Test Customer API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType		=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$results			=	new Registry();

		// Build the expected results values needed to process a subscription in processSubscriptionPaymentIntent
		$results->set( 'test_clock', $clockResults->getString( 'id', '' ) );
		$results->set( 'customer', $customerResults->getString( 'id', '' ) );
		$results->set( 'payment_method', $customerResults->getString( 'invoice_settings.default_payment_method', '' ) );

		return $clockResults->getString( 'id', '' );
	}

	/**
	 * Handles processing a subscription payments setup intents and payment intents
	 *
	 * @param string              $source
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $logType
	 * @param Registry            $results
	 * @param array               $requestParams
	 * @return string|array
	 */
	private function processSubscriptionPaymentIntentSchedule( string $source, cbpaidPaymentBasket $paymentBasket, string &$logType, Registry &$results, array &$requestParams )
	{
		$productId						=	$this->getStripeProductID( $paymentBasket );

		if ( ! $productId ) {
			// Product create failed; lets abort:
			return $this->returnErrorResponse();
		}

		$initialPeriodLimits			=	$this->getStripePeriodLimits( explode( ' ', $paymentBasket->getString( 'period1', '' ) ) );
		$initialAmount					=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount1', 0 ) );

		if ( $initialPeriodLimits['error'] ) {
			// Period limits are not valid for a recurring subscription at stripe so abort creating the subscription:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Invalid plans duration', $initialPeriodLimits['error'] );

			return $this->returnErrorResponse();
		}

		$recurringPeriodLimits			=	$this->getStripePeriodLimits( explode( ' ', $paymentBasket->getString( 'period3', '' ) ) );
		$recurringAmount				=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount3', 0 ) );

		if ( $recurringPeriodLimits['error'] ) {
			// Period limits are not valid for a recurring subscription at stripe so abort creating the subscription:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Invalid plans duration', $recurringPeriodLimits['error'] );

			return $this->returnErrorResponse();
		}

		$initialEnd						=	$recurringPeriodLimits['start'];

		// Create Subscription Schedule:
		// https://stripe.com/docs/api/subscription_schedules
		$requestParams					=	[	'customer'														=>	$results->getString( 'customer', '' ),
												'start_date'													=>	'now',
												'end_behavior'													=>	'release',
												'default_settings[default_payment_method]'						=>	$results->getString( 'payment_method', '' ),
												'default_settings[description]'									=>	$paymentBasket->getString( 'item_name', '' ),
												'phases[0][items][0][price_data][currency]'						=>	strtolower( $paymentBasket->getString( 'mc_currency', '' ) ),
												'phases[0][items][0][price_data][product]'						=>	$productId,
												'phases[0][items][0][price_data][recurring][interval]'			=>	$initialPeriodLimits['interval'],
												'phases[0][items][0][price_data][recurring][interval_count]'	=>	$initialPeriodLimits['interval_count'],
												'phases[0][items][0][price_data][unit_amount]'					=>	( $initialAmount * 100 ),
												'phases[0][items][0][metadata][user_id]'						=>	$paymentBasket->getInt( 'user_id', 0 ),
												'phases[0][items][0][metadata][order_id]'						=>	$paymentBasket->getInt( 'id', 0 ),
												'phases[0][items][0][metadata][invoice]'						=>	$paymentBasket->getString( 'invoice', '' ),
												'phases[0][items][0][metadata][gateway]'						=>	(int) $this->getAccountParam( 'id', 0 ),
												'phases[0][description]'										=>	$paymentBasket->getString( 'item_name', '' ),
												'phases[0][metadata][user_id]'									=>	$paymentBasket->getInt( 'user_id', 0 ),
												'phases[0][metadata][order_id]'									=>	$paymentBasket->getInt( 'id', 0 ),
												'phases[0][metadata][invoice]'									=>	$paymentBasket->getString( 'invoice', '' ),
												'phases[0][metadata][gateway]'									=>	(int) $this->getAccountParam( 'id', 0 ),
												'phases[0][proration_behavior]'									=>	'none',
												'phases[0][end_date]'											=>	$initialEnd,
												'phases[1][items][0][price_data][currency]'						=>	strtolower( $paymentBasket->getString( 'mc_currency', '' ) ),
												'phases[1][items][0][price_data][product]'						=>	$productId,
												'phases[1][items][0][price_data][recurring][interval]'			=>	$recurringPeriodLimits['interval'],
												'phases[1][items][0][price_data][recurring][interval_count]'	=>	$recurringPeriodLimits['interval_count'],
												'phases[1][items][0][price_data][unit_amount]'					=>	( $recurringAmount * 100 ),
												'phases[1][items][0][metadata][user_id]'						=>	$paymentBasket->getInt( 'user_id', 0 ),
												'phases[1][items][0][metadata][order_id]'						=>	$paymentBasket->getInt( 'id', 0 ),
												'phases[1][items][0][metadata][invoice]'						=>	$paymentBasket->getString( 'invoice', '' ),
												'phases[1][items][0][metadata][gateway]'						=>	(int) $this->getAccountParam( 'id', 0 ),
												'phases[1][description]'										=>	$paymentBasket->getString( 'item_name', '' ),
												'phases[1][metadata][user_id]'									=>	$paymentBasket->getInt( 'user_id', 0 ),
												'phases[1][metadata][order_id]'									=>	$paymentBasket->getInt( 'id', 0 ),
												'phases[1][metadata][invoice]'									=>	$paymentBasket->getString( 'invoice', '' ),
												'phases[1][metadata][gateway]'									=>	(int) $this->getAccountParam( 'id', 0 ),
												'phases[1][proration_behavior]'									=>	'none',
												'metadata[user_id]'												=>	$paymentBasket->getInt( 'user_id', 0 ),
												'metadata[order_id]'											=>	$paymentBasket->getInt( 'id', 0 ),
												'metadata[invoice]'												=>	$paymentBasket->getString( 'invoice', '' ),
												'metadata[gateway]'												=>	(int) $this->getAccountParam( 'id', 0 ),
												'expand'														=>	[ 'subscription.latest_invoice.charge' ],
											];

		$results						=	$this->httpsRequestStripe( 'post', '/subscription_schedules', $requestParams );

		if ( $results->has( 'error' ) ) {
			$error						=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions Schedules API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

				$logType				=	'C';

				return $this->returnErrorResponse();
			}

			$status						=	$results->getString( 'error.code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions Schedules API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType					=	'W';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$paymentIntent					=	$results->getString( 'subscription.latest_invoice.payment_intent', '' );

		if ( $paymentIntent ) {
			$paymentBasket->setParam( 'payment_intent.recurring', $paymentIntent, 'integrations' );

			$paymentBasket->storeParams( 'integrations' );
		}

		$subscriptionId					=	$results->getString( 'subscription.id', '' );
		$chargeId						=	$results->getString( 'subscription.latest_invoice.charge.id', '' );
		$chargeStatus					=	$results->getString( 'subscription.latest_invoice.charge.status', '' );

		if ( ! $chargeId ) {
			$logType					=	'A';

			$results->set( 'charge_status', 'pending' );

			return $subscriptionId;
		}

		if ( ! in_array( $chargeStatus, [ 'succeeded', 'pending' ], true ) ) {
			$error						=	$results->getString( 'subscription.latest_invoice.charge.failure_message', '' );
			$status						=	$results->getString( 'subscription.latest_invoice.charge.failure_code', '' );

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions Schedules API error returned. ERROR: ' . $error . ' CODE: ' . $status, $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			$logType					=	'V';

			return $this->returnErrorResponse( 'fatal', $error, $status );
		}

		$logType						=	'A';

		$results->set( 'charge_id', $chargeId );
		$results->set( 'charge_status', $chargeStatus );

		return $subscriptionId;
	}

	/**
	 * Attempts to unsubscribe a subscription of a payment basket.
	 *
	 * @param  cbpaidPaymentBasket              $paymentBasket
	 * @param  cbpaidPaymentItem[]              $paymentItems
	 * @param  cbpaidsubscriptionsNotification  $ipn                        returns the stored notification
	 * @param  string                           $authorize_subscription_id
	 * @return string|array                                                 subscriptionId if subscription request succeeded, otherwise ARRAY( 'level' => 'inform', 'spurious' or 'fatal', 'errorText', 'errorCode' => string ) of error to display
	 */
	protected function processSubscriptionCancellation( $paymentBasket, $paymentItems, &$ipn, $authorize_subscription_id )
	{
		global $_CB_framework;

		// https://stripe.com/docs/api#cancel_subscription
		// https://stripe.com/docs/subscriptions/canceling-pausing
		if ( ! $this->hasStripeApi() ) {
			return $this->returnErrorResponse( 'fatal', CBTxt::T( 'Needed Stripe API Publishable Key and Secret Key not set.' ) );
		}

		$results					=	$this->getStripeSubscription( $authorize_subscription_id );

		if ( $results->has( 'error' ) ) {
			if ( $results->getString( 'error.code', '' ) === '404' ) {
				// Just treat missing resources as cancelled since they don't exist
				$return				=	$authorize_subscription_id;
				$logType			=	'5';
			} else {
				$error				=	$results->getString( 'error.message', '' );

				if ( ! $error ) {
					$return			=	$this->returnErrorResponse();
					$logType		=	'C';
				} else {
					$return			=	$this->returnErrorResponse( 'fatal', $error, $results->getString( 'error.code', '' ) );
					$logType		=	'W';
				}
			}
		} elseif ( $results->getString( 'status', '' ) === 'canceled' ) {
			$return					=	$authorize_subscription_id;
			$logType				=	'5';
		} else {
			$results				=	$this->cancelStripeSubscription( $authorize_subscription_id );

			if ( $results->has( 'error' ) ) {
				if ( $results->getString( 'error.code', '' ) === '404' ) {
					// Just treat missing resources as cancelled since they don't exist
					$return			=	$authorize_subscription_id;
					$logType		=	'5';
				} else {
					$error			=	$results->getString( 'error.message', '' );

					if ( ! $error ) {
						$return		=	$this->returnErrorResponse();
						$logType	=	'C';
					} else {
						$return		=	$this->returnErrorResponse( 'fatal', $error, $results->getString( 'error.code', '' ) );
						$logType	=	'W';
					}
				}
			} else {
				$return				=	$authorize_subscription_id;
				$logType			=	'5';
			}
		}

		$ipn						=	$this->_logNotification( $logType, $_CB_framework->now(), $paymentBasket, null, [], $results, $return );

		return $return;
	}

	/**
	 * Handles a gateway notification
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param array               $postdata
	 * @return null|bool                          always null as resultNotification > handleNotify return value is never used except to echo it
	 */
	protected function handleNotify( $paymentBasket, $postdata ): ?bool
	{
		global $_CB_framework;

		// https://stripe.com/docs/api#events
		// https://stripe.com/docs/webhooks
		$eventBody										=	@file_get_contents( 'php://input' );

		if ( ! $eventBody ) {
			return null;
		}

		$eventData										=	new Registry( json_decode( $eventBody, true ) );

		if ( $eventData->getString( 'object', '' ) !== 'event' ) {
			echo 'webhook skipped: unknown object';

			return null;
		}

		$eventType										=	$eventData->getString( 'type', '' );

		if ( ! in_array( $eventType, $this->getStripeWebhooks(), true ) ) {
			echo 'webhook skipped: handling for ' . htmlspecialchars( $eventType ) . ' is not implemented';

			return null;
		}

		// Delay invoice, payment intent, and checkout session processing to give priority to charges followed by invoices, intents, and finally checkout session
		// This is necessary as a payment intent webhook can arrive and process concurrently with a charge webhook causing a double payment
		// However we can't just stop processing payment intents as there may not even be a charge yet and the intent would give us a pending status
		switch ( $eventType ) {
			case 'checkout.session.completed':
				sleep( 4 );
				break;
			case 'payment_intent.succeeded':
			case 'payment_intent.payment_failed':
				sleep( 3 );
				break;
			case 'invoice.payment_succeeded':
			case 'invoice.payment_failed':
				sleep( 2 );
				break;
		}

		$eventVersion									=	$eventData->getString( 'api_version', $this->stripeApiVersion );

		if ( version_compare( $eventVersion, $this->stripeApiVersion, '<' ) ) {
			// Log info regarding webhook version being out of date so they can update it, but we'll try to be full B/C below with older webhooks
			$this->_setLogErrorMSG( 6, null, $this->getPayName() . ' Webhook version mismatch.' . "\n" . 'Received=' . var_export( $_GET, true ) . "\n" . 'Expected=' . var_export( $eventData->asArray(), true ) . "\n", '' );
		}

		// Explicitly define our ids so we know what is available, but in most cases we only need $transactionId which typically is the charge id
		$chargeId										=	'';
		$refundId										=	'';
		$disputeId										=	'';
		$invoiceId										=	'';
		$intentId										=	'';
		$transactionId									=	'';
		$subscriptionId									=	'';

		/** @var Registry $eventObject */
		$eventObject									=	$eventData->subTree( 'data.object' );
		$objectType										=	$eventObject->getString( 'object', '' );

		// Build the necessary ids for processing an event based off the object in that event
		switch ( $objectType ) {
			case 'invoice';
				$invoiceId								=	$eventObject->getString( 'id', '' );
				$intentId								=	$eventObject->getString( 'payment_intent', '' );
				$chargeId								=	$eventObject->getString( 'charge', '' );
				$transactionId							=	$chargeId;
				$subscriptionId							=	$eventObject->getString( 'subscription', '' );

				if ( ! $transactionId ) {
					echo 'webhook skipped: waiting for charge';

					return null; // Ignore invoices without a charge
				}
				break;
			case 'subscription';
				$subscriptionId							=	$eventObject->getString( 'id', '' );
				$invoiceId								=	$eventObject->getString( 'latest_invoice', '' );

				if ( $invoiceId ) {
					$chargeId							=	$this->getStripeInvoice( $invoiceId )->getString( 'charge', '' );
					$transactionId						=	$chargeId;
				}
				break;
			case 'dispute';
				$disputeId								=	$eventObject->getString( 'id', '' );
				$transactionId							=	$disputeId;
				$chargeId								=	$eventObject->getString( 'charge', '' );

				if ( $chargeId ) {
					$disputeCharge						=	$this->getStripeCharge( $chargeId );

					$intentId							=	$disputeCharge->getString( 'payment_intent', '' );
					$invoiceId							=	$disputeCharge->getString( 'invoice', '' );

					if ( $invoiceId ) {
						$subscriptionId					=	$this->getStripeInvoice( $invoiceId )->getString( 'subscription', '' );
					}
				}
				break;
			case 'charge';
				$chargeId								=	$eventObject->getString( 'id', '' );
				$transactionId							=	$chargeId;
				$intentId								=	$eventObject->getString( 'payment_intent', '' );
				$invoiceId								=	$eventObject->getString( 'invoice', '' );

				if ( $invoiceId ) {
					$subscriptionId						=	$this->getStripeInvoice( $invoiceId )->getString( 'subscription', '' );
				}

				if ( $eventType === 'charge.refunded' ) {
					// If this is a refund then set the transaction id to the refund id as we don't want a double-refund logged
					$refundId							=	$this->getStripeCharge( $eventObject->getString( 'id', '' ), [ 'refunds' ] )->getString( 'refunds.data.0.id', '' );
					$transactionId						=	$refundId;

					if ( ! $transactionId ) {
						echo 'webhook skipped: waiting for refund';

						return null; // Ignore charges without a refund id
					}
				}
				break;
			case 'payment_intent';
				$intentId								=	$eventObject->getString( 'id', '' );
				$invoiceId								=	$eventObject->getString( 'invoice', '' );
				$chargeId								=	$eventObject->getString( 'latest_charge', '' );
				$transactionId							=	$chargeId;

				if ( ! $transactionId ) {
					echo 'webhook skipped: waiting for charge';

					return null; // Ignore payment intents without a charge
				}

				if ( $invoiceId ) {
					$subscriptionId						=	$this->getStripeInvoice( $invoiceId )->getString( 'subscription', '' );
				}
				break;
			case 'checkout.session';
				$intentId								=	$eventObject->getString( 'payment_intent', '' );
				$subscriptionId							=	$eventObject->getString( 'subscription', '' );

				if ( $intentId ) {
					$checkoutIntent						=	$this->getStripePaymentIntent( $intentId );

					$invoiceId							=	$checkoutIntent->getString( 'invoice', '' );
					$chargeId							=	$checkoutIntent->getString( 'latest_charge.id', '' );
				} elseif ( $subscriptionId ) {
					$checkoutSubscription				=	$this->getStripeSubscription( $subscriptionId );

					$invoiceId							=	$checkoutSubscription->getString( 'latest_invoice.id', '' );
					$chargeId							=	$checkoutSubscription->getString( 'latest_invoice.charge', '' );
				}

				$transactionId							=	$chargeId;

				if ( ! $transactionId ) {
					echo 'webhook skipped: waiting for charge';

					return null; // Ignore checkout sessions without a charge
				}
				break;
		}

		if ( ! ( $transactionId || $subscriptionId ) ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Stripe Charge ID or Subscription ID (object.id) missing.' . "\n" . '$_GET=' . var_export( $_GET, true ) . "\n" . '$_POST=' . var_export( $eventData->asArray(), true ) . "\n", CBTxt::T( 'Transaction not found.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			echo 'webhook skipped: missing transaction or subscription id';

			return null;
		}

		$paymentStatus									=	'';
		$disputeStatus									=	'';

		// Explicitly process events by their type so we know exactly what information we have and what to do
		switch ( $eventType ) {
			case 'charge.succeeded':
			case 'payment_intent.succeeded':
				switch ( $eventObject->getString( 'status', '' ) ) {
					case 'succeeded';
						$paymentStatus					=	'Completed';
						break;
					case 'failed';
						$paymentStatus					=	'Denied';
						break;
					case 'pending';
						$paymentStatus					=	'Pending';
						break;
				}
				break;
			case 'charge.refunded':
				$paymentStatus							=	'Refunded';
				break;
			case 'charge.dispute.created':
				switch ( $eventObject->getString( 'status', '' ) ) {
					case 'lost';
					case 'charge_refunded';
						// Dispute was lost immediately so reverse the payment and cancel the recurring subscription
						$paymentStatus					=	'Reversed';
						$disputeStatus					=	'Lost_Reversal'; // This is specific to Stripe implementation below
						break;
					case 'needs_response';
					case 'warning_needs_response';
						$paymentStatus					=	'Reversed';
						break;
				}
				break;
			case 'charge.dispute.closed':
				switch ( $eventObject->getString( 'status', '' ) ) {
					case 'won';
					case 'warning_closed';
						$paymentStatus					=	'Canceled_Reversal';
						break;
					case 'lost';
					case 'charge_refunded';
						$paymentStatus					=	'Lost_Reversal'; // This is specific to Stripe implementation below
						break;
				}
				break;
			case 'charge.failed':
			case 'invoice.payment_failed':
			case 'payment_intent.payment_failed':
				$paymentStatus							=	'Denied';
				break;
			case 'invoice.payment_succeeded':
				switch ( $eventObject->getString( 'status', '' ) ) {
					case 'paid';
						$paymentStatus					=	'Completed';
						break;
					case 'uncollectible';
					case 'void';
						$paymentStatus					=	'Denied';
						break;
					case 'draft';
					case 'open';
						$paymentStatus					=	'Pending';
						break;
				}
				break;
			case 'checkout.session.completed':
				if ( $intentId ) {
					switch ( $this->getStripePaymentIntent( $intentId )->getString( 'latest_charge.status', '' ) ) {
						case 'succeeded';
							$paymentStatus				=	'Completed';
							break;
						case 'failed';
							$paymentStatus				=	'Denied';
							break;
						case 'pending';
							$paymentStatus				=	'Pending';
							break;
					}
				} elseif ( $subscriptionId ) {
					switch ( $this->getStripeSubscription( $subscriptionId )->getString( 'latest_invoice.status', '' ) ) {
						case 'paid';
							$paymentStatus				=	'Completed';
							break;
						case 'uncollectible';
						case 'void';
							$paymentStatus				=	'Denied';
							break;
						case 'draft';
						case 'open';
							$paymentStatus				=	'Pending';
							break;
					}
				} else {
					switch ( $eventObject->getString( 'payment_status', '' ) ) {
						case 'paid';
							$paymentStatus				=	'Completed';
							break;
						case 'unpaid';
							$paymentStatus				=	'Pending';
							break;
					}
				}
				break;
			case 'customer.subscription.deleted':
				$paymentStatus							=	'Unsubscribed';
				break;
		}

		if ( ! $paymentStatus ) {
			echo 'webhook skipped: unknown payment status';

			return null;
		}

		if ( $objectType === 'checkout.session' ) {
			$orderId									=	$eventObject->getInt( 'client_reference_id', 0 );
		} else {
			$orderId									=	$eventObject->getInt( 'metadata.order_id', 0 );
		}

		$exists											=	false;

		if ( $orderId ) {
			$exists										=	$paymentBasket->load( $orderId );
		}

		// If we failed to find the basket using metadata it's time to desperately try to locate it based off data that may be stored
		if ( ( ! $exists ) && $subscriptionId ) {
			$exists										=	$paymentBasket->loadThisMatching( [ 'subscr_id' => $subscriptionId ] );
		}

		if ( ( ! $exists ) && $transactionId ) {
			$exists										=	$paymentBasket->loadThisMatching( [ 'txn_id' => $transactionId ] );
		}

		if ( ( ! $exists ) && $chargeId && ( $chargeId !== $transactionId ) ) {
			$exists										=	$paymentBasket->loadThisMatching( [ 'txn_id' => $chargeId ] );
		}

		if ( ( ! $exists ) && $transactionId ) {
			$exists										=	$paymentBasket->loadThisMatching( [ 'parent_txn_id' => $transactionId ] );
		}

		if ( ( ! $exists ) && $chargeId && ( $chargeId !== $transactionId ) ) {
			$exists										=	$paymentBasket->loadThisMatching( [ 'parent_txn_id' => $chargeId ] );
		}

		if ( ( ! $exists ) && ( ! $orderId ) ) {
			// We can't seam to locate the basket AND it doesn't have any metadata so lets give it one more try by parsing through the objects to find the metadata
			// This can commonly happen when a subscription schedule updates the subscription as it doesn't propogate the metadata down from the schedule to its charges or intents
			$this->setStripeMetadata( $eventObject );

			$orderId									=	$eventObject->getInt( 'metadata.order_id', 0 );

			if ( $orderId ) {
				$exists									=	$paymentBasket->load( $orderId );
			}
		}

		$orgTransactionId								=	$paymentBasket->getString( 'txn_id', '' );

		if ( ! $subscriptionId ) {
			// IPN did not contain subscription id so lets just grab it from the basket as it should exist since we set this immediately when we create the subscriptions:
			$subscriptionId								=	$paymentBasket->getString( 'subscr_id', '' );
		}

		$signingSecret									=	'';

		if ( $exists ) {
			$gatewayId									=	(int) $this->getAccountParam( 'id', 0 );
			$basketGateway								=	$paymentBasket->getInt( 'gateway_account', 0 );

			if ( ! $basketGateway ) {
				$basketGateway							=	$eventObject->getInt( 'metadata.gateway', 0 );
			}

			if ( $basketGateway && ( $gatewayId !== $basketGateway ) ) {
				// IPN for a different Stripe gateway came through so lets update the gateway we're acting against here so we only need 1 webhook URL for all of Stripe:
				$payAccount								=	cbpaidControllerPaychoices::getInstance()->getPayAccount( $basketGateway );

				if ( ! $payAccount )  {
					echo 'webhook skipped: unknown payment gateway';

					return null;
				}

				// Set the signing secret to the currently loaded gateway as the IPN will be validated against it and not the gateway we're about to load
				$signingSecret							=	$this->getAccountParam( 'stripe_ss', '' );

				// Only override the current gateway object if the basket gateway actually exists:
				$this->__construct( $payAccount );

				// Delay the processing to be sure we don't process 2 webhooks from different sources at the same time incase more than 1 webhook url for CBSubs is configured
				sleep( 3 );

				// Reload the basket encase it changed during the delay
				$exists									=	$paymentBasket->load( $paymentBasket->getInt( 'id', 0 ) );
			}
		}

		if ( $exists && (
				( ( $paymentStatus === 'Completed' ) &&
				  ( in_array( $paymentBasket->getString( 'payment_status', '' ), [ 'Pending', 'NotInitiated' ], true ) // New payments from a current unpaid state
					|| ( $subscriptionId && ( $paymentBasket->getString( 'payment_status', '' ) !== 'Reversed' ) && ( $orgTransactionId !== $transactionId ) ) // Subscription recurring payments (we're checking that it's a new transaction as we want to skip duplicate IPNs)
				  )
				)
				|| ( ( $paymentStatus === 'Unsubscribed' ) && $subscriptionId && ( $paymentBasket->getString( 'payment_status', '' ) !== 'Unsubscribed' ) ) // Subscription cancelled
				|| ( ( $paymentStatus === 'Denied' ) && $subscriptionId && ( $paymentBasket->getString( 'payment_status', '' ) === 'Completed' ) ) // Recurring payment failed
				|| ( ( $paymentStatus === 'Denied' ) && ( $paymentBasket->getString( 'payment_status', '' ) === 'Pending' ) ) // Pending payment failed
				|| ( ( $paymentStatus === 'Reversed' ) && ( $paymentBasket->getString( 'payment_status', '' ) === 'Completed' ) ) // Dispute created
				|| ( ( $paymentStatus === 'Canceled_Reversal' ) && ( $paymentBasket->getString( 'payment_status', '' ) === 'Reversed' ) ) // Dispute won
				|| ( ( $paymentStatus === 'Lost_Reversal' ) && ( $paymentBasket->getString( 'payment_status', '' ) === 'Reversed' ) ) // Dispute lost
				|| ( ( $paymentStatus === 'Refunded' ) && in_array( $paymentBasket->getString( 'payment_status', '' ), [ 'Completed', 'Partially-Refunded' ], true ) ) // Refund of completed payment
			) ) {
			if ( $paymentStatus === 'Lost_Reversal' ) {
				// If this is a subscription force Stripe to cancel it since we lost the dispute and can't charge the customer:
				if ( $subscriptionId ) {
					$stopResults						=	$this->cancelStripeSubscription( $subscriptionId );

					if ( $stopResults->has( 'error' ) && ( $stopResults->getString( 'error.code', '' ) !== '404' ) ) {
						$stopError						=	$stopResults->getString( 'error.message', '' );

						if ( ! $stopError ) {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription cancellation didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						} else {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API error returned. ERROR: ' . $stopError . ' CODE: ' . $stopResults->getString( 'error.code', '' ), $stopError . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						}
					}
				}

				echo 'webhook skipped: dispute lost so nothing to do';

				return null; // Lost the dispute so there's nothing more to do
			}

			// The API calls were successful, but CBSubs timed out or errored during basket storage; we need to let Stripe IPN pay the basket or we end up with a payment made, but no subscription:
			if ( $paymentBasket->getString( 'payment_status', '' ) === 'NotInitiated' ) {
				$paymentBasket->set( 'payment_method', $this->getPayName() );
				$paymentBasket->set( 'gateway_account', (int) $this->getAccountParam( 'id', 0 ) );
			}

			$ipn										=	$this->_prepareIpn( 'I', $paymentStatus, $this->getStripePaymentType( $eventObject ), null, $_CB_framework->now(), 'utf-8' );

			$ipn->set( 'test_ipn', (int) ( ! $eventData->getBool( 'livemode', true ) ) );

			$ipn->bindBasket( $paymentBasket );

			$rawData									=	'$eventBody="' . preg_replace( '/([^\s]{100})/', '$1 ', $eventBody ) . "\"\n"
														.	'$eventData=' . var_export( $eventData->asArray(), true ) . ";\n";

			$ipn->setRawData( $rawData );

			$address									=	[];

			switch ( $objectType ) {
				case 'charge':
					$address							=	[	'name'					=>	'billing_details.name',
																'address_street'		=>	'billing_details.address.line1',
																'address_city'			=>	'billing_details.address.city',
																'address_state'			=>	'billing_details.address.state',
																'address_country_code'	=>	'billing_details.address.country',
																'address_zip'			=>	'billing_details.address.postal_code',
															];
					break;
				case 'invoice':
					$address							=	[	'name'					=>	'customer_name',
																'address_street'		=>	'customer_address.line1',
																'address_city'			=>	'customer_address.city',
																'address_state'			=>	'customer_address.state',
																'address_country_code'	=>	'customer_address.country',
																'address_zip'			=>	'customer_address.postal_code',
															];
					break;
				case 'checkout.session':
					$address							=	[	'name'					=>	'customer_details.name',
																'address_street'		=>	'customer_details.address.line1',
																'address_city'			=>	'customer_details.address.city',
																'address_state'			=>	'customer_details.address.state',
																'address_country_code'	=>	'customer_details.address.country',
																'address_zip'			=>	'customer_details.address.postal_code',
															];
					break;
			}

			foreach ( $address as $addressTo => $addressFrom ) {
				if ( $addressTo === 'name' ) {
					$name								=	explode( ' ', $eventObject->getString( $addressFrom, '' ) );

					$ipn->set( 'first_name', ( $name[0] ?? '' ) );
					$ipn->set( 'last_name', ( $name[1] ?? '' ) );

					continue;
				}

				$ipn->set( $addressTo, $eventObject->getString( $addressFrom, '' ) );
			}

			if ( $ipn->getString( 'payment_status', '' ) === 'Refunded' ) {
				$ipn->set( 'txn_id', $refundId );
				$ipn->set( 'parent_txn_id', $chargeId );
			} elseif ( in_array( $ipn->getString( 'payment_status', '' ), [ 'Reversed', 'Canceled_Reversal' ], true ) ) {
				$ipn->set( 'txn_id', $disputeId );
				$ipn->set( 'parent_txn_id', $chargeId );
			} else {
				$ipn->set( 'txn_id', $transactionId );
			}

			$ipn->set( 'user_id', $paymentBasket->getInt( 'user_id', 0 ) );
			$ipn->set( 'txn_type', 'web_accept' );

			if ( $subscriptionId ) {
				$ipn->set( 'subscr_id', $subscriptionId );

				if ( ( $paymentStatus === 'Completed' ) && in_array( $paymentBasket->getString( 'payment_status', '' ), [ 'Pending', 'NotInitiated' ], true ) ) {
					$ipn->set( 'txn_type', 'subscr_signup' );
					$ipn->set( 'subscr_date', $ipn->getString( 'payment_date', '' ) );
				} elseif ( $paymentStatus === 'Denied' ) {
					if ( ( $paymentBasket->getInt( 'reattempts_tried', 0 ) + 1 ) <= cbpaidScheduler::getInstance( $this )->retries ) {
						$ipn->set( 'txn_type', 'subscr_failed' );
					} else {
						$ipn->set( 'txn_type', 'subscr_cancel' );
					}
				} elseif ( $paymentStatus === 'Unsubscribed' ) {
					$ipn->set( 'txn_type', 'subscr_cancel' );
				} elseif ( $paymentStatus === 'Reversed' ) {
					$ipn->set( 'txn_type', 'subscr_failed' );
				} elseif ( in_array( $paymentStatus, [ 'Completed', 'Processed', 'Pending', 'Canceled_Reversal' ], true ) ) {
					$ipn->set( 'txn_type', 'subscr_payment' );
				} else {
					$ipn->set( 'txn_type', 'subscr_modify' );
				}
			}

			switch ( $objectType ) {
				case 'charge':
				case 'dispute':
				case 'payment_intent':
					$ipn->set( 'mc_gross', ( $eventObject->getFloat( 'amount', 0 ) / 100 ) );
					$ipn->set( 'mc_currency', strtoupper( $eventObject->getString( 'currency', '' ) ) );
					break;
				case 'invoice':
					$ipn->set( 'mc_gross', ( $eventObject->getFloat( 'amount_due', 0 ) / 100 ) );
					$ipn->set( 'mc_currency', strtoupper( $eventObject->getString( 'currency', '' ) ) );
					break;
				case 'subscription':
					$ipn->set( 'mc_gross', ( $eventObject->getFloat( 'plan.amount', 0 ) / 100 ) );
					$ipn->set( 'mc_currency', strtoupper( $eventObject->getString( 'plan.currency', '' ) ) );
					break;
				case 'checkout.session':
					$ipn->set( 'mc_gross', ( $eventObject->getFloat( 'amount_total', 0 ) / 100 ) );
					$ipn->set( 'mc_currency', strtoupper( $eventObject->getString( 'currency', '' ) ) );
					break;
			}

			if ( $ipn->getString( 'payment_status', '' ) === 'Refunded' ) {
				// We need to subtract data.previous_attributes.amount_refunded because if this is a second refund then stripe reports back the TOTAL refunded and not the refunded amount of this instance
				// e.g. refund 1 = -10, IPN reports -10, refund 2 = -10, IPN reports = -20 !!
				$ipn->set( 'mc_gross', ( - ( ( $eventObject->getFloat( 'amount_refunded', 0 ) - $eventData->getFloat( 'data.previous_attributes.amount_refunded', 0 ) ) / 100 ) ) );
			} elseif ( $ipn->getString( 'payment_status', '' ) === 'Reversed' ) {
				$ipn->set( 'mc_gross', ( - ( $eventObject->getFloat( 'amount', 0 ) / 100 ) ) );
			}

			$valid										=	$this->_validateIPN( $ipn, $paymentBasket, $eventBody, $signingSecret );

			if ( $valid === true ) {
				$this->_bindNotificationToBasket( $ipn, $paymentBasket );

				$ipn->setRawResult( 'SUCCESS' );

				$autorecurring_type						=	( in_array( $ipn->getString( 'txn_type', '' ), [ 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed' ], true ) ? 2 : 0 );
				$autorenew_type							=	( $autorecurring_type ? ( ( ( (int) $this->getAccountParam( 'enabled', 0 ) === 3 ) && ( $paymentBasket->isAnyAutoRecurring() === 2 ) ) ? 1 : 2 ) : 0 );

				if ( $autorecurring_type && ( $ipn->getString( 'txn_type', '' ) === 'subscr_signup' ) && ( ( $paymentBasket->getString( 'period1', '' ) ) && ( $paymentBasket->getFloat( 'mc_amount1', 0 ) === 0 ) ) && ( $ipn->getString( 'payment_status', '' ) === '' ) ) {
					$ipn->set( 'payment_status', 'Completed' );
				}

				if ( ( $ipn->getString( 'payment_status', '' ) === 'Refunded' ) && ( $paymentBasket->getFloat( 'mc_gross', 0 ) !== ( - $ipn->getFloat( 'mc_gross', 0 ) ) ) ) {
					$ipn->set( 'payment_status', 'Partially-Refunded' );
				}

				if ( in_array( $ipn->getString( 'txn_type', '' ), [ 'subscr_eot', 'subscr_cancel', 'subscr_failed' ], true ) ) {
					$autorecurring_type					=	0;
				}

				$this->updatePaymentStatus( $paymentBasket, $ipn->getString( 'txn_type', '' ), $ipn->getString( 'payment_status', '' ), $ipn, 1, $autorecurring_type, $autorenew_type, ( ! $orgTransactionId ? 'firstpayment' : false ) );

				echo 'webhook process: success';

				// If subscription recurring limit was hit or subscription is being cancelled by CBSubs then we need to cancel it at Stripe:
				if ( $subscriptionId ) {
					$recurringLimit						=	$paymentBasket->getInt( 'recur_times', 0 );
					$stopResults						=	null;

					if ( $disputeStatus === 'Lost_Reversal' ) {
						$stopResults					=	$this->cancelStripeSubscription( $subscriptionId );
					} elseif ( $paymentStatus === 'Reversed' ) {
						// Recurring payment is being disputed so pause recurring payment collection:
						$stopResults					=	$this->httpsRequestStripe( 'post', '/subscriptions/' . $subscriptionId, [ 'pause_collection[behavior]' => 'void' ] );
					} elseif ( $paymentStatus === 'Canceled_Reversal' ) {
						// Recurring payment dispute was won so resume recurring payment collection:
						$stopResults					=	$this->httpsRequestStripe( 'post', '/subscriptions/' . $subscriptionId, [ 'pause_collection' => '' ] );
					} elseif ( ( ( $paymentStatus === 'Completed' ) && $recurringLimit && ( $paymentBasket->getInt( 'recur_times_used', 0 ) >= $recurringLimit ) )
						 || ( ( $paymentStatus !== 'Unsubscribed' ) && ( $ipn->getString( 'txn_type', '' ) === 'subscr_cancel' ) )
					) {
						// Recurring payments cancelled or we've reached the recurring limit:
						if ( $eventObject->getBool( 'cancel_at_period_end', false ) || $eventObject->getBool( 'at_period_end', false ) ) {
							$stopImmediately			=	true;
						} else {
							$stopImmediately			=	false;
						}

						$stopResults					=	$this->cancelStripeSubscription( $subscriptionId, $stopImmediately );
					}

					if ( ( $stopResults !== null ) && $stopResults->has( 'error' ) && ( $stopResults->getString( 'error.code', '' ) !== '404' ) ) {
						$stopError						=	$stopResults->getString( 'error.message', '' );

						if ( ! $stopError ) {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription cancellation didn't return an error but didn't complete" ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						} else {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscriptions API error returned. ERROR: ' . $stopError . ' CODE: ' . $stopResults->getString( 'error.code', '' ), $stopError . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						}
					}
				}
			} else {
				echo 'webhook process: failed';

				$ipn->set( 'log_type', 'O' );

				$ipn->setRawResult( 'MISMATCH' );

				$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ' Stripe IPN fraud attempt. ERROR: ' . $valid, CBTxt::T( 'Invalid transaction.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}

			$ipn->store();
		} elseif ( $_CB_framework->getCfg( 'debug' ) ) {
			// If we're debugging keep track of all basket status and access attempts to help determine why the status check maybe failing:
			if ( $exists ) {
				$this->_setLogErrorMSG( 7, $paymentBasket, $this->getPayName() . ' Payment Status change from ' . $paymentBasket->getString( 'payment_status', '' ) . ' to ' . $paymentStatus . ' not allowed' . "\n" . '$orderId=' . $orderId . "\n" . '$invoiceId=' . $invoiceId . "\n" . '$intentId=' . $intentId . "\n" . '$subscriptionId=' . $subscriptionId . "\n" . '$transactionId=' . $transactionId . "\n" . '$orgTransactionId=' . $orgTransactionId . "\n" . '$_GET=' . var_export( $_GET, true ) . "\n" . '$_POST=' . var_export( $eventData->asArray(), true ) . "\n", null );
			} else {
				$this->_setLogErrorMSG( 7, null, $this->getPayName() . ' Payment Basket missing' . "\n" . '$orderId=' . $orderId . "\n" . '$invoiceId=' . $invoiceId . "\n" . '$intentId=' . $intentId . "\n" . '$subscriptionId=' . $subscriptionId . "\n" . '$transactionId=' . $transactionId . "\n" . '$_GET=' . var_export( $_GET, true ) . "\n" . '$_POST=' . var_export( $eventData->asArray(), true ) . "\n", null );
			}

			echo 'webhook skipped: payment status change not permitted';
		} else {
			echo 'webhook skipped: nothing to do';
		}

		return null;
	}

	/**
	 * Maps payment handler payment status to standard cpay status
	 *
	 * @param  string    $paymentStatus     payment handler payment status
	 * @return string                       standard cpay status: Completed, Processed, Denied, Pending, Unknown
	 */
	protected function mapPaymentStatus( $paymentStatus ): string
	{
		if ( $paymentStatus === 'Canceled_Reversal' ) {
			return 'Processed';
		}

		return parent::mapPaymentStatus( $paymentStatus );
	}

	/**
	 * PRIVATE METHODS OF THIS CLASS
	 */

	/**
	 * Checks if a Stripe API is set
	 *
	 * @return bool
	 */
	private function hasStripeApi(): bool
	{
		return ( $this->getAccountParam( 'stripe_pk', '' ) && $this->getAccountParam( 'stripe_sk', '' ) );
	}

	/**
	 * Checks if this gateway is in test mode and test clocks are enabled
	 *
	 * @return bool
	 */
	private function hasStripeTestClocks(): bool
	{
		return ( ( (int) $this->getAccountParam( 'normal_gateway', 1 ) === 0 ) && $this->getAccountParam( 'stripe_test_clocks', 0 ) );
	}

	/**
	 * Returns a list of webhooks being listened to
	 *
	 * @return string[]
	 */
	private function getStripeWebhooks(): array
	{
		return	[	'charge.refunded', 'charge.succeeded', 'charge.failed', 'charge.dispute.created',
					'charge.dispute.closed', 'payment_intent.succeeded', 'payment_intent.payment_failed',
					'checkout.session.completed', 'invoice.payment_succeeded', 'invoice.payment_failed',
					'customer.subscription.deleted',
				];
	}

	/**
	 * Returns the stripe customer id for a basket user
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @return string
	 */
	private function getStripeCustomerID( cbpaidPaymentBasket $paymentBasket ): string
	{
		// https://stripe.com/docs/api#customers
		/** @var Registry[] $cache */
		static $cache						=	[];

		$customer							=	new cbpaidConfig();

		$customer->load( [ 'user_id' => $paymentBasket->getInt( 'user_id', 0 ), 'type' => 'stripe' ] );

		$customerId							=	$customer->getParam( 'customer_id', '', 'params', GetterInterface::STRING );

		if ( $customerId ) {
			if ( ! array_key_exists( $customerId, $cache ) ) {
				// Check if the stored customer id still exists and caching the results as we don't need to keep re-verifying the same id exists:
				$results					=	$this->httpsRequestStripe( 'get', '/customers/' . $customerId );

				if ( $results->has( 'error' ) ) {
					$error					=	$results->getString( 'error.message', '' );

					if ( ! $error ) {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Customers API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted customer request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					} else {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Customers API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}

					$cache[$customerId]		=	false;
				} elseif ( $results->getBool( 'deleted', false ) ) {
					$cache[$customerId]		=	false;
				} else {
					$cache[$customerId]		=	$results;
				}
			}

			if ( ! $cache[$customerId] ) {
				// The stored customer id does not exist so lets clear it and create a new one:
				$customerId					=	'';
			}
		}

		// Create a new customer if one doesn't exist:
		if ( ! $customerId ) {
			$requestParams								=	[	'email'					=>	$paymentBasket->getString( 'payer_email', '' ),
																'name'					=>	trim( $paymentBasket->getString( 'first_name', '' ) . ' ' . $paymentBasket->getString( 'last_name', '' ) ),
																'phone'					=>	$paymentBasket->getString( 'contact_phone', '' ),
																'metadata[user_id]'		=>	$paymentBasket->getInt( 'user_id', 0 ),
															];

			if ( $paymentBasket->getString( 'address_street', '' ) ) {
				// Line 1 is required if address is provided so skip adding address if line 1 doesn't have a value:
				$address								=	[	'address[line1]'		=>	$paymentBasket->getString( 'address_street', '' ),
																'address[country]'		=>	$paymentBasket->getInvoiceCountry( 2 ),
																'address[state]'		=>	$paymentBasket->getInvoiceState(),
																'address[city]'			=>	$paymentBasket->getString( 'address_city', '' ),
																'address[postal_code]'	=>	$paymentBasket->getString( 'address_zip', '' ),
															];

				$requestParams							=	array_merge( $requestParams, $address );
			}

			$language									=	$this->getStripeLocale( $paymentBasket->getInt( 'user_id', 0 ) );

			if ( $language ) {
				$requestParams['preferred_locales[0]']	=	$language;
			}

			$results									=	$this->httpsRequestStripe( 'post', '/customers', $requestParams );

			if ( $results->has( 'error' ) ) {
				$error									=	$results->getString( 'error.message', '' );

				if ( ! $error ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Customers API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted customer request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				} else {
					$this->_setLogErrorMSG( 5, null, $this->getPayName() . ' Stripe Customers API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
				}

				return '';
			}

			$customerId									=	$results->getString( 'id', '' );

			$cache[$customerId]							=	$results;

			$customer->set( 'user_id', $paymentBasket->getInt( 'user_id', 0 ) );
			$customer->set( 'type', 'stripe' );
			$customer->set( 'last_updated_date', Application::Database()->getUtcDateTime() );

			$customer->setParam( 'customer_id', $customerId );
			$customer->storeParams();

			$customer->store();
		} else {
			// Check if any customer information needs to be updated:
			$customer									=	$cache[$customerId];
			$requestParams								=	[];

			if ( $customer->getString( 'email', '' ) !== $paymentBasket->getString( 'payer_email', '' ) ) {
				$requestParams['email']					=	$paymentBasket->getString( 'payer_email', '' );
			}

			$name										=	trim( $paymentBasket->getString( 'first_name', '' ) . ' ' . $paymentBasket->getString( 'last_name', '' ) );

			if ( $customer->getString( 'name', '' ) !== $name ) {
				$requestParams['name']					=	$name;
			}

			if ( $customer->getString( 'phone', '' ) !== $paymentBasket->getString( 'contact_phone', '' ) ) {
				$requestParams['phone']					=	$paymentBasket->getString( 'contact_phone', '' );
			}

			if ( $customer->getString( 'address.line1', '' ) !== $paymentBasket->getString( 'address_street', '' ) ) {
				$requestParams['address[line1]']		=	$paymentBasket->getString( 'address_street', '' );
			}

			$country									=	$paymentBasket->getInvoiceCountry( 2 );

			if ( $customer->getString( 'address.country', '' ) !== $country ) {
				$requestParams['address[country]']		=	$country;
			}

			$state										=	$paymentBasket->getInvoiceState();

			if ( $customer->getString( 'address.state', '' ) !== $state ) {
				$requestParams['address[state]']		=	$state;
			}

			if ( $customer->getString( 'address.city', '' ) !== $paymentBasket->getString( 'address_city', '' ) ) {
				$requestParams['address[city]']			=	$paymentBasket->getString( 'address_city', '' );
			}

			if ( $customer->getString( 'address.postal_code', '' ) !== $paymentBasket->getString( 'address_zip', '' ) ) {
				$requestParams['address[postal_code]']	=	$paymentBasket->getString( 'address_zip', '' );
			}

			if ( $requestParams ) {
				$results								=	$this->httpsRequestStripe( 'post', '/customers/' . $customerId, $requestParams );

				if ( $results->has( 'error' ) ) {
					// Log the error, but continue as this won't break payments but the customer update did fail for some reason:
					$error								=	$results->getString( 'error.message', '' );

					if ( ! $error ) {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Customers API HTTPS POST request to payment gateway server failed.', null );
					} else {
						$this->_setLogErrorMSG( 5, null, $this->getPayName() . ' Stripe Customers API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
					}
				} else {
					$cache[$customerId]					=	$results;
				}
			}
		}

		return $customerId;
	}

	/**
	 * Returns the customer portal url for the supplied customer id
	 *
	 * @param string        $customerId
	 * @param null|Registry $results The results of a customer API request
	 * @return string
	 */
	private function getStripeCustomerPortalURL( string $customerId, ?Registry &$results = null ): string
	{
		global $_CB_framework;

		if ( ! $customerId ) {
			return '';
		}

		static $cache			=	[];

		if ( array_key_exists( $customerId, $cache ) ) {
			$results			=	$cache[$customerId];

			return $cache[$customerId]->getString( 'url', '' );
		}

		// https://stripe.com/docs/api/customer_portal
		$requestParams			=	[	'customer'		=>	$customerId,
										'return_url'	=>	$_CB_framework->pluginClassUrl( [ 'plugin' => 'cbpaidsubscriptions', 'do' => 'displayplans' ], false ),
									];
		$results				=	$this->httpsRequestStripe( 'post', '/billing_portal/sessions', $requestParams );

		$cache[$customerId]		=	$results;

		return $cache[$customerId]->getString( 'url', '' );
	}

	/**
	 * Returns the stripe product id for a basket
	 * This is based off the plans in the basket and will attempt product reuse at Stripe
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param null|Registry       $results        The results of a product API request
	 * @return string
	 */
	private function getStripeProductID( cbpaidPaymentBasket $paymentBasket, ?Registry &$results = null ): string
	{
		static $cache			=	[];

		// https://stripe.com/docs/api/products
		// https://stripe.com/docs/api/products/create
		$planIds				=	[];
		$planNames				=	[];

		foreach ( $paymentBasket->loadPaymentItems() as $item ) {
			$planIds[]			=	$item->getInt( 'plan_id', 0 );
			$planNames[]		=	$item->getPlanParam( 'alias', '', null, GetterInterface::STRING );
		}

		$id						=	md5( implode( '', $planIds ) );

		if ( array_key_exists( $id, $cache ) ) {
			$results			=	$cache[$id];

			return $cache[$id]->getString( 'id', '' );
		}

		$results				=	$this->httpsRequestStripe( 'get', '/products/' . $id );

		if ( $results->has( 'error' ) ) {
			$requestParams		=	[	'id'				=>	$id,
										'name'				=>	cbIsoUtf_substr( implode( ', ', $planNames ), 0, 250 ),
										'metadata[plans]'	=>	implode( ', ', $planIds ),
									];
			$results			=	$this->httpsRequestStripe( 'post', '/products', $requestParams );

			if ( $results->has( 'error' ) ) {
				$error			=	$results->getString( 'error.message', '' );

				if ( ! $error ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Products API HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted plan request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				} else {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Products API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), $error . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				}
			}
		}

		$cache[$id]				=	$results;

		return $cache[$id]->getString( 'id', '' );
	}

	/**
	 * Returns subscription object details
	 *
	 * @param string $subscriptionId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeSubscription( string $subscriptionId, array $expand = [ 'latest_invoice' ] ): Registry
	{
		if ( ! $subscriptionId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$subscriptionId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/subscriptions/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/subscriptions/' . $subscriptionId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Cancels a stripe subscription
	 *
	 * @param string $subscriptionId
	 * @param bool   $immediately
	 * @return Registry
	 */
	private function cancelStripeSubscription( string $subscriptionId, bool $immediately = true ): Registry
	{
		if ( ! $subscriptionId ) {
			return new Registry();
		}

		static $cache				=	[];

		if ( array_key_exists( $subscriptionId, $cache ) ) {
			return $cache[$subscriptionId];
		}

		// First check if the subscription even exists and if it's in a status we can cancel:
		$results					=	$this->getStripeSubscription( $subscriptionId );

		if ( $results->getString( 'id', '' ) && ( $results->getString( 'status', '' ) !== 'canceled' ) ) {
			$canceled				=	false;

			if ( $immediately ) {
				// If there's a test clock then this is a test subscription so cancel the entire chain (schedule and subscription)
				if ( $results->getString( 'test_clock', '' ) ) {
					$results		=	$this->cancelStripeTestClock( $results->getString( 'test_clock', '' ) );

					// Don't let test clock errors stop the process we'll try to just cancel the schedule or subscription directly, but lets still log the error
					if ( $results->has( 'error' ) ) {
						$error		=	$results->getString( 'error.message', '' );

						if ( ! $error ) {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Test Clocks API HTTPS POST request to payment gateway server failed.', null );
						} else {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Test Clocks API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
						}
					} else {
						$canceled	=	true;
					}
				}

				// If there's a subscription schedule attached lets try to cancel it which will also cancel the subscription
				if ( ( ! $canceled ) && $results->getString( 'schedule', '' ) ) {
					$results		=	$this->cancelStripeSubscriptionSchedule( $results->getString( 'schedule', '' ) );

					// Don't let scheduler errors stop the process we'll try to just cancel the subscription directly, but lets still log the error
					if ( $results->has( 'error' ) ) {
						$error		=	$results->getString( 'error.message', '' );

						if ( ! $error ) {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscription Schedules API HTTPS POST request to payment gateway server failed.', null );
						} else {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Subscription Schedules API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
						}
					} else {
						$canceled	=	( $results->getString( 'status', '' ) === 'canceled' );
					}
				}
			}

			// Finally lets just directly cancel the subscription itself
			if ( ! $canceled ) {
				// https://stripe.com/docs/api/subscriptions/cancel
				if ( $immediately ) {
					$results		=	$this->httpsRequestStripe( 'delete', '/subscriptions/' . $subscriptionId );
				} else {
					$results		=	$this->httpsRequestStripe( 'post', '/subscriptions/' . $subscriptionId, [ 'cancel_at_period_end' => true ] );
				}
			}
		}

		$cache[$subscriptionId]		=	$results;

		return $cache[$subscriptionId];
	}

	/**
	 * Returns test clock object details
	 *
	 * @param string $clockId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeTestClock( string $clockId, array $expand = [] ): Registry
	{
		if ( ! $clockId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$clockId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/test_clocks/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/test_helpers/test_clocks/' . $clockId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Cancels a stripe test clock
	 *
	 * @param string $clockId
	 * @return Registry
	 */
	private function cancelStripeTestClock( string $clockId ): Registry
	{
		if ( ! $clockId ) {
			return new Registry();
		}

		static $cache		=	[];

		if ( array_key_exists( $clockId, $cache ) ) {
			return $cache[$clockId];
		}

		// First check if the test clock even exists and if it's in a status we can cancel:
		$results			=	$this->getStripeTestClock( $clockId );

		if ( $results->getString( 'id', '' ) ) {
			// https://stripe.com/docs/api/test_clocks/delete
			$results		=	$this->httpsRequestStripe( 'delete', '/test_helpers/test_clocks/' . $clockId );
		}

		$cache[$clockId]	=	$results;

		return $cache[$clockId];
	}

	/**
	 * Returns a subscription schedule object details
	 *
	 * @param string $scheduleId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeSubscriptionSchedule( string $scheduleId, array $expand = [] ): Registry
	{
		if ( ! $scheduleId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$scheduleId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/subscription_schedules/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/subscription_schedules/' . $scheduleId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Returns a subscription schedule object details
	 *
	 * @param string $scheduleId
	 * @return Registry
	 */
	private function cancelStripeSubscriptionSchedule( string $scheduleId ): Registry
	{
		if ( ! $scheduleId ) {
			return new Registry();
		}

		static $cache		=	[];

		if ( array_key_exists( $scheduleId, $cache ) ) {
			return $cache[$scheduleId];
		}

		// First check if the subscription schedule even exists and if it's in a status we can cancel:
		$results			=	$this->getStripeSubscriptionSchedule( $scheduleId );

		if ( in_array( $results->getString( 'status', '' ), [ 'active', 'not_started' ], true ) ) {
			// https://stripe.com/docs/api/subscription_schedules/cancel
			$results		=	$this->httpsRequestStripe( 'post', '/subscription_schedules/' . $scheduleId . '/cancel' );
		}

		$cache[$scheduleId]	=	$results;

		return $cache[$scheduleId];
	}

	/**
	 * Returns a charge object details
	 *
	 * @param string $chargeId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeCharge( string $chargeId, array $expand = [] ): Registry
	{
		if ( ! $chargeId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$chargeId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/charges/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/charges/' . $chargeId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Returns a invoice object details
	 *
	 * @param string $invoiceId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeInvoice( string $invoiceId, array $expand = [] ): Registry
	{
		if ( ! $invoiceId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$invoiceId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/invoices/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/invoices/' . $invoiceId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Returns a Payment Intents object details
	 *
	 * @param string $intentId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripePaymentIntent( string $intentId, array $expand = [ 'latest_charge' ] ): Registry
	{
		if ( ! $intentId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$intentId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/payment_intents/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/payment_intents/' . $intentId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Returns a Setup Intents object details
	 *
	 * @param string $intentId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeSetupIntent( string $intentId, array $expand = [] ): Registry
	{
		if ( ! $intentId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$intentId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/setup_intents/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/setup_intents/' . $intentId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Create a payment intent secret id
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param bool                $recurring
	 * @return string
	 */
	private function getStripePaymentIntentSecret( cbpaidPaymentBasket $paymentBasket, bool $recurring = false ): string
	{
		static $cache			=	[];

		$basketId				=	$paymentBasket->getInt( 'id', 0 );

		if ( array_key_exists( $basketId, $cache ) ) {
			return $cache[$basketId];
		}

		$customerId				=	'';

		if ( $recurring || $this->getAccountParam( 'customer_account', 0 ) ) {
			$customerId			=	$this->getStripeCustomerID( $paymentBasket );

			if ( ! $customerId ) {
				// Customer create failed; lets abort:
				return '';
			}
		}


		$paymentMethods			=	$this->getAccountParam( ( $recurring ? 'card_element_recurring_filter_methods' : 'card_element_filter_methods' ), '' );

		if ( ( ! $paymentMethods ) && $recurring ) {
			// SetupIntents must have payment methods explicitly provided so fallback to atleast accepting card payments
			$paymentMethods		=	[ 'card' ];
		}

		if ( $paymentMethods ) {
			if ( ! is_array( $paymentMethods ) ) {
				$paymentMethods	=	explode( '|*|', $paymentMethods );
			}

			$paymentMethods		=	$this->validatePaymentMethods( $paymentMethods, $paymentBasket->getString( 'mc_currency', '' ), $recurring );
		}

		$requestParams			=	[	'description'					=>	$paymentBasket->getString( 'item_name', '' ),
										'metadata[user_id]'				=>	$paymentBasket->getInt( 'user_id', 0 ),
										'metadata[order_id]'			=>	$basketId,
										'metadata[invoice]'				=>	$paymentBasket->getString( 'invoice', '' ),
										'metadata[gateway]'				=>	(int) $this->getAccountParam( 'id', 0 ),
									];

		if ( $customerId ) {
			$requestParams['customer']									=	$customerId;
		}

		if ( $recurring ) {
			if ( ! $paymentMethods ) {
				// Payment methods are required for setup intents and we have none so abort
				return '';
			}

			foreach ( $paymentMethods as $i => $paymentMethod ) {
				$requestParams["payment_method_types[$i]"]				=	$paymentMethod;
			}

			$intentSecret		=	$this->getStripeSubscriptionPaymentIntentSecret( $paymentBasket, $requestParams );
		} else {
			if ( $paymentMethods ) {
				foreach ( $paymentMethods as $i => $paymentMethod ) {
					$requestParams["payment_method_types[$i]"]			=	$paymentMethod;
				}
			} else {
				$requestParams['automatic_payment_methods[enabled]']	=	'true';
			}

			$intentSecret		=	$this->getStripeSinglePaymentIntentSecret( $paymentBasket, $requestParams );
		}

		$cache[$basketId]		=	$intentSecret;

		return $cache[$basketId];
	}

	/**
	 * Create a PaymentIntent for instant billing
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param array               $requestParams
	 * @return string
	 */
	private function getStripeSinglePaymentIntentSecret( cbpaidPaymentBasket $paymentBasket, array $requestParams ): string
	{
		$requestParams['amount']	=	( sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) * 100 );
		$requestParams['currency']	=	strtolower( $paymentBasket->getString( 'mc_currency', '' ) );

		// Check if this request matches the existing request to see if we even need to make a call to stripe
		$requestHash				=	md5( json_encode( $requestParams ) );
		$paymentIntent				=	$paymentBasket->getParam( 'payment_intent.single', '', 'integrations', GetterInterface::STRING );

		if ( $paymentIntent ) {
			$intentSecret			=	$paymentBasket->getParam( 'payment_intent.client_secret', '', 'integrations', GetterInterface::STRING );

			if ( $intentSecret && ( $requestHash === $paymentBasket->getParam( 'payment_intent_hash', '', 'integrations', GetterInterface::STRING ) ) && in_array( $this->getStripePaymentIntent( $paymentIntent )->getString( 'status', '' ), [ 'requires_payment_method', 'requires_capture', 'requires_confirmation', 'requires_action' ], true ) ) {
				// Existing payment intent is a match and is still in a usable status so just reuse it instead of making a new one
				return $intentSecret;
			}

			// There's already an existing payment intent for this basket so lets get rid of it before generating a new one encase payment details changed (e.g. user went back and applied a coupon):
			$this->cancelStripePaymentIntent( $paymentIntent );
		}

		$results					=	$this->httpsRequestStripe( 'post', '/payment_intents', $requestParams );

		if ( $results->has( 'error' ) ) {
			$error					=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intents API HTTPS POST request to payment gateway server failed.', null );
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Payment Intents API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
			}

			$intentSecret			=	'';
		} else {
			$intentSecret			=	$results->getString( 'client_secret', '' );
		}

		// Attach the payment intent to the basket encase we need it later when processing the payment:
		$paymentBasket->setParam( 'payment_intent_hash', $requestHash, 'integrations' );
		$paymentBasket->setParam( 'payment_intent.single', $results->getString( 'id', '' ), 'integrations' );
		$paymentBasket->setParam( 'payment_intent.client_secret', $intentSecret, 'integrations' );

		$paymentBasket->storeParams( 'integrations' );

		$paymentBasket->store();

		return $intentSecret;
	}

	/**
	 * Create a SetupIntent for future billing
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param array               $requestParams
	 * @return string
	 */
	private function getStripeSubscriptionPaymentIntentSecret( cbpaidPaymentBasket $paymentBasket, array $requestParams ): string
	{
		// Create a SetupIntent for ALL recurring payments as we'll attach it later after we've confirmed their payment method
		// Check if this request matches the existing request to see if we even need to make a call to stripe
		$requestHash		=	md5( json_encode( $requestParams ) );
		$setupIntent		=	$paymentBasket->getParam( 'setup_intent.recurring', '', 'integrations', GetterInterface::STRING );

		if ( $setupIntent ) {
			$intentSecret	=	$paymentBasket->getParam( 'setup_intent.client_secret', '', 'integrations', GetterInterface::STRING );

			if ( $intentSecret && ( $requestHash === $paymentBasket->getParam( 'setup_intent_hash', '', 'integrations', GetterInterface::STRING ) ) && in_array( $this->getStripeSetupIntent( $setupIntent )->getString( 'status', '' ), [ 'requires_payment_method', 'requires_capture', 'requires_confirmation', 'requires_action' ], true ) ) {
				// Existing setup intent is a match and is still in a usable status so just reuse it instead of making a new one
				return $intentSecret;
			}

			// There's already an existing setup intent for this basket so lets get rid of it before generating a new one encase payment details changed (e.g. user went back and applied a coupon):
			$this->cancelStripeSetupIntent( $setupIntent );
		}

		$results			=	$this->httpsRequestStripe( 'post', '/setup_intents', $requestParams );

		if ( $results->has( 'error' ) ) {
			$error			=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intents API HTTPS POST request to payment gateway server failed.', null );
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Setup Intents API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
			}

			$intentSecret	=	'';
		} else {
			$intentSecret	=	$results->getString( 'client_secret', '' );
		}

		// Attach the setup intent to the basket encase we need it later when processing the payment:
		$paymentBasket->setParam( 'setup_intent_hash', $requestHash, 'integrations' );
		$paymentBasket->setParam( 'setup_intent.recurring', $results->getString( 'id', '' ), 'integrations' );
		$paymentBasket->setParam( 'setup_intent.client_secret', $intentSecret, 'integrations' );

		$paymentBasket->storeParams( 'integrations' );

		$paymentBasket->store();

		return $intentSecret;
	}

	/**
	 * Cancels a payment intent
	 *
	 * @param string $intentId
	 * @return Registry
	 */
	private function cancelStripePaymentIntent( string $intentId ): Registry
	{
		if ( ! $intentId ) {
			return new Registry();
		}

		static $cache		=	[];

		if ( array_key_exists( $intentId, $cache ) ) {
			return $cache[$intentId];
		}

		// First check if the payment intent even exists and if it's in a status we can cancel:
		$results			=	$this->getStripePaymentIntent( $intentId );

		if ( in_array( $results->getString( 'status', '' ), [ 'requires_payment_method', 'requires_capture', 'requires_confirmation', 'requires_action' ], true ) ) {
			// https://stripe.com/docs/api/payment_intents/cancel
			$results		=	$this->httpsRequestStripe( 'post', '/payment_intents/' . $intentId . '/cancel' );
		}

		$cache[$intentId]	=	$results;

		return $cache[$intentId];
	}

	/**
	 * Cancels a setup intent
	 *
	 * @param string $intentId
	 * @return Registry
	 */
	private function cancelStripeSetupIntent( string $intentId ): Registry
	{
		if ( ! $intentId ) {
			return new Registry();
		}

		static $cache		=	[];

		if ( array_key_exists( $intentId, $cache ) ) {
			return $cache[$intentId];
		}

		// First check if the setup intent even exists and if it's in a status we can cancel:
		$results			=	$this->getStripeSetupIntent( $intentId );

		if ( in_array( $results->getString( 'status', '' ), [ 'requires_payment_method', 'requires_capture', 'requires_confirmation', 'requires_action' ], true ) ) {
			// https://stripe.com/docs/api/setup_intents/cancel
			$results		=	$this->httpsRequestStripe( 'post', '/setup_intents/' . $intentId . '/cancel' );
		}

		$cache[$intentId]	=	$results;

		return $cache[$intentId];
	}

	/**
	 * Returns a Checkout Session object details
	 *
	 * @param string $sessionId
	 * @param array  $expand
	 * @return Registry
	 */
	private function getStripeCheckoutSession( string $sessionId, array $expand = [] ): Registry
	{
		if ( ! $sessionId ) {
			return new Registry();
		}

		static $cache		=	[];

		$cacheId			=	$sessionId . ( $expand ? ':' . implode( ':', $expand ) : '' );

		if ( array_key_exists( $cacheId, $cache ) ) {
			return $cache[$cacheId];
		}

		// https://stripe.com/docs/api/checkout/sessions/retrieve
		$cache[$cacheId]	=	$this->httpsRequestStripe( 'get', '/checkout/sessions/' . $sessionId, ( $expand ? [ 'expand' => $expand ] : [] ) );

		return $cache[$cacheId];
	}

	/**
	 * Create a hosted checkout session url
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param bool                $recurring
	 * @return string
	 */
	private function getStripeCheckoutSessionURL( cbpaidPaymentBasket $paymentBasket, bool $recurring = false ): string
	{
		static $cache					=	[];

		$basketId						=	$paymentBasket->getInt( 'id', 0 );
		$cacheId						=	$basketId . ':' . (int) $recurring;

		if ( isset( $cache[$cacheId] ) ) {
			return $cache[$cacheId];
		}

		$customerId						=	'';

		if ( $recurring || $this->getAccountParam( 'customer_account_checkout', 0 ) ) {
			$customerId					=	$this->getStripeCustomerID( $paymentBasket );

			if ( ! $customerId ) {
				// Customer create failed; lets abort:
				return '';
			}
		}

		$requestParams					=	[	'cancel_url'				=>	$this->getStripeCancelURL( $paymentBasket ),
												'client_reference_id'		=>	$basketId,
												'metadata[user_id]'			=>	$paymentBasket->getInt( 'user_id', 0 ),
												'metadata[order_id]'		=>	$basketId,
												'metadata[invoice]'			=>	$paymentBasket->getString( 'invoice', '' ),
												'metadata[gateway]'			=>	(int) $this->getAccountParam( 'id', 0 ),
											];

		if ( $customerId ) {
			$requestParams['customer']										=	$customerId;
		}

		$paymentMethods					=	$this->getAccountParam( ( $recurring ? 'card_checkout_recurring_filter_methods' : 'card_checkout_filter_methods' ), '' );

		if ( $paymentMethods ) {
			if ( ! \is_array( $paymentMethods ) ) {
				$paymentMethods			=	explode( '|*|', $paymentMethods );
			}

			$paymentMethods				=	$this->validatePaymentMethods( $paymentMethods, $paymentBasket->getString( 'mc_currency', '' ), $recurring, true );
		}

		if ( $paymentMethods ) {
			foreach ( $paymentMethods as $i => $paymentMethod ) {
				if ( ( $paymentMethod === 'bacs_debit' ) && ( ! $recurring ) ) {
					$requestParams['payment_intent_data[setup_future_usage]']	=	'off_session';
				}

				$requestParams["payment_method_types[$i]"]						=	$paymentMethod;
			}
		}

		$language						=	$this->getStripeLocale( $paymentBasket->getInt( 'user_id', 0 ) );

		// https://stripe.com/docs/payments/checkout/server
		// https://stripe.com/docs/api/checkout/sessions/create
		if ( $recurring ) {
			$recurringParams			=	$this->getStripeCheckoutSubscriptionSession( $paymentBasket );

			if ( ! $recurringParams ) {
				return '';
			}

			$requestParams				=	array_merge( $requestParams, $recurringParams );
		} else {
			$paymentParams				=	$this->getStripeCheckoutSingleSession( $paymentBasket );

			if ( ! $paymentParams ) {
				return '';
			}

			$requestParams				=	array_merge( $requestParams, $paymentParams );
		}

		if ( $language ) {
			$requestParams['locale']	=	$language;
		}

		// Check if this request matches the existing request to see if we even need to make a call to stripe
		$requestHash					=	md5( json_encode( $requestParams ) );
		$checkoutSession				=	$paymentBasket->getParam( 'checkout_session.' . ( $recurring ? 'recurring' : 'single' ), '', 'integrations', GetterInterface::STRING );

		if ( $checkoutSession ) {
			$checkoutUrl				=	$paymentBasket->getParam( 'checkout_url.' . ( $recurring ? 'recurring' : 'single' ), '', 'integrations', GetterInterface::STRING );

			if ( $checkoutUrl && ( $requestHash === $paymentBasket->getParam( 'checkout_hash', '', 'integrations', GetterInterface::STRING ) ) && ( $this->getStripeCheckoutSession( $checkoutSession )->getString( 'status', '' ) === 'open' ) ) {
				// Existing checkout session is a match and is still in a usable status so just reuse it instead of making a new one
				$cache[$cacheId]		=	$checkoutUrl;

				return $cache[$cacheId];
			}

			// There's already an existing checkout session for this basket and it doesn't match the existing hash meaning the request has changed so cancel the old session:
			$this->cancelStripeCheckoutSession( $checkoutSession );
		}

		$results						=	$this->httpsRequestStripe( 'post', '/checkout/sessions', $requestParams );

		if ( $results->has( 'error' ) ) {
			$error						=	$results->getString( 'error.message', '' );

			if ( ! $error ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API HTTPS POST request to payment gateway server failed.', null );
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Stripe Checkout API error returned. ERROR: ' . $error . ' CODE: ' . $results->getString( 'error.code', '' ), null );
			}

			$checkoutUrl				=	'';
		} else {
			$checkoutUrl				=	$results->getString( 'url', '' );
		}

		// Attach the checkout session id, payment intent id, and setup intent id so we can use them later since Stripe PDT doesn't return any payment data:
		$paymentBasket->setParam( 'checkout_hash', $requestHash, 'integrations' );
		$paymentBasket->setParam( 'checkout_session.' . ( $recurring ? 'recurring' : 'single' ), $results->getString( 'id', '' ), 'integrations' );
		$paymentBasket->setParam( 'checkout_url.' . ( $recurring ? 'recurring' : 'single' ), $checkoutUrl, 'integrations' );
		$paymentBasket->setParam( 'payment_intent.' . ( $recurring ? 'recurring' : 'single' ), $results->getString( 'payment_intent', '' ), 'integrations' );
		$paymentBasket->setParam( 'setup_intent.' . ( $recurring ? 'recurring' : 'single' ), $results->getString( 'setup_intent', '' ), 'integrations' );

		$paymentBasket->storeParams( 'integrations' );

		$paymentBasket->store();

		$cache[$cacheId]				=	$checkoutUrl;

		return $cache[$cacheId];
	}

	/**
	 * Returns request parameters for checkout single payment session
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @return array
	 */
	private function getStripeCheckoutSingleSession( cbpaidPaymentBasket $paymentBasket ): array
	{
		if ( $paymentBasket->getFloat( 'mc_amount3', 0 ) ) {
			if ( $paymentBasket->getString( 'period1', '' ) ) {
				$amount		=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount1', 0 ) );
			} else {
				$amount		=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount3', 0 ) );
			}
		} else {
			$amount			=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) );
		}

		return [	'mode'											=>	'payment',
					'success_url'									=>	$this->getStripeReturnURL( $paymentBasket, false, true ),
					'line_items[0][price_data][currency]'			=>	strtolower( $paymentBasket->getString( 'mc_currency', '' ) ),
					'line_items[0][price_data][product_data][name]'	=>	$paymentBasket->getString( 'item_name', '' ),
					'line_items[0][price_data][unit_amount]'		=>	( $amount * 100 ),
					'line_items[0][quantity]'						=>	1,
					'payment_intent_data[metadata][user_id]'		=>	$paymentBasket->getInt( 'user_id', 0 ),
					'payment_intent_data[metadata][order_id]'		=>	$paymentBasket->getInt( 'id', 0 ),
					'payment_intent_data[metadata][invoice]'		=>	$paymentBasket->getString( 'invoice', '' ),
					'payment_intent_data[metadata][gateway]'		=>	(int) $this->getAccountParam( 'id', 0 ),
				];
	}

	/**
	 * Returns request parameters for checkout recurring payment session
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @return array
	 */
	private function getStripeCheckoutSubscriptionSession( cbpaidPaymentBasket $paymentBasket ): array
	{
		$recurringPeriod				=	$paymentBasket->getString( 'period3', '' );
		$recurringPeriodLimits			=	$this->getStripePeriodLimits( explode( ' ', $recurringPeriod ) );
		$recurringAmount				=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount3', 0 ) );

		if ( $recurringPeriodLimits['error'] ) {
			// Period limits are not valid for a recurring subscription at stripe so abort creating the subscription checkout session:
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Invalid plans duration', $recurringPeriodLimits['error'] );

			return [];
		}

		$requestParams					=	[	'mode'													=>	'subscription',
												'success_url'											=>	$this->getStripeReturnURL( $paymentBasket, true, true ),
												'line_items[0][price_data][currency]'					=>	strtolower( $paymentBasket->getString( 'mc_currency', '' ) ),
												'line_items[0][price_data][product_data][name]'			=>	$paymentBasket->getString( 'item_name', '' ),
												'line_items[0][price_data][recurring][interval]'		=>	$recurringPeriodLimits['interval'],
												'line_items[0][price_data][recurring][interval_count]'	=>	$recurringPeriodLimits['interval_count'],
												'line_items[0][price_data][unit_amount]'				=>	( $recurringAmount * 100 ),
												'line_items[0][quantity]'								=>	1,
												'subscription_data[metadata][user_id]'					=>	$paymentBasket->getInt( 'user_id', 0 ),
												'subscription_data[metadata][order_id]'					=>	$paymentBasket->getInt( 'id', 0 ),
												'subscription_data[metadata][invoice]'					=>	$paymentBasket->getString( 'invoice', '' ),
												'subscription_data[metadata][gateway]'					=>	(int) $this->getAccountParam( 'id', 0 ),
											];

		$initialPeriod					=	$paymentBasket->getString( 'period1', '' );

		if ( $initialPeriod ) {
			$initialAmount				=	sprintf( '%.2f', $paymentBasket->getFloat( 'mc_amount1', 0 ) );

			if ( ( $recurringAmount !== $initialAmount ) || ( $recurringPeriod !== $initialPeriod ) ) {
				if ( $initialAmount > 0 ) {
					if ( ( $initialAmount > $recurringAmount ) && ( $recurringPeriod === $initialPeriod ) ) {
						// The initial price is greater than the recurring price, but the duration is the same so lets take the difference between recurring and initial price and add it as an initial charge
						$requestParams['line_items[1][price_data][currency]']							=	strtolower( $paymentBasket->getString( 'mc_currency', '' ) );
						$requestParams['line_items[1][price_data][product_data][name]']					=	$paymentBasket->getString( 'item_name', '' );
						$requestParams['line_items[1][price_data][unit_amount]']						=	( ( $initialAmount - $recurringAmount ) * 100 );
						$requestParams['line_items[1][quantity]']										=	1;
					} else {
						// The initial price is lower than recurring or duration are different so the only option is to add a separate charge and push the recurring subscription using trial_end
						$requestParams['line_items[1][price_data][currency]']							=	strtolower( $paymentBasket->getString( 'mc_currency', '' ) );
						$requestParams['line_items[1][price_data][product_data][name]']					=	$paymentBasket->getString( 'item_name', '' );
						$requestParams['line_items[1][price_data][unit_amount]']						=	( $initialAmount * 100 );
						$requestParams['line_items[1][quantity]']										=	1;
						$requestParams['subscription_data[trial_end]']									=	$recurringPeriodLimits['start'];
					}
				} else {
					// If the initial price is $0 then it's just a free trial so we can just add trial_end and be done
					$requestParams['subscription_data[trial_end]']										=	$recurringPeriodLimits['start'];
				}
			}

		}

		return $requestParams;
	}

	/**
	 * Cancels a checkout session
	 *
	 * @param string $sessionId
	 * @return Registry
	 */
	private function cancelStripeCheckoutSession( string $sessionId ): Registry
	{
		if ( ! $sessionId ) {
			return new Registry();
		}

		static $cache		=	[];

		if ( array_key_exists( $sessionId, $cache ) ) {
			return $cache[$sessionId];
		}

		// First check if the checkout session even exists and if it's in a status we can cancel:
		$results			=	$this->getStripeCheckoutSession( $sessionId );

		if ( $results->getString( 'status', '' ) === 'open' ) {
			// https://stripe.com/docs/api/checkout/sessions/expire
			$results		=	$this->httpsRequestStripe( 'post', '/checkout/sessions/' . $sessionId . '/expire' );
		}

		$cache[$sessionId]	=	$results;

		return $cache[$sessionId];
	}

	/**
	 * This attempts to find the metadata for an object and set it into the object if it's nested
	 * This is necessary since sometimes Stripe chains objects together and some of those objects don't directly store the metadata
	 *
	 * @param Registry $stripeObject
	 * @return void
	 */
	private function setStripeMetadata( Registry $stripeObject ): void
	{
		if ( $stripeObject->getInt( 'metadata.order_id', 0 ) ) {
			return;
		}

		switch ( $stripeObject->getString( 'object', '' ) ) {
			case 'invoice';
				if ( $stripeObject->getInt( 'lines.data.0.metadata.order_id', 0 ) ) {
					$stripeObject->set( 'metadata', $stripeObject->subTree( 'lines.data.0.metadata' )->asArray() );
					return;
				}

				if ( $stripeObject->getString( 'subscription', '' ) ) {
					$results	=	$this->getStripeSubscription( $stripeObject->getString( 'subscription', '' ), [ 'schedule' ] );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'schedule.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'schedule.metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'schedule.phases.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'schedule.phases.0.metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'items.data.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'items.data.0.metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'payment_intent', '' ) ) {
					$results	=	$this->getStripePaymentIntent( $stripeObject->getString( 'payment_intent', '' ), [] );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'charge', '' ) ) {
					$results	=	$this->getStripeCharge( $stripeObject->getString( 'charge', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}
				}
				break;
			case 'subscription';
				if ( $stripeObject->getInt( 'items.data.0.metadata.order_id', 0 ) ) {
					$stripeObject->set( 'metadata', $stripeObject->subTree( 'items.data.0.metadata' )->asArray() );
					return;
				}

				if ( $stripeObject->getString( 'schedule', '' ) ) {
					$results	=	$this->getStripeSubscriptionSchedule( $stripeObject->getString( 'schedule', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'phases.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'phases.0.metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'last_invoice', '' ) ) {
					$results	=	$this->getStripeInvoice( $stripeObject->getString( 'last_invoice', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'lines.data.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'lines.data.0.metadata' )->asArray() );
						return;
					}
				}
				break;
			case 'dispute';
				if ( $stripeObject->getString( 'charge', '' ) ) {
					$results	=	$this->getStripeCharge( $stripeObject->getString( 'charge', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}
				}
				break;
			case 'charge';
				if ( $stripeObject->getString( 'invoice', '' ) ) {
					$results	=	$this->getStripeInvoice( $stripeObject->getString( 'invoice', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'lines.data.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'lines.data.0.metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'payment_intent', '' ) ) {
					$results	=	$this->getStripePaymentIntent( $stripeObject->getString( 'payment_intent', '' ), [] );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}
				}
				break;
			case 'payment_intent';
				if ( $stripeObject->getString( 'subscription', '' ) ) {
					$results	=	$this->getStripeSubscription( $stripeObject->getString( 'subscription', '' ), [ 'schedule' ] );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'schedule.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'schedule.metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'schedule.phases.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'schedule.phases.0.metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'items.data.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'items.data.0.metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'invoice', '' ) ) {
					$results	=	$this->getStripeInvoice( $stripeObject->getString( 'invoice', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'lines.data.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'lines.data.0.metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'charge', '' ) ) {
					$results	=	$this->getStripeCharge( $stripeObject->getString( 'charge', '' ) );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}
				}
				break;
			case 'checkout.session';
				if ( $stripeObject->getString( 'payment_intent', '' ) ) {
					$results	=	$this->getStripePaymentIntent( $stripeObject->getString( 'payment_intent', '' ), [] );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}
				}

				if ( $stripeObject->getString( 'subscription', '' ) ) {
					$results	=	$this->getStripeSubscription( $stripeObject->getString( 'subscription', '' ), [ 'schedule' ] );

					if ( $results->getInt( 'metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'schedule.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'schedule.metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'schedule.phases.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'schedule.phases.0.metadata' )->asArray() );
						return;
					}

					if ( $results->getInt( 'items.data.0.metadata.order_id', 0 ) ) {
						$stripeObject->set( 'metadata', $results->subTree( 'items.data.0.metadata' )->asArray() );
						return;
					}
				}
				break;
		}
	}

	/**
	 * Returns a IETF language tag supported by Stripe based off the users preferred language
	 *
	 * @param int $userId
	 * @return string
	 */
	private function getStripeLocale( int $userId ): string
	{
		$language		=	CBuser::getUserDataInstance( $userId )->getUserLanguage();

		if ( ! $language ) {
			$language	=	Application::Cms()->getLanguageTag();
		}

		if ( ! $language ) {
			return '';
		}

		$language		=	strtolower( substr( $language, 0, 2 ) );

		if ( ! in_array( $language, [ 'da', 'de', 'en', 'es', 'fi', 'fr', 'it', 'ja', 'nb', 'nl', 'pl', 'pt', 'sv', 'zh' ], true ) ) {
			return '';
		}

		return $language;
	}

	/**
	 * Limits and converts periods to Stripe limits
	 *
	 * @param array    $periodTypeArray ( int $periodCount, string $periodType ) : $periodType: 'D','W','M','Y'
	 * @param null|int $now             unix timestamp of now
	 * @return array                    array containing interval (day, week, month, year), interval_count (number of times of interval), start (when period starts), error (if the period limits are invalid)
	 */
	private function getStripePeriodLimits( array $periodTypeArray, ?int $now = null ): array
	{
		global $_CB_framework;

		$periodCount			=	(int) $periodTypeArray[0];
		$periodType				=	$periodTypeArray[1];

		if ( ! $now ) {
			$now				=	$_CB_framework->now();
		}

		$startTime				=	$now;
		$error					=	null;

		if ( $periodType === 'D' ) {
			$startTime			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . $periodCount . ' DAY', $now );
			$periodType			=	'day';

			if ( $periodCount % 7 === 0 ) {
				// Convert days to weeks:
				$periodCount	/=	7;
				$periodType		=	'W';
			} elseif ( $periodCount % 30 === 0 ) {
				// Convert days to months:
				$periodCount	/=	30;
				$periodType		=	'M';
			} elseif ( $periodCount % 365 === 0 ) {
				// Convert days to years:
				$periodCount	/=	365;
				$periodType		=	'Y';
			} elseif ( $periodCount > 365 ) {
				$error			=	CBTxt::T( 'STRIPE_INVALID_PLANS_DURATION_DAYS', 'Stripe: Trying to subscribe [days] days. Maximum supported is 365 days (1 year).', [ '[days]' => $periodCount ] );
			}
		}

		if ( $periodType === 'W' ) {
			$startTime			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . $periodCount . ' WEEK', $now );
			$periodType			=	'week';

			if ( ( $periodCount * 7 ) % 30 === 0 ) {
				// Convert weeks to months:
				$periodCount	=	( ( $periodCount * 7 ) / 30 );
				$periodType		=	'M';
			} elseif ( ( $periodCount * 7 ) % 365 === 0 ) {
				// Convert weeks to years:
				$periodCount	=	( ( $periodCount * 7 ) / 365 );
				$periodType		=	'Y';
			} elseif ( $periodCount > 52 ) {
				$error			=	CBTxt::T( 'STRIPE_INVALID_PLANS_DURATION_WEEKS', 'Stripe: Trying to subscribe [weeks] weeks. Maximum supported is 52 weeks (1 year).', [ '[weeks]' => $periodCount ] );
			}
		}

		if ( $periodType === 'M' ) {
			$startTime			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . $periodCount . ' MONTH', $now );
			$periodType			=	'month';

			if ( ( $periodCount * 30 ) % 365 === 0 ) {
				// Convert months to years:
				$periodCount	=	( ( $periodCount * 30 ) / 365 );
				$periodType		=	'Y';
			} elseif ( $periodCount > 12 ) {
				$error			=	CBTxt::T( 'STRIPE_INVALID_PLANS_DURATION_MONTHS', 'Stripe: Trying to subscribe [months] months. Maximum supported is 12 months (1 year).', [ '[months]' => $periodCount ] );
			}
		}

		if ( $periodType === 'Y' ) {
			$startTime			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . $periodCount . ' YEAR', $now );
			$periodType			=	'year';

			if ( $periodCount > 1 ) {
				$error			=	CBTxt::T( 'STRIPE_INVALID_PLANS_DURATION_YEARS', 'Stripe: Trying to subscribe [years] years. Maximum supported is 1 year.', [ '[years]' => $periodCount ] );
			}
		}

		return [ 'interval' => $periodType, 'interval_count' => $periodCount, 'start' => $startTime, 'error' => $error ];
	}

	/**
	 * This will attempt to find the payment type from stripe api or ipn data in all possible locations
	 *
	 * @param Registry $data
	 * @return string
	 */
	private function getStripePaymentType( Registry $data ): string
	{
		// Try to directly detect what type of card they used (only applicable to card payments)
		$paymentBrand				=	$data->getString( 'payment_method_details.card.brand', '' );

		if ( ! $paymentBrand ) {
			$paymentBrand			=	$data->getString( 'latest_charge.payment_method_details.card.brand', '' );
		}

		if ( ! $paymentBrand ) {
			$paymentBrand			=	$data->getString( 'payment_method.card.brand', '' );
		}

		if ( ! $paymentBrand ) {
			$paymentBrand			=	$data->getString( 'last_payment_error.payment_method.card.brand', '' );
		}

		if ( $paymentBrand ) {
			// Card brand was found so lets see how it was funded (e.g. credit, debit, etc..)
			$paymentFunding			=	$data->getString( 'payment_method_details.card.funding', '' );

			if ( ! $paymentFunding ) {
				$paymentFunding		=	$data->getString( 'latest_charge.payment_method_details.card.funding', '' );
			}

			if ( ! $paymentFunding ) {
				$paymentFunding		=	$data->getString( 'payment_method.card.funding', '' );
			}

			if ( ! $paymentFunding ) {
				$paymentFunding		=	$data->getString( 'last_payment_error.payment_method.card.funding', '' );
			}

			return $paymentBrand . ( $paymentFunding ? ' ' . $paymentFunding : '' );
		}

		// Fallback to type which we'll translate later in getTxtUsingAccount
		$paymentType				=	$data->getString( 'payment_method_details.type', '' );

		if ( ! $paymentType ) {
			$paymentType			=	$data->getString( 'last_payment_error.payment_method_details.type', '' );
		}

		if ( ! $paymentType ) {
			$paymentType			=	$data->getString( 'payment_method.type', '' );
		}

		if ( ! $paymentType ) {
			$paymentType			=	$data->getString( 'last_payment_error.payment_method.type', '' );
		}

		return ( $paymentType ?: 'stripe' );
	}

	/**
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param bool                $recurring
	 * @param bool                $checkout
	 * @return string
	 */
	private function getStripeReturnURL( cbpaidPaymentBasket $paymentBasket, bool $recurring = false, bool $checkout = false ): string
	{
		global $_CB_framework;

		$url	=	$this->_getPayNowUrl( $paymentBasket, [ 'paymenttype' => ( $recurring ? 2 : 1 ), 'shopuser' => $this->shopuserParam( $paymentBasket ), 'cardtype' => ( $checkout ? ( $recurring ? 'stripe_recurring_checkout' : 'stripe_single_checkout' ) : ( $recurring ? 'stripe_recurring' : 'stripe_single' ) ) ] );

		if ( (int) $_CB_framework->getCfg( 'unicodeslugs' ) === 1 ) {
			// Stripe does not accept Non-ASCII characters in return_urls, which can happen if Joomla is set to allow unicode aliases so lets escape them
			return preg_replace_callback( '/[^\x20-\x7f]/', static function( $match ) {
						return urlencode( $match[0] );
					}, $url );
		}

		return $url;
	}

	/**
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @return string
	 */
	private function getStripeCancelURL( cbpaidPaymentBasket $paymentBasket ): string
	{
		global $_CB_framework;

		$url	=	$this->getCancelUrl( $paymentBasket );

		if ( (int) $_CB_framework->getCfg( 'unicodeslugs' ) === 1 ) {
			// Stripe does not accept Non-ASCII characters in cancel_urls, which can happen if Joomla is set to allow unicode aliases so lets escape them
			return preg_replace_callback( '/[^\x20-\x7f]/', static function( $match ) {
						return urlencode( $match[0] );
					}, $url );
		}

		return $url;
	}

	/**
	 * Make a versioned Stripe API HTTPS request
	 *
	 * @param string $type
	 * @param string $url
	 * @param array  $request
	 * @return Registry
	 */
	private function httpsRequestStripe( string $type, string $url, array $request = [] ): Registry
	{
		$results		=	[];
		$response		=	'';
		$status			=	0;
		$error			=	$this->httpsRequestGuzzle( $this->gatewayUrl( 'psp' ) . $url, $request, 105, $response, $status, $type, 'normal', '*/*', true, 443, $this->getAccountParam( 'stripe_sk', '' ), '', false, null, [ 'Stripe-Version' => $this->stripeApiVersion ] );

		if ( $response ) {
			$results	=	json_decode( $response, true );
		}

		if ( ( $error || ( $status !== 200 ) || ( ! $response ) || ( ! $results ) ) && ( ! Get::get( $results, 'error', [], GetterInterface::RAW ) ) ) {
			$results	=	[ 'error' => [ 'type' => 'unknown', 'message' => $error, 'code' => $status ] ];
		}

		return new Registry( $results );
	}

	/**
	 * perform anti fraud checks on ipn values
	 *
	 * @param cbpaidPaymentNotification $ipn
	 * @param cbpaidPaymentBasket       $paymentBasket
	 * @param string                    $payload
	 * @param string                    $signingSecret
	 * @return bool|string
	 */
	private function _validateIPN( cbpaidPaymentNotification $ipn, cbpaidPaymentBasket $paymentBasket, string $payload, string $signingSecret )
	{
		global $_CB_database;

		$matching						=	true;

		if ( in_array( $ipn->getString( 'payment_status', '' ), [ 'Completed', 'Processed', 'Canceled_Reversal' ], true ) ) {
			if ( in_array( $ipn->getString( 'txn_type', '' ), [ 'subscr_payment', 'subscr_signup' ], true ) ) {
				$payments				=	$paymentBasket->getPaymentsTotals( $ipn->getString( 'txn_id', '' ) );

				if ( ( $paymentBasket->getFloat( 'mc_amount1', 0 ) !== 0 ) && ( $payments->count === 0 ) ) {
					$amount				=	$paymentBasket->getFloat( 'mc_amount1', 0 );
				} else {
					$amount				=	$paymentBasket->getFloat( 'mc_amount3', 0 );
				}

				if ( ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) !== sprintf( '%.2f', $amount ) )
					 && ( ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) < sprintf( '%.2f', $amount ) )
						  || ( sprintf( '%.2f', ( $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ) ) !== sprintf( '%.2f', $amount ) ) )
					 && ( ( ! ( ( $paymentBasket->getFloat( 'mc_amount1', 0 ) !== 0 ) && ( $payments->count === 0 ) ) )
						  && ( ( (float) sprintf( '%.2f', ( $ipn->getFloat( 'mc_gross', 0 ) - abs( $ipn->getFloat( 'tax', 0 ) ) ) ) ) < ( (float) sprintf( '%.2f', $amount ) ) ) )
				) {
					$matching			=	CBTxt::T( 'amount mismatch on recurring_payment: amount: [amount] != IPN mc_gross: [gross] or IPN mc_gross - IPN tax: [net] where IPN tax = [tax]', null, [ '[amount]' => $amount, '[net]' => ( $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ), '[gross]' => $ipn->getFloat( 'mc_gross', 0 ), '[tax]' => $ipn->getFloat( 'tax', 0 ) ] );
				}
			} elseif ( ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) !== sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) )
					   && ( ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) ) < sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) )
							|| ( sprintf( '%.2f', $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ) !== sprintf( '%.2f', $paymentBasket->getFloat( 'mc_gross', 0 ) ) ) )
			) {
				$matching				=	CBTxt::T( 'amount mismatch on webaccept: BASKET mc_gross: [basket_gross] != IPN mc_gross: [gross] or IPN mc_gross - IPN tax: [net] where IPN tax = [tax]', null, [ '[basket_gross]' => $paymentBasket->getFloat( 'mc_gross', 0 ), '[net]' => ( $ipn->getFloat( 'mc_gross', 0 ) - $ipn->getFloat( 'tax', 0 ) ), '[gross]' => $ipn->getFloat( 'mc_gross', 0 ), '[tax]' => $ipn->getFloat( 'tax', 0 ) ] );
			}
		}

		if ( in_array( $ipn->getString( 'txn_type', '' ), [ 'subscr_payment', 'subscr_signup', 'subscr_cancel', 'subscr_eot', 'subscr_failed' ], true ) && ( ! $paymentBasket->isAnyAutoRecurring() ) ) {
			$matching					=	CBTxt::T( 'Stripe subscription IPN type [txn_type] for a basket without auto-recurring items', null, [ '[txn_type]' => $ipn->translatedTransactionType() ] );
		}

		if ( ! in_array( $ipn->getString( 'txn_type', '' ), [ 'subscr_signup', 'subscr_cancel', 'subscr_eot', 'subscr_failed' ], true ) ) {
			if ( $ipn->getString( 'txn_id', '' ) === '' ) {
				$matching				=	CBTxt::T( 'illegal transaction id' );
			} else {
				$countBaskets			=	$paymentBasket->countRows( "txn_id = '" . $_CB_database->getEscaped( $ipn->getString( 'txn_id', '' ) ) . "' AND payment_status = 'Completed'" );

				if ( ( ( $countBaskets === 1 ) && ( $paymentBasket->getString( 'txn_id', '' ) !== $ipn->getString( 'txn_id', '' ) ) ) || ( $countBaskets > 1 ) ) {
					$matching			=	CBTxt::T( 'transaction already used for [count] other already completed payment(s)', null, [ '[count]' => $countBaskets ] );
				}
			}
		}

		// https://stripe.com/docs/webhooks#verify-manually
		$signingSecret					=	( $signingSecret ?: $this->getAccountParam( 'stripe_ss', '' ) );

		if ( $signingSecret ) {
			$signature					=	Application::Input()->getNamespaceRegistry( 'server' )->getString( 'HTTP_STRIPE_SIGNATURE', '' );

			if ( $signature ) {
				$signatureParts			=	explode( ',', $signature );
				$timestamp				=	'';
				$signed					=	'';

				foreach ( $signatureParts as $signaturePart ) {
					$signaturePieces	=	explode( '=', $signaturePart );
					$signatureType		=	( $signaturePieces[0] ?? '' );
					$signatureValue		=	( $signaturePieces[1] ?? '' );

					if ( $signatureType === 't' ) {
						$timestamp		=	$signatureValue;
					} elseif ( $signatureType === 'v1' ) {
						$signed			=	$signatureValue;
					}
				}

				if ( ! $signed ) {
					$matching			=	CBTxt::T( 'missing signature returned by Stripe' );
				} elseif ( $signed !== hash_hmac( 'sha256', $timestamp . '.' . $payload, $signingSecret ) ) {
					$matching			=	CBTxt::T( 'signature [signed] returned by Stripe does not match the value we expected', null, [ '[signed]' => htmlspecialchars( $signed ) ] );
				}
			} else {
				$matching				=	CBTxt::T( 'missing signature secret returned by Stripe' );
			}
		}

		return $matching;
	}

	/**
	 * Logs payment notification
	 *
	 * @param string              $logType
	 * @param int                 $now
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param array|null          $card
	 * @param array               $request
	 * @param Registry            $results
	 * @param null|string|array   $return
	 * @return cbpaidPaymentNotification
	 */
	private function _logNotification( string $logType, int $now, cbpaidPaymentBasket$paymentBasket, ?array $card, array $request, Registry $results, $return ): cbpaidPaymentNotification
	{
		$paymentType					=	( $card['type'] ?? '' );

		if ( in_array( $paymentType, [ 'stripe_single', 'stripe_recurring', 'stripe_single_checkout', 'stripe_recurring_checkout' ], true ) ) {
			$paymentType				=	$this->getStripePaymentType( $results );
		} elseif ( ! in_array( $paymentType, [ 'sepa', 'sepa_debit', 'ach', 'ach_credit', 'ach_debit', 'bacs_debit', 'us_bank_account', 'alipay', 'ideal', 'bancontact', 'giropay', 'p24', 'sofort', 'eps', 'multibanco', 'fpx', 'au_becs_debit' ], true ) ) {
			$paymentType				=	( $paymentType ? ucwords( $paymentType ) . ' ' : '' ) . 'Credit Card';
		}

		if ( is_string( $return ) ) {
			$transactionId				=	$return;

			if ( $results->getString( 'charge_status', '' ) === 'pending' ) {
				if ( $this->getAccountParam( 'accept_payment_condition', '' ) === 'pending' ) {
					$paymentStatus		=	'Completed';
					$reason				=	'';
				} else {
					$paymentStatus		=	'Pending';
					$reason				=	'Authorized';
				}
			} else {
				$paymentStatus			=	'Completed';
				$reason					=	'';
			}

			$rawResult					=	'SUCCESS';

			if ( $logType === '5' ) {
				$paymentStatus			=	'Unsubscribed';
			}
		} else {
			$transactionId				=	'';
			$paymentStatus				=	'Denied';
			$reason						=	$results->getString( 'error.code', '' );
			$rawResult					=	'FAILED';
		}

		$ipn							=	$this->_prepareIpn( $logType, $paymentStatus, $paymentType, $reason, $now, 'utf-8' );

		$ipn->bindBasket( $paymentBasket );

		$ipn->set( 'user_id', $paymentBasket->getInt( 'user_id', 0 ) );

		$resultsExport					=	$results->asArray();

		// Remove private data from the response:
		if ( isset( $resultsExport['client_secret'] ) ) {
			unset( $resultsExport['client_secret'] );
		}

		// Remove private data from the request:
		if ( isset( $request['client_secret'] ) ) {
			unset( $request['client_secret'] );
		}

		$ipn->setPayerNameId( $paymentBasket->getString( 'first_name', '' ), $paymentBasket->getString( 'last_name', '' ) );
		$ipn->setRawResult( $rawResult );

		$rawData						=	'$results=' . var_export( $resultsExport, true ) . ";\n"
										.	'$return=' . var_export( $return, true ) . ";\n"
										.	'$request=' . var_export( $request, true ) . ";\n";

		$ipn->setRawData( $rawData );

		if ( is_string( $return ) ) {
			if ( in_array( $logType, [ 'P', 'B', 'Q', 'V', 'X' ], true ) ) {
				$ipn->setTxnSingle( $results->getString( 'charge_id', '' ) );
			} elseif ( in_array( $logType, [ 'A', 'C', 'Z', 'U', 'Y', 'W', '5' ], true ) ) {
				if ( ! $ipn->getString( 'txn_id', '' ) ) {
					$ipn->set( 'txn_id', $results->getString( 'charge_id', '' ) );
				}

				if ( ! $paymentBasket->getString( 'subscr_id', '' ) ) {
					$firstPayment		=	true;
				} else {
					$firstPayment		=	false;
				}

				$ipn->setTxnSubscription( $paymentBasket, $transactionId, $now );

				if ( $logType === '5' ) {
					$ipn->set( 'txn_type', 'subscr_cancel' );
				} elseif ( $firstPayment ) {
					$ipn->set( 'txn_type', 'subscr_signup' );
				}

				$this->_bindNotificationToBasket( $ipn, $paymentBasket );
			}
		}

		$ipn->store();

		return $ipn;
	}

	/**
	 * FUNCTIONS FOR BACKEND INTERFACE:
	 */

	/**
	 * Renders URL to set in the gateway interface for notifications
	 *
	 * @param string $urlType
	 * @return string
	 */
	public function adminUrlRender( string $urlType ): string
	{
		switch ( $urlType ) {
			case 'successurl':
				return $this->getSuccessUrl( null );
			case 'cancelurl':
				return $this->getCancelUrl( null );
			case 'notifyurl':
				return $this->getNotifyUrl( null );
			default:
				return 'Error: Unknown url type: ' . htmlspecialchars( $urlType );
		}
	}

	/**
	 * Returns list of webhooks used for rendering in admin interface
	 *
	 * @return string
	 */
	public function adminWebhooksRender(): string
	{
		return '<span class="badge badge-primary">' . implode( '</span> <span class="badge badge-primary">', $this->getStripeWebhooks() ) . '</span>';
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountstripeoem extends cbpaidGatewayAccountCreditCards
{

	/**
	 * USED by XML interface ONLY !!! Renders URL for successful returns
	 *
	 * @param  string             $value Variable value ( 'successurl', 'cancelurl', 'notifyurl' )
	 * @param  ParamsInterface    $params
	 * @param  string             $name  The name of the form element
	 * @param  CBSimpleXMLElement $node  The xml element for the parameter
	 * @return string                       HTML to display
	 */
	public function renderUrl( $value, $params, $name, $node ): string
	{
		return str_replace( 'http://', 'https://', $this->getPayMean()->adminUrlRender( $node->attributes( 'value' ) ) );
	}

	/**
	 * USED by XML interface ONLY !!! Renders webhooks used by this gateway
	 *
	 * @return string
	 */
	public function renderWebhooks(): string
	{
		return $this->getPayMean()->adminWebhooksRender();
	}
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidstripe extends cbpaidstripeoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountstripe extends cbpaidGatewayAccountstripeoem
{
}
