<?php
/**
* @version $Id: cbpaidsubscriptions.twocheckout.php 1598 2012-12-28 02:34:03Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Registry\ParamsInterface;
use CBLib\Xml\SimpleXMLElement;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php');

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
* Paid Subscriptions Tab Class for handling the CB tab api
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @author Beat
*/
class cbpaidtwocheckout extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";
	/**
	 * Array of gateway API urls: normally:
	 * array(	'single+normal' 	=>	'normal.gateway.com',
	 *			'single+test'		=>	'tests.gateway.com',
	 *			'recurring+normal'	=>	'recurring.gateway.com',
	 *			'recurring+test'	=>	'recurring-tests.gateway.com' );
	 * @var array of string
	 */
	protected $_gatewayUrls		=	array(	'twocheckout+normal' 	=>	'www.2checkout.com/checkout/purchase',
											'twocheckout+test'	 	=>	'sandbox.2checkout.com/checkout/purchase',
											'api+normal'		 	=>	'www.2checkout.com/api',
											'api+test'			 	=>	'sandbox.2checkout.com/api' );

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		return $this->_getPaymentVars( $paymentBasket, 0 );
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Popoulates subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		$varsArray								=	$this->_getPaymentVars( $paymentBasket, 1 );

		if ( $paymentBasket->period3 ) {
			$varsArray['li_0_price']			=	$this->_toGatewayCurrency( $paymentBasket, $paymentBasket->mc_amount3 );

			list( $recur1, $dur1 )				=	$this->_periodsLimits( explode( ' ', $paymentBasket->period3 ), $paymentBasket->recur_times );

			if ( $paymentBasket->period1 ) {
				list( $recur2, /* $dur2 */ )	=	$this->_periodsLimits( explode( ' ', $paymentBasket->period1 ), $paymentBasket->recur_times );

				if ( $recur1 != $recur2 ) {
					$this->_setLogErrorMSG( 4, null, '2Checkout: Invalid initial duration', CBTxt::T( '2CHECKOUT_INVALID_INITIAL_DURATION', '2Checkout: Trying to subscribe for different initial duration of [initial_duration], not supported by 2Checkout, subscription augmented to regular duration of [regular_duration].', array( '[initial_duration]' => $recur2, '[regular_duration]' => $recur1 ) ) );
				}

				// Startup fee is a price adjustment so for a lower initial price we need to discount it by the difference and for higher price we need to increase by the difference:
				if ( $paymentBasket->mc_amount1 != $paymentBasket->mc_amount3 ) {
					$priceDifference				=	( $paymentBasket->mc_amount1 - $paymentBasket->mc_amount3 );

					if ( $paymentBasket->mc_amount1 == 0.00 ) {
						$this->_setLogErrorMSG( 4, null, '2Checkout: Invalid first amount of 0.00', CBTxt::T( '2CHECKOUT_INVALID_INITIAL_AMOUNT', '2Checkout: Trying to subscribe with initial amount of 0.00, not supported by 2Checkout' ) );
					}

					$varsArray['li_0_startup_fee']	=	$this->_toGatewayCurrency( $paymentBasket, $priceDifference );
				}
			}

			// # Week, # Month, or # Year:
			$varsArray['li_0_recurrence']		=	$recur1;

			// Forever or # Week, # Month, # Year:
			$varsArray['li_0_duration']			=	$dur1;
		}

		return $varsArray;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['merchant_order_id'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata	=	$postdata;
		} else {
			// but if gateway needs GET, we will work with it too:
			$requestdata	=	$this->_getGetParams();
		}

		return $this->_handle2coReturn( $paymentBasket, $requestdata );
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// We do not have any cancel action in this gateway:
		return null;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		$return				=	$this->_handle2coINS( $paymentBasket, $postdata );

		// Also proxy the request if setup that way:
		$proxyUrl			=	$this->getAccountParam( 'proxy_ipn_url' );

		if ( $proxyUrl && preg_match( '/^https?:\/\//', $proxyUrl ) ) {
			$https			=	( substr( $proxyUrl, 0, 8 ) === 'https://' );
			$strippedUrl	=	substr( $proxyUrl, $https ? 8 : 7 );
			cbpaidWebservices::httpsRequest( $strippedUrl, $postdata, 30, $result, $status, 'post', 'normal', '*/*', $https, $https ? 443 : 80 );
		}

		return $return;
	}

	/**
	 * Cancels an existing recurring subscription
	 * (optional)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  cbpaidPaymentItem[]  $paymentItems   redirect immediately instead of returning HTML for output
	 *
	 * @return boolean|string              TRUE if unsubscription done successfully, STRING if error
	 */
	protected function handleStopPaymentSubscription( $paymentBasket, $paymentItems )
	{
		return $this->stop2coPaymentSubscription( $paymentBasket, $paymentItems );
	}

	/**
	 * Refunds a payment
	 *
	 * @param  cbpaidPaymentBasket       $paymentBasket  paymentBasket object
	 * @param  cbpaidPayment             $payment        payment object
	 * @param  cbpaidPaymentItem[]|null  $paymentItems   Array of payment items to refund completely (if no $amount)
	 * @param  boolean                   $lastRefund     Last refund for $payment ?
	 * @param  float                     $amount         Amount in currency of the payment
	 * @param  string                    $reasonText     Refund reason comment text for gateway
	 * @param  string                    $returnText     RETURN param : Text of success message or of error message
	 * @return boolean                                   true if refund done successfully, false if error
	 */
	public function refundPayment( $paymentBasket, $payment, $paymentItems, $lastRefund, $amount, $reasonText, &$returnText )
	{
		$coAPI							=	$this->_api_getInstance();
		if ( ! $coAPI ) {
			$returnText					=	CBTxt::T("2Checkout API not set");
			return false;
		}

		// prepare the notification record:
		$ipn							=	$this->_prepareIpn( 'R', $paymentBasket->payment_status, $paymentBasket->payment_type, 'Refund', null, 'utf-8' );
		$ipn->test_ipn					=	$paymentBasket->test_ipn;
		$ipn->raw_data					=	'$message_type="' . 'REFUND_ISSUED' . "\";\n"
										.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
										.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n"
										;

		// get the official Sale record from 2co:
		$saleRecord						=	$coAPI->_api_detail_sale( $paymentBasket->sale_id, null, $ipn->raw_data );
		if ( is_string( $saleRecord ) ) {
			$returnText					=	CBTxt::T("Error finding 2Checkout sales record: please contact site administrator and ask him to check error logs for details.");
			$this->_setLogErrorMSG( 3, $payment, sprintf('2Checkout: refund API error: "%s" : Is CBSubs 2co API setting correct?', $saleRecord ), $returnText );
			return false;
		}

		if ( $paymentItems ) {
			if ( ! isset( $saleRecord->invoices[0]->lineitems ) ) {
				$returnText				=	CBTxt::T("Error finding 2Checkout sales record: please contact site administrator and ask him to check error logs for details.");
				$this->_setLogErrorMSG( 3, $payment, '2Checkout: stopRecurring error: no invoices line items found', $returnText );
				return false;
			}

			$successOfAllItems			=	true;
			foreach ($paymentItems as $paymItem) {
				$success				=	null;
				// find 2co system product id for the paymentItem: This is only for old subscriptions in 2CO up to CBSubs 4.1.0:
				$twocoProductId			=	$paymItem->getPlanParam( 'twocheckout_auto_product_sysid', '0', 'integrations' );
				foreach ( $saleRecord->invoices[0]->lineitems as $lineItem ) {
					if ( ( ( (string) $lineItem->product_is_cart ) != '1' ) && ( ( $twocoProductId == 0 ) || ( ( (string) $lineItem->product_id ) == $twocoProductId ) ) ) {
						$result			=	$coAPI->_api_refund_lineitem((string) $lineItem->lineitem_id, 10, $reasonText, $ipn->raw_data );
						if ( is_string( $result ) ) {
							$returnText	=	CBTxt::T("Error: unable to refund 2Checkout line-item: please contact site administrator and ask him to check error logs for details.");
							$this->_setLogErrorMSG( 3, $payment, sprintf('2Checkout: refund_lineitem API error: %s for sale id: %s and lineitem_id %s', $result, $paymentBasket->sale_id, $lineItem->lineitem_id ), $returnText );
							$success	=	false;
						} else {
							// refund succeeded:
							$success	=	true;
						}
					}
				}
				if ( $success === null ) {
					$returnText			=	CBTxt::T("Error: did not find 2Checkout line-item in the invoice: please contact site administrator and ask him to check error logs for details.");
					$this->_setLogErrorMSG( 3, $payment, '2Checkout: refund_lineitem error: lineitem not found in first invoice', $returnText );
					$success			=	false;
				}
				$successOfAllItems		=	$successOfAllItems && $success;
			}
		} else {
			if ( abs( $amount - $paymentBasket->mc_gross ) < 0.00001 ) {
				$amount					=	null;
			} else {
				$amount					=	$this->_toGatewayCurrency( $paymentBasket, $amount );
			}
			$result						=	$coAPI->_api_refund_invoice( $saleRecord->invoices[0]->invoice_id, 10, $amount, $reasonText, $ipn->raw_data );
			if ( is_string( $result ) ) {
				$returnText				=	CBTxt::T("Error: unable to refund 2Checkout sale: please contact site administrator and ask him to check error logs for details.");
				$this->_setLogErrorMSG( 3, $payment, sprintf('2Checkout: refund_invoice API error: %s for sale id: %s and invoice_id %s', $result, $paymentBasket->sale_id, $saleRecord->invoices[0]->invoice_id ), $returnText );
				$successOfAllItems		=	false;
			} else {
				// refund succeeded:
				$successOfAllItems		=	true;
			}

		}
		if ( $successOfAllItems ) {
			$ipn->bindBasket( $paymentBasket );
			$ipn->mc_gross				=	- $ipn->mc_gross;
			$ipn->computeRefundedTax( $payment );
			$ipn->parent_txn_id			=	$payment->txn_id;
			$ipn->store();

			$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, true );
		}

		return $successOfAllItems;
	}

	/**
	 * INTERNAL METHODS THAT CAN BE RE-IMPLEMENTED IN PAYMENT HANDLER IF NEEDED:
	 */

	/**
	 * gives gateway button URL server name from gateway URL list
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  boolean              $autoRecurring   TRUE: autorecurring payment, FALSE: single payment
	 * @return string  server-name (with 'https://' )
	 */
	protected function pspUrl( $paymentBasket, $autoRecurring )
	{
		return $this->gatewayUrl( 'twocheckout' );
	}

	/**
	 * Returns an array for the 'radios' array of $redirectNow type:
	 * return array( account_id, submethod, paymentMethod:'single'|'subscribe', array(cardtypes), 'label for radio', 'description for radio' )
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  string               $subMethod
	 * @param  string               $paymentType
	 * @param  string               $defaultLabel
	 * @return array
	 */
	protected function getPayRadioRecepie( $paymentBasket, $subMethod, $paymentType, $defaultLabel )
	{
		// Override default default label 'Credit Card':
		$defaultLabel				=	( ( $paymentType == '' ) ?
										'Pay with 2Checkout.com, our authorized retailer, with automatic following payments'		// CBTxt::T("Pay with 2Checkout.com, our authorized retailer, with automatic following payments")
										:
										'Pay with 2Checkout.com, our authorized retailer' );			// CBTxt::T("Pay with 2Checkout.com, our authorized retailer")

		return parent::getPayRadioRecepie( $paymentBasket, $subMethod, $paymentType, $defaultLabel );
	}

	/**
	 * Returns a cbpaidGatewaySelectorButton object parameters for rendering an HTML form with a visible button and hidden fields for the gateway
	 * Or a string with HTML content instead (not recommended)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  string               $subMethod      'single', 'subscribe' or gateway-specific string (e.g. credit-card brand)
	 * @param  string               $paymentType    'single' or 'subscribe' or for subscriptions 'cancel'
	 * @return cbpaidGatewaySelectorButton                  or string with HTML
	 */
	protected function getPayButtonRecepie( $paymentBasket, $subMethod, $paymentType )
	{
		$requestParams			=	( ( $paymentType == 'single' ) ? $this->getSinglePaymentRequstParams( $paymentBasket ) : $this->getSubscriptionRequstParams( $paymentBasket ) );
		$baseParamName			=	$this->getPayName() . ( $paymentType == 'single' ? '' : '_' . $paymentType );
		$prmImg					=	$baseParamName . '_image';
		$prmCustImg				=	$baseParamName . '_custom_image';
		$prmCustCSS				=	$baseParamName . '_custom_css';
		$prmAltText				=	$baseParamName . '_image_alttext';
		$butId					=	'cbpaidButt' . $this->getPayName() . ( $paymentType == 'single' ? '' : $paymentType );

		$customImage			=	trim( $this->getAccountParam( $prmCustImg ) );
		if ( $customImage == '' ) {
			$customImage		=	cbpaidApp::renderCCImage( $this->getAccountParam( $prmImg ), true );
		}

		$altText				=	trim( $this->getAccountParam( $prmAltText ) );
		if ( trim( $this->getAccountParam( 'twocheckout_title_alt_text' ) ) ) {
			$altText			.=	'. ' . $this->getAccountParam( 'twocheckout_title_alt_text' );
		}
		$titleText				=	$altText;
		$payNameForCssClass		=	$this->getPayName();

		$pspUrl					=	$this->pspUrl( $paymentBasket, ( $paymentType == 'subscribe' ) );
		return cbpaidGatewaySelectorButton::getPaymentButton( $this->getAccountParam( 'id' ), $subMethod, $paymentType, $pspUrl, $requestParams, $customImage, $this->getAccountParam( $prmCustCSS ), $altText, $titleText, $payNameForCssClass, $butId );
	}

	/**
	 * Returns a cbpaidGatewaySelectorButton object parameters for rendering an HTML form with a visible button and hidden fields for the gateway
	 * For just switching currency of gateway.
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  string               $subMethod      'single', 'subscribe' or gateway-specific string (e.g. credit-card brand)
	 * @param  string               $paymentType    'single' or 'subscribe' or for subscriptions 'cancel'
	 * @return cbpaidGatewaySelectorButton                  or string with HTML
	 */
	protected function getChangeOfCurrencyButton( $paymentBasket, $subMethod, $paymentType )
	{
		$baseParamName			=	$this->getPayName() . ( $paymentType == 'single' ? '' : '_' . $paymentType );
		$prmImg					=	$baseParamName . '_image';
		$prmCustImg				=	$baseParamName . '_custom_image';
		$prmAltText				=	$baseParamName . '_image_alttext';
		$butId					=	'cbpaidButt' . $this->getPayName() . ( $paymentType == 'single' ? '' : $paymentType );

		$customImage			=	trim( $this->getAccountParam( $prmCustImg ) );
		if ( $customImage == '' ) {
			$customImage		=	cbpaidApp::renderCCImage( $this->getAccountParam( $prmImg ), true );
		}

		$altText				=	trim( $this->getAccountParam( $prmAltText ) );
		if ( trim( $this->getAccountParam( 'twocheckout_title_alt_text' ) ) ) {
			$altText			.=	'. ' . $this->getAccountParam( 'twocheckout_title_alt_text' );
		}

		$titleText				=	CBTxt::T( $this->getAccountParam( 'currency_acceptance_text' ) );

		$payNameForCssClass		=	$this->getPayName();
		$newCurrency			=	$this->mainCurrencyOfGateway();
		return cbpaidGatewaySelectorButton::getChangeOfCurrencyButton( $paymentBasket, $newCurrency, $customImage, $altText, $titleText, $payNameForCssClass . ' ' . 'cbregconfirmtitleonclick', $butId );
	}

	/**
	 * GATEWAY-INTERNAL SPECIFIC METHODS:
	 */

	/**
	* Stops a payment subscription:
	*
	* @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	* @param  cbpaidPaymentItem[]  $paymentItems   redirect immediately instead of returning HTML for output
	* @return boolean                              TRUE if unsubscription done successfully, FALSE if error
	*/
	private function stop2coPaymentSubscription( $paymentBasket, $paymentItems )
	{
		$coAPI					=	$this->_api_getInstance();
		if ( ! $coAPI ) {
			return false;
		}

		// prepare the notification record:
		$ipn					=	$this->_prepareIpn( 'R', $paymentBasket->payment_status, $paymentBasket->payment_type, 'Unsubscribe', null, 'utf-8' );
		$ipn->test_ipn			=	$paymentBasket->test_ipn;
		$ipn->raw_data			=	'$message_type="' . 'STOP_PAYMENT_SUBSCRIPTION' . "\";\n"
								.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
								.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n"
								;
		// get the official Sale record from 2co:
		$saleRecord				=	$coAPI->_api_detail_sale( $paymentBasket->sale_id, null, $ipn->raw_data );
		if ( is_string( $saleRecord ) ) {
			$this->_setLogErrorMSG( 3, $paymentBasket, sprintf('2Checkout: stopRecurring API error: "%s" : Is CBSubs 2co API setting correct?', $saleRecord ), CBTxt::T("Error finding 2Checkout sales record: please contact site administrator and ask him to check error logs for details.") );
			return false;
		}
		if ( ! isset( $saleRecord->invoices[0]->lineitems ) ) {
			$this->_setLogErrorMSG( 3, $paymentBasket, '2Checkout: stopRecurring error: no invoices line items found', CBTxt::T("Error finding 2Checkout sales record: please contact site administrator and ask him to check error logs for details.") );
			return false;
		}
		$successOfAllItems			=	true;
		foreach ($paymentItems as $paymItem) {
			$success				=	null;
			// find 2co system product id for the paymentItem: This is only for old subscriptions in 2CO up to CBSubs 4.1.0:
			$twocoProductId			=	$paymItem->getPlanParam( 'twocheckout_auto_product_sysid', '0', 'integrations' );
			foreach ( $saleRecord->invoices[0]->lineitems as $lineItem ) {
				if ( ( ( (string) $lineItem->product_is_cart ) != '1' ) && ( ( $twocoProductId == 0 ) || ( ( (string) $lineItem->product_id ) == $twocoProductId ) ) ) {
					$result			=	$coAPI->_api_stop_lineitem_recurring( (string) $lineItem->lineitem_id, $ipn->raw_data );
					if ( is_string( $result ) ) {
						$this->_setLogErrorMSG( 3, $paymentBasket, sprintf('2Checkout: stopRecurring API error: %s for sale id: %s and lineitem_id %s', $result, $paymentBasket->sale_id, $lineItem->lineitem_id ), CBTxt::T("Error: unable to stop 2Checkout line-item: please contact site administrator and ask him to check error logs for details.") );
						$success	=	false;
					} else {
						// unsubscribe succeeded:
						$ipn->bindBasket( $paymentBasket );
						$ipn->store();

						$success	=	true;
					}
				}
			}
			if ( $success === null ) {
				$this->_setLogErrorMSG( 3, $paymentBasket, '2Checkout: stopRecurring error: lineitem not found in first invoice', CBTxt::T("Error: did not find 2Checkout line-item in the invoice: please contact site administrator and ask him to check error logs for details.") );
				$success			=	false;
			}
			$successOfAllItems		=	$successOfAllItems && $success;
		}
		return $successOfAllItems;
	}

	/**
	 * Converts gateway-specific payment $method identifier to a human text
	 *
	 * @param  string  $method
	 * @return string
	 */
	private function _getPayment_type_from_method( $method )
	{
		static $methods2co	=	array(	'CC'  => 'Credit Card',
										'CK'  => 'Check',
										'AL'  => 'Acculynk PIN-debit',
										'PPI' => 'PayPal',
										'PPL' => 'PayPal Pay Later'
								);
		if ( isset( $methods2co[$method] ) ) {
			return $methods2co[$method];
		}
		return $method;
	}

	/**
	 * Comupte the CBSubs payment_status based on 2Checkout's invoice_status and fraud_status
	 *
	 * invoice_status   payment status normally changes in this sequence: 'approved' -> 'pending' -> 'deposited'
	 * fraud_status     fraud status normally changes in this sequence: 'wait' -> 'pass' (but can then do this too -> 'wait' -> 'pass' -> ... 'fail')
	 *
	 * @param  string  $invoice_status  2Checkout invoice_status
	 * @param  string  $fraud_status    2Checkout fraud_status
	 * @param  string  $previousStatus  previous CBSubs status
	 * @return string
	 */
	private function _paymentStatus( $invoice_status, $fraud_status, $previousStatus )
	{
		$accept_payment_condition			=	$this->getAccountParam( 'twocheckout_accept_payment_condition', 'fraudcheck_passed' );
		switch ( $invoice_status ) {
			case 'approved':
				if ( $accept_payment_condition == 'authorized' ) {
					$paymentStatus			=	'Completed';
					break;
				} elseif ( $accept_payment_condition == 'deposited' ) {
					$paymentStatus			=	'Pending';
					break;
				} // else $accept_payment_condition == 'fraudcheck_passed' :
				switch ( $fraud_status ) {
					case 'wait':
						if ( $previousStatus != 'Completed' ) {
							$paymentStatus	=	'Pending';
						} else {
							$paymentStatus	=	'Reversed';
						}
						break;
					case 'pass':
						$paymentStatus		=	'Completed';
						break;
					case 'fail':
						$paymentStatus		=	'Denied';
						break;
					default:
						$paymentStatus		=	'Pending';
						break;
				}
				break;
			case 'pending':
			case 'deposited':
				if ( $invoice_status == 'pending' ) {
					if ( $accept_payment_condition == 'deposited' ) {
						$paymentStatus			=	'Pending';
						break;
					}
				}
				if ( $accept_payment_condition == 'authorized' ) {
					$paymentStatus			=	'Completed';
					break;
				}
				// elseif $accept_payment_condition == 'fraudcheck_passed' OR == 'deposited' (and $invoice_status == 'deposited'):
				switch ( $fraud_status ) {
					case 'wait':
						if ( $previousStatus != 'Completed' ) {
							$paymentStatus	=	'Pending';
						} else {
							$paymentStatus	=	'Reversed';
						}
						break;
					case 'pass':
						$paymentStatus		=	'Completed';
						break;
					case 'fail':
						$paymentStatus		=	'Denied';		// this deposited+fail shouldn't happen except as transient state
						break;
					default:
						$paymentStatus		=	'Completed';
						break;
				}
				break;
			case 'declined':
				$paymentStatus				=	'Denied';
				break;
			case '':
			case null:
				$paymentStatus				=	'Pending';
				break;
			default:
				$paymentStatus				=	'Unknown';
				break;
		}
		return $paymentStatus;
	}

	/**
	 * Computes request variables for gateway single payment request/post
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket        Payment basket to pay
	 * @param  int                  $autorecurring        1 if the payment should be autorecurring, 0 for single-payment
	 * @return string[]
	 */
	private function _getPaymentVars( $paymentBasket, $autorecurring )
	{
		$varsArray['sid']						=	trim( $this->getAccountParam( 'twocheckout_account' ) );
		$varsArray['mode']						=	'2CO';
		$varsArray['li_0_type']					=	'product';
		$varsArray['li_0_product_id']			=	$paymentBasket->item_number;
		$varsArray['li_0_name']					=	$paymentBasket->item_name;
		$varsArray['li_0_quantity']				=	1;
		$varsArray['li_0_price']				=	$this->_toGatewayCurrency( $paymentBasket );
		$varsArray['li_0_tangible']				=	'N';

		if ( $this->getAccountParam( 'twocheckout_demosales' ) ) {
			$varsArray['demo']					=	'Y';
		}

		$varsArray['currency_code']				=	$paymentBasket->mc_currency;
		$varsArray['lang']						=	$this->getAccountParam( 'twocheckout_lang', 'en' );
		$varsArray['merchant_order_id']			=	$paymentBasket->id;
		$varsArray['x_receipt_link_url']		=	$this->getSuccessUrlTwoCheckout( $paymentBasket, $autorecurring );

		$name									=	$paymentBasket->first_name
												.	( ( $paymentBasket->first_name && $paymentBasket->last_name ) ? ' ' : '' )
												.	$paymentBasket->last_name;

		// Add the billing address:
		if ( $this->getAccountParam( 'twocheckout_prepopulate_bill' ) == 1 ) {
			$varsArray['card_holder_name']		=	cbIsoUtf_substr( $name, 0, 128 );
			$varsArray['street_address']		=	cbIsoUtf_substr( $paymentBasket->address_street, 0, 64 );
			$varsArray['street_address2']		=	'';
			$varsArray['city']					=	cbIsoUtf_substr( $paymentBasket->address_city, 0, 64 );
			$varsArray['state']					=	$paymentBasket->getInvoiceState();
			$varsArray['zip']					=	cbIsoUtf_substr( $paymentBasket->address_zip, 0, 16 );
			$varsArray['country']				=	$paymentBasket->getInvoiceCountry( 3 );
			$varsArray['email']					=	cbIsoUtf_substr( $paymentBasket->payer_email, 0, 64 );
			$varsArray['phone']					=	cbIsoUtf_substr( $paymentBasket->contact_phone, 0, 16 );
			$varsArray['phone_extension']		=	'';
		}

		// Add the shipping address:
		if ( $this->getAccountParam( 'twocheckout_prepopulate_ship' ) == 1 ) {
			$varsArray['ship_name']				=	cbIsoUtf_substr( $name, 0, 128 );
			$varsArray['ship_street_address']	=	cbIsoUtf_substr( $paymentBasket->address_street, 0, 64 );
			$varsArray['ship_street_address2']	=	'';
			$varsArray['ship_city']				=	cbIsoUtf_substr( $paymentBasket->address_city, 0, 64 );
			$varsArray['ship_state']			=	$paymentBasket->getInvoiceState();
			$varsArray['ship_zip']				=	cbIsoUtf_substr( $paymentBasket->address_zip, 0, 16 );
			$varsArray['ship_country']			=	$paymentBasket->getInvoiceCountry( 3 );
		}

		return $varsArray;
	}

	/**
	 * Returns success redirect-back URL with http in front
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  int                  $autorecurring  1 if the payment should be autorecurring, 0 for single-payment
	 * @return string
	 */
	private function getSuccessUrlTwoCheckout( $paymentBasket, $recurring = 0 )
	{
		return $this->cbsubsGatewayUrl( 'success', null, $paymentBasket, array( 'recurring' => (int) $recurring ), false, false, true );
	}

	/**
	 * Convert to the single 2co currency, in '%.2f' format
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  float|int|null       $amount
	 * @param  string|null          $currency
	 * @return float                                $amount in gateway's currency
	 */
	private function _toGatewayCurrency( $paymentBasket, $amount = null, $currency = null )
	{
		if ( $currency === null ) {
			$currency						=	$paymentBasket->mc_currency;
		}
		if ( $amount === null ) {
			$amount						=	$paymentBasket->mc_gross;
		}
		$cbpaidMoney						=	cbpaidMoney::getInstance();
		$twocoCurrency						=	$cbpaidMoney->currency( $this->getAccountParam( 'twocheckout_currency' ) );
		if ( $currency != $twocoCurrency ) {
			// check if conversion already done for this basket:
			$twocheckout_conversions		=	$paymentBasket->getParam( 'twocheckout_conversions', null, 'integrations' );
			if ( $twocheckout_conversions ) {
				$existingConversions		=	explode( ',', $twocheckout_conversions );
				foreach ($existingConversions as $v ) {
					if ( cbStartOfStringMatch( $v, $currency . sprintf( '%.5f', $amount ) . '=' ) ) {
						// yes found the conversion !
						$parts				=	explode( '=', $v );
						return sprintf( '%.2f', $parts[1] );
					}
				}
			}
			// conversion not found, convert:
			$_CBPAY_CURRENCIES				=	cbpaidApp::getCurrenciesConverter();
			$convertedAmount				=	$_CBPAY_CURRENCIES->convertCurrency( $currency, $twocoCurrency, $amount );
			// store conversion:
			$twocheckout_conversions		.=	( $twocheckout_conversions ? ',' : '' ) . $currency . sprintf( '%.5f', $amount ) . '=' . $convertedAmount;
			$paymentBasket->setParam( 'twocheckout_conversions', $twocheckout_conversions, 'integrations' );
			$paymentBasket->store();
			$amount							=	$convertedAmount;
		}
		return sprintf( '%.2f', $amount );
	}

	/**
	 * Convert from the single 2co currency to the baskets' currency, in '%.2f' format
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  float                $amount         if NULL: take $paymentBasket->mc_gross
	 * @param  string               $currency       if NULL: take $paymentBasket->mc_currency
	 * @return float                                $amount in gateway's currency
	 */
	private function _fromGatewayCurrency( $paymentBasket, $amount, $currency = null )
	{
		if ( $currency === null ) {
			$currency						=	$paymentBasket->mc_currency;
		}
		if ( $amount === null ) {
			$amount							=	$paymentBasket->mc_gross;
		}
		$cbpaidMoney						=	cbpaidMoney::getInstance();
		$twocoCurrency						=	$cbpaidMoney->currency( $this->getAccountParam( 'twocheckout_currency' ) );
		if ( $currency != $twocoCurrency ) {
			// check if conversion already done for this basket:
			$largestConversion				=	'0.0';
			$largestOriginal				=	'0.0';
			$twocheckout_conversions		=	$paymentBasket->getParam( 'twocheckout_conversions', null, 'integrations' );
			if ( $twocheckout_conversions ) {
				$existingConversions		=	explode( ',', $twocheckout_conversions );
				foreach ($existingConversions as $v ) {
					$parts					=	explode( '=', $v );
					if ( cbStartOfStringMatch( $v, $currency ) && ( sprintf( '%.2f', $amount ) == sprintf( '%.2f', $parts[1] ) ) ) {
						// yes found the conversion !
						return sprintf( '%.2f', substr( $parts[0], strlen( $currency ) ) );
					}
					if ( ( (float) $parts[1] ) > ( (float) $largestConversion ) ) {
						$largestConversion	=	$parts[1];
						$largestOriginal	=	substr( $parts[0], strlen( $currency ) );
					}
				}
			}
			// conversion not found:
			if ( $largestConversion ) {
				// try converting first on any stored rate available:
				$amount						=	$amount * ( $largestOriginal / $largestConversion );
			} else {
				// otherwise take latest conversion rate at last resort (that case should really not be happening):
				$_CBPAY_CURRENCIES			=	cbpaidApp::getCurrenciesConverter();
				$amount						=	$_CBPAY_CURRENCIES->convertCurrency( $twocoCurrency, $currency, $amount );
			}
		}
		return sprintf( '%.2f', $amount );
	}

	/**
	 * Checks the POST hash for a 2co INS notification
	 *
	 * @param  array  $postdata
	 * @return boolean
	 */
	private function _checkHash_INS( $postdata )
	{
		$secretword		=	$this->getAccountParam( 'twocheckout_secretword' );
		$account		=	trim( $this->getAccountParam( 'twocheckout_account' ) );
		$sale_id		=	cbGetParam( $postdata, 'sale_id' );
		$vendor_id		=	cbGetParam( $postdata, 'vendor_id' );
		$invoice_id		=	cbGetParam( $postdata, 'invoice_id' );
		$mdHash			=	cbGetParam( $postdata, 'md5_hash' );
		if ( $account !== $vendor_id ) {
			// missmatch of account and vendor_id: although this might be a problem, an account # change might occur, so log error, but not fatal:
			$null		=	null;
			$this->_setLogErrorMSG( 3, $null, sprintf( '2checkout: INS mismatch warning: 2Checkout Vendor Account Number set "%s" does not match the received account number "%s".', $account, $vendor_id ), CBTxt::T("2Checkout vendor account mismatch error: Please contact site administrator to check his error logs and settings.") );
		}
		$calculatedHash	=	strtoupper( md5( $sale_id . $vendor_id . $invoice_id . $secretword ) );
		if ( $mdHash !== $calculatedHash ) {
			// hash missmatch: log the hack attempt, and deny:
			$null		=	null;
			$this->_setLogErrorMSG( 3, $null, '2checkout: INS hash error: Fraud attempt detected or 2Checkout secret word setting wrong.', CBTxt::T("2Checkout return hash error: Please contact site administrator to check his error logs.") );
			return false;
		}
		return true;
	}

	/**
	 * Checks the POST hash for a 2co return to website
	 *
	 * @param  array  $postdata
	 * @return boolean
	 */
	private function _checkHash_Return( $postdata )
	{
		$secretword		=	$this->getAccountParam( 'twocheckout_secretword' );
		$account		=	trim( $this->getAccountParam( 'twocheckout_account' ) );
		if ( ! $this->getAccountParam( 'twocheckout_demosales' ) ) {
			$sale_id	=	cbGetParam( $postdata, 'order_number' );
		} else {
			$sale_id	=	1;		// in demo mode, the hash is computed with a order_number of 1.
		}
		$vendor_id		=	cbGetParam( $postdata, 'sid' );
		$total			=	cbGetParam( $postdata, 'total', '' );
		$mdHash			=	cbGetParam( $postdata, 'key', '' );
		if ( $account !== $vendor_id ) {
			// missmatch of account and vendor_id: although this might be a problem, an account # change might occur, so log error, but not fatal:
			$null		=	null;
			$this->_setLogErrorMSG( 3, $null, sprintf( '2checkout: return mismatch warning: 2Checkout Vendor Account Number set "%s" does not match the received account number "%s".', $account, $vendor_id ), CBTxt::T("2Checkout vendor account mismatch error: Please contact site administrator to check his error logs and settings.") );
			return false;
		}
		$calculatedHash	=	strtoupper( md5( $secretword . $vendor_id . $sale_id . $total ) );
		if ( $mdHash !== $calculatedHash ) {
			// hash missmatch: log the hack attempt, and deny:
			$null		=	null;
			$this->_setLogErrorMSG( 3, $null, '2checkout: return hash error: Fraud attempt detected or 2Checkout secret word setting wrong.', CBTxt::T("2Checkout return hash error: Please contact site administrator to check his error logs.") );
			return false;
		}
		return true;
	}

	/**
	 * Checks that Invoice-level-message INS is not a fraud
	 *
	 * @param  array                $postdata
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return boolean
	 */
	private function _verifyReturnFraud( $postdata, $paymentBasket )
	{
		// check invoice id:
		if ( cbGetParam( $postdata, 'merchant_order_id' ) != $paymentBasket->id ) {
			return sprintf( '2Checkout: Return Fraud detected: Userid: %d, Returned merchant_order_id: %s, Payment Basket id: %s', $paymentBasket->user_id, cbGetParam( $postdata, 'merchant_order_id' ), $paymentBasket->id );
		}
		// check amount:
		// Convert to the single 2co currency:
		$total				=	$this->_toGatewayCurrency( $paymentBasket );
		if ( cbGetParam( $postdata, 'total' ) != $total ) {
			return sprintf( '2Checkout: Return Fraud detected: Userid: %d, Returned total: %s, Payment Basket gross total: %s', $paymentBasket->user_id, cbGetParam( $postdata, 'total' ), $paymentBasket->mc_gross );
		}
		// check currency: No info, can't check !
		return true;
	}

	/**
	 * Verify API Return against fraud
	 *
	 * @param  array                $postdata
	 * @param  SimpleXMLElement     $saleRecord
	 * @param  SimpleXMLElement     $invoice
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return boolean|string
	 */
	private function _verifyReturnAPIFraud( $postdata, $saleRecord, $invoice, $paymentBasket )
	{
		if ( cbGetParam( $postdata, 'order_number' ) == (string) $saleRecord->sale_id ) {
			if ( cbGetParam( $postdata, 'merchant_order_id' ) == (string) $invoice->vendor_order_id ) {
				if ( cbGetParam( $postdata, 'total' ) == (string) $invoice->vendor_total ) {
					if ( ( (string) $this->_getReqParam( 'recurring', 0 ) ) == (string) $invoice->recurring ) {
						return true;
					} else {
						$error	=	sprintf( 'Returned recurring: %s, API-checked recuring: %s', cbGetParam( $postdata, 'recurring' ), (string) $invoice->recurring );
					}
				} else {
					$error	=	sprintf( 'Returned total: %s, API-checked vendor_total: %s', cbGetParam( $postdata, 'total' ), (string) $invoice->vendor_total );
				}
			} else {
				$error	=	sprintf( 'Returned merchant_order_id: %s, API-checked vendor_order_id: %s', cbGetParam( $postdata, 'merchant_order_id' ), (string) $invoice->vendor_order_id );
			}
		} else {
			$error	=	sprintf( 'Returned order_number: %s, API-checked sale_id: %s', cbGetParam( $postdata, 'order_number' ), (string) $saleRecord->sale_id );
		}
		return sprintf( '2Checkout: Return Fraud detected: Userid: %d, %s', $paymentBasket->user_id, $error );
	}

	/**
	 * Checks that Invoice-level-message INS is not a fraud
	 *
	 * @param  array                $postdata
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return boolean
	 */
	private function _verifyINSfraud_invoice( $postdata, $paymentBasket )
	{
		// check invoice id: ALREADY CHECKED ABOVE ! :
		/*
		if ( cbGetParam( $postdata, 'invoice_id' ) != $paymentBasket->invoice ) {
			return sprintf( '2Checkout: INS Fraud detected: invoice_id: %d not corresponding to basket invoice number %s', cbGetParam( $postdata, 'invoice_id' ), $paymentBasket->invoice );
		}
		*/
		// check amount:
		// Convert to the single 2co currency:
		$total				=	$this->_toGatewayCurrency( $paymentBasket );
		if ( cbGetParam( $postdata, 'invoice_list_amount' ) != $total ) {
			return sprintf( '2Checkout: INS Fraud detected: invoice_list_amount: %s not corresponding to basket total expressed in gateway currency: %s', cbGetParam( $postdata, 'invoice_list_amount' ), $total );
		}
		// check currency:
		$cbpaidMoney			=	cbpaidMoney::getInstance();
		$twocoCurrency			=	$cbpaidMoney->currency( $this->getAccountParam( 'twocheckout_currency' ) );
		if ( cbGetParam( $postdata, 'list_currency' ) != $twocoCurrency ) {
			// Nope they do not match either: it may be a fraud...
			return sprintf( '2Checkout: INS Fraud detected: list_currency: %s not corresponding to gateway currency: %s', cbGetParam( $postdata, 'list_currency' ), $twocoCurrency );
		}
		return true;
	}

	/**
	 * Checks that Invoice-level-message INS is not a fraud
	 *
	 * @param  array                $postdata
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return boolean
	 */
	private function _verifyINSfraud_refund( $postdata, $paymentBasket )
	{
		// check amount:
		// Convert to the single 2co currency:
		$total				=	$this->_toGatewayCurrency( $paymentBasket );
		if ( cbGetParam( $postdata, 'item_list_amount_1' ) != $total ) {
			return sprintf( '2Checkout: INS Fraud detected: invoice_list_amount: %s not corresponding to basket total expressed in gateway currency: %s', cbGetParam( $postdata, 'invoice_list_amount' ), $total );
		}
		// check currency:
		$cbpaidMoney			=	cbpaidMoney::getInstance();
		$twocoCurrency			=	$cbpaidMoney->currency( $this->getAccountParam( 'twocheckout_currency' ) );
		if ( cbGetParam( $postdata, 'list_currency' ) != $twocoCurrency ) {
			// Nope they do not match either: it may be a fraud...
			return sprintf( '2Checkout: INS Fraud detected: list_currency: %s not corresponding to gateway currency: %s', cbGetParam( $postdata, 'list_currency' ), $twocoCurrency );
		}
		return true;
	}

	/**
	 * Converts YYYY-MM-DD HH:mm:SS date to IPN payment_date 'H:i:s M d, Y T' paypal style date with timezone.
	 *
	 * @param  string|null  $dateEST
	 * @return string       'H:i:s M d, Y T'
	 */
	private function _dateESTtoDateTZ( $dateEST )
	{
		if ( is_string( $dateEST ) ) {
			list($y, $c, $d, $h, $m, $s)	=	sscanf( $dateEST, '%d-%d-%d %d:%d:%d' );
			$convMonth						=	array( 1 => 'Jan', 2 => 'Feb', 3 => 'Mar', 4 => 'Apr', 5 => 'May', 6 => 'Jun', 7 => 'Jul', 8=> 'Aug', 9 => 'Sep', 10 => 'Oct', 11 => 'Nov', 12 => 'Dec' );
			return sprintf( '%02d:%02d:%02d %s %02d, %04d EST', $h, $m, $s, $convMonth[$c], $d, $y );	// Paypal-style
		} else {
			global $_CB_framework;
			return gmdate( 'H:i:s M d, Y T', $_CB_framework->now() );		// Paypal-style
		}
	}

	/**
	 * Instanciator initializing 2co API instance according to gateway parameters
	 *
	 * @return cbpaidtwocheckoutAPI
	 */
	private function _api_getInstance( )
	{
		static $api				=	null;

		if ( ! $api ) {
			$api_username		=	trim( $this->getAccountParam( 'twocheckout_api_username' ) );
			$api_password		=	$this->getAccountParam( 'twocheckout_api_password' );
			if ( $api_username && $api_password ) {
				cbpaidApp::import( 'processors.twocheckout.twocheckout_salesapi' );
				$api			=	new cbpaidtwocheckoutAPI( $api_username, $api_password );
				$api->apiUrl	=	$this->gatewayUrl( 'api' );
			}
		}
		return $api;
	}

	/**
	 * Gets information of company of $vendor_id
	 *
	 * @param  string                               $vendor_id
	 * @return SimpleXMLElement|string|boolean
	 */
	public function _api_getCompanyInfo( $vendor_id )
	{
		$coAPI					=	$this->_api_getInstance();
		if ( $coAPI ) {
			$raw_data			=	null;
			$companyInfo		=	$coAPI->_api_detail_company_info( $vendor_id, $raw_data );
		} else {
			$companyInfo		=	false;
		}
		return $companyInfo;
	}

	/**
	 * Returns the EST-timezoned date of the sale from $saleRecord,
	 * or if there is no date_placed in the $saleRecord, of the current time
	 *
	 * @param  SimpleXMLElement  $saleRecord
	 * @return string
	 */
	private function _api_saleDate( $saleRecord )
	{
		if ( $saleRecord && isset( $saleRecord->date_placed ) ) {
			return $this->_dateESTtoDateTZ( (string) $saleRecord->date_placed );
		} else {
			return $this->_dateESTtoDateTZ( null );
		}
	}

	/**
	 * Checks in $saleRecord if there are comments on fraud status
	 * @param  SimpleXMLElement    $saleRecord
	 * @param  SimpleXMLElement    $invoice
	 * @param  string              $invoice_status  IN/OUT: Invoice status (to be updated with $saleRecord latest info)
	 * @param  string              $fraud_status    IN/OUT: Fraud status (to be updated with $saleRecord latest info)
	 * @return null|string         latest fraud_status from API ( 'pass', 'fail', 'wait', null means probably 'wait' )
	 */
	private function _api_updateInvoiceFraudStatus( $saleRecord, $invoice, &$invoice_status, &$fraud_status )
	{
		// determine latest fraud_status from saleRecord comments (as suggested by 2checkout development support):
		$last_status			=	null;
		if ( isset( $saleRecord->comments ) && ( count( $saleRecord->comments ) > 0 ) ) {
			$matches			=	null;
			$last_timestamp		=	'';
			foreach ( $saleRecord->comments as $comment ) {
				if ( isset( $comment->comment ) && isset( $comment->timestamp ) ) {
					if ( ( ! isset( $comment->username ) ) || ( ( (string) $comment->username ) == 'fraudsystem' ) ) {
						if ( ( (string) $comment->timestamp ) > $last_timestamp ) {
							if ( preg_match( '/^Sale reviewed by fraud: status is (.+)$/i', (string) $comment->comment, $matches ) ) {
								$last_timestamp		=	(string) $comment->timestamp;
								$last_status		=	$matches[1];
							}
						}
					}
				}
			}
		}
		// update params with latest information from API saleRecord:
		$invoice_status			=	(string) $invoice->status;
		if ( $last_status ) {
			$fraud_status		=	$last_status;
		}
		return $last_status;
	}

	/**
	 * Checks in $saleRecord if there are comments on Refunds starting with: 'Refund Issued'
	 *
	 * <comments>
	 *	<changed_by_ip>123.456.789.123</changed_by_ip>
	 *	<comment>Refund Issued On Lineitem: Refunding test-payment by myself.</comment>
	 *	<timestamp>2009-12-11 11:54:08</timestamp>
	 *	<username>theShopUsername</username>
	 * </comments>
	 *
	 * @param  SimpleXMLElement  $saleRecord
	 * @return null|string         latest fraud_status from API ( 'pass', 'fail', 'wait', null means probably 'wait' )
	 */
	private function _api_InvoiceRefundedStatus( $saleRecord )
	{
		if ( isset( $saleRecord->comments ) && ( count( $saleRecord->comments ) > 0 ) ) {
			foreach ( $saleRecord->comments as $comment ) {
				if ( isset( $comment->comment ) ) {
					if ( cbStartOfStringMatch( (string) $comment->comment, 'Refund Issued' ) ) {
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Verifies the posted INS against the API record
	 *
	 * @param  array               $postdata
	 * @param  SimpleXMLElement    $saleRecord
	 * @param  SimpleXMLElement    $invoice
	 * @param  string              $message_type
	 * @param  boolean             $invoiceLevelMessage
	 * @return boolean|string      TRUE: ok, otherwise error message.
	 */
	private function verifyINSsalerecord_invoice( $postdata, $saleRecord, $invoice, /** @noinspection PhpUnusedParameterInspection */ $message_type, $invoiceLevelMessage )
	{
		// Mandatory post values for all INS messages:
		$toCheck	=	array(	'vendor_id'			=>	(string) $invoice->vendor_id,
								'sale_id'			=>	(string) $saleRecord->sale_id,
								'sale_date_placed'	=>	(string) $saleRecord->date_placed,
								'vendor_order_id'	=>	(string) $invoice->vendor_order_id,
								'invoice_id'		=>	(string) $invoice->invoice_id
						);
		if ( $invoiceLevelMessage ) {
			// Mandatory post values for INVOICE-LEVEL INS messages:
			$toCheck['invoice_list_amount']	=	(string) $invoice->vendor_total;
			// the currency symbol is missing for the list/vendor_total so compare USD value
			$toCheck['invoice_usd_amount']	=	(string) $invoice->usd_total;
			// the recurring value is '0' or '1', and not always present if 0 in the INS post, defaulting to 0:
			$postName						=	'recurring';
			if ( ( (string) cbGetParam( $postdata, $postName, 0 ) ) != (string) $invoice->recurring ) {
				return sprintf( '2Checkout: INS Fraud detected: the INS post record "%s" does not match the 2checkout system accessed by the API.', $postName );
			}
		}
		// check each of the mandatory key values corresponding to the INS:
		foreach ($toCheck as $postName => $value ) {
			if ( cbGetParam( $postdata, $postName ) != $value ) {
				return sprintf( '2Checkout: INS Fraud detected: the INS post record "%s" does not match the 2checkout system accessed by the API.', $postName );
			}
		}
		// special case to accept INS messages resent later, where the invoice_status value chaged:
		if ( $invoiceLevelMessage ) {
			if ( cbGetParam( $postdata, 'invoice_status' ) != (string) $invoice->status ) {
				// this one could change... if it changed, use the status from the API to avoid fraud: simply log the difference at INFO level:
				$this->_setLogErrorMSG( 6, null, sprintf('2Checkout: Info: Invoice status in INS "%s" differs from API sales record status "%s"', cbGetParam( $postdata, 'invoice_status' ), (string) $invoice->status ), null );
				$postdata['invoice_status']		=	(string) $invoice->status;
			}
		}
		return true;
	}

	/**
	 * Handles a 2co INS notification
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  array                $postdata
	 * @return boolean|null
	 */
	private function _handle2coINS( /** @noinspection PhpUnusedParameterInspection */ $paymentBasket, $postdata )
	{
		global $_CB_database;

		// check that required params are here:
		if (!isset( $postdata['message_type'],
					$postdata['message_description'],
					$postdata['timestamp'],
					$postdata['md5_hash'],
					$postdata['message_id'],
					$postdata['key_count'],
					$postdata['vendor_id'],
					$postdata['sale_id'],
					$postdata['sale_date_placed'] ) ) {
			return false;
		}

		$message_type						=	cbGetParam( $postdata, 'message_type' );

		// check MD5 hash:
		if ( ! $this->_checkHash_INS( $postdata ) ) {
			global $_CB_framework;
			$ipn							=	$this->_prepareIpn( 'O', '', '', 'INS Hash failed', $_CB_framework->now(), 'utf-8' );
			$ipn->raw_data					=	'$message_type="' . $message_type . "\";\n"
											.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
											.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n"
											;
			$ipn->store();

			return false;
		}

		// get the mandatory params, such as sale_id:
		$sale_id							=	cbGetParam( $postdata, 'sale_id' );
		$invoice_id							=	cbGetParam( $postdata, 'invoice_id' );
		$paymentTime						=	strtotime( cbGetParam( $postdata, 'sale_date_placed' ) . ' EST' );
		$paymentType						=	cbGetParam( $postdata, 'payment_type' );
		if ( ( ! $sale_id ) || ( ! $paymentTime ) || ( ! $paymentType ) ) {
			return false;
		}

		$invoiceLevelMessage				=	in_array( $message_type, array( 'ORDER_CREATED', 'FRAUD_STATUS_CHANGED', 'INVOICE_STATUS_CHANGED', 'SHIP_STATUS_CHANGED' ) );
		if ( $invoiceLevelMessage ) {
			// invoice-level message:
			$invoice_status					=	cbGetParam( $postdata, 'invoice_status' );
			$fraud_status					=	cbGetParam( $postdata, 'fraud_status' );
			if ( ( ! $invoice_status ) || ( ! $fraud_status ) ) {
				return false;
			}
		} else {
			$invoice_status					=	null;
			$fraud_status					=	null;
		}
		// Log the INS:
		$log_type							=	'I';
		$paymentStatus						=	$this->_paymentStatus( $invoice_status, $fraud_status, ( null ) );
		$reasonCode							=	null;

		$ipn								=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reasonCode, $paymentTime, 'utf-8' );
		$ipn->test_ipn						=	0;		// demo IPNs do not generate INS and do not appear in API at all...
		$ipn->raw_data						=	'$message_type="' . $message_type . "\";\n"
											.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
											.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n"
											;

		// get the official Sale record from 2co:
		$coAPI								=	$this->_api_getInstance();
		if ( $coAPI ) {
			$saleRecord						=	$coAPI->_api_detail_sale( $sale_id, null, $ipn->raw_data );
			if ( is_object( $saleRecord ) ) {
				$verificationResult			=	false;
				// Search invoice in the $saleRecord invoices[] array:
				$invoice					=	null;
				/** @var $invoice SimpleXMLElement */
				foreach ($saleRecord->invoices as $invoice ) {
					if ( $invoice_id == (string) $invoice->invoice_id ) {
						$verificationResult	=	$this->verifyINSsalerecord_invoice( $postdata, $saleRecord, $invoice, $message_type, $invoiceLevelMessage );
						break;
					}
				}
			} else {	// is_string( $saleRecord ):
				$ipn->raw_result			=	'ERROR';
				$ipn->raw_data				=	'$errorMessage="'
											.	(string) $saleRecord
											.	"\";\n"
											.	$ipn->raw_data;
				$ipn->log_type				=	'D';		// IPN payment gateway communication error
				$ipn->store();

				return false;
			}
		} else {
			$saleRecord						=	false;
			$invoice						=	null;
			$verificationResult				=	true;		// for now... more below...
		}

		// handle the message:
		switch ( $message_type ) {
			// INVOICE-LEVEL MESSAGES:
			case 'ORDER_CREATED':			// The order has been created at 2co:
			case 'FRAUD_STATUS_CHANGED':	// The order fraud status has changed (can change multiple times):
			case 'INVOICE_STATUS_CHANGED':	// The order's invoice payment status has changed (typically 'approved' -> 'pending' -> 'deposited')
			case 'SHIP_STATUS_CHANGED':		//TODO : Check if below is correct... :
			// ITEM-LEVEL MESSAGES:
			case 'REFUND_ISSUED':
			case 'RECURRING_INSTALLMENT_SUCCESS':

				// check that the sale_id exists at 2co: done in $tcoAPI->_api_detail_sale() above !

				$paymentBasketId			=	cbGetParam( $postdata, 'vendor_order_id' );
				$paymentBasket				=	new cbpaidPaymentBasket( $_CB_database );
				if ( $paymentBasketId ) {
					$basketLoaded			=	$paymentBasket->load( (int) $paymentBasketId );
				} else {
					$basketLoaded			=	false;
				}
				if ( $basketLoaded ) {
					$ipn->user_id			=	$paymentBasket->user_id;

					// check in $s
					if ( is_object( $saleRecord ) ) {
						$this->_api_updateInvoiceFraudStatus( $saleRecord, $invoice, $invoice_status, $fraud_status );		// $invoice NULL won't happen because of the if.
					}
					if ( $invoiceLevelMessage || ( $message_type == 'RECURRING_INSTALLMENT_SUCCESS' ) ) {
						$paymentStatus		=	$this->_paymentStatus( $invoice_status, $fraud_status, ( $paymentBasket->payment_status ) );		//FIXME: should payment_status not be from the payment record ?
					} else {
						$paymentStatus		=	'Refunded';
					}
					$ipn->payment_status	=	$paymentStatus;
					if ( ( $paymentStatus == 'Pending' ) && ( $fraud_status == 'wait' ) ) {
						$ipn->pending_reason =	'paymentreview';
					}
					$ipn->bindBasket( $paymentBasket );
				}
				$insToIpn					=	array(	'vendor_order_id'		=>	'invoice',
														// 'list_currency'			=>	'mc_currency',		// not directly as gateway currency can be different from basket
														// 'invoice_list_amount'	=>	'mc_gross',
														'sale_id'				=>	'sale_id',
														'invoice_id'			=>	'txn_id',
														'customer_first_name'	=>	'first_name',
														'customer_last_name'	=>	'last_name',
														'customer_name'			=>	'address_name',
														'customer_email'		=>	'payer_email',
														'customer_phone'		=>	'contact_phone',
														'bill_street_address'	=>	'address_street',		// + bill_street_address2 below
														'bill_city'				=>	'address_city',
														'bill_state'			=>	'address_state',
														'bill_postal_code'		=>	'address_zip',
													 );
				foreach ($insToIpn as $k => $v ) {
					$ipn->$v							=	cbGetParam( $postdata, $k );
				}
				$bill_address2							=	cbGetParam( $postdata, 'bill_street_address2' );
				$bill_country_threeLetters				=	cbGetParam( $postdata, 'bill_country' );
				if ( $bill_address2 ) {
					$ipn->address_street				.=	' ' . $bill_address2;
				}
				$ipn->address_country					=	$this->countryToLetters( $bill_country_threeLetters, -3 );
				$ipn->address_country_code				=	$this->countryToLetters( $bill_country_threeLetters, 32 );

				// convert payment_date to Paypal-style:
				if ( $invoiceLevelMessage && ( ! ( $message_type == 'RECURRING_INSTALLMENT_SUCCESS' ) ) && ( ! ( cbGetParam( $postdata, 'item_rec_install_billed_1', 0 ) > 1 ) ) ) {
					$ipn->payment_date					=	$this->_dateESTtoDateTZ( cbGetParam( $postdata, 'sale_date_placed' ) );
				} else {
					$ipn->payment_date					=	$this->_dateESTtoDateTZ( cbGetParam( $postdata, 'timestamp' ) );
				}
				$recurring								=	cbGetParam( $postdata, 'recurring', 0 );

				if ( $basketLoaded ) {
					if ( $invoiceLevelMessage || ( $message_type == 'REFUND_ISSUED' ) ) {
						if ( $message_type != 'REFUND_ISSUED' ) {
							$verificationResult			=	$this->_verifyINSfraud_invoice( $postdata, $paymentBasket );
						} else {
							$verificationResult			=	$this->_verifyINSfraud_refund( $postdata, $paymentBasket );
						}
						if ( $recurring ) {
							if ( $message_type == 'ORDER_CREATED' ) {
								$ipn->txn_type			=	'subscr_signup';
								// ipn->subscr_id		=	???;
								$ipn->subscr_date		=	$ipn->payment_date;
							} else {
								$ipn->txn_type			=	'subscr_payment';
							}
						} else {
							$ipn->txn_type				=	'web_accept';
						}
					} elseif ( $message_type == 'RECURRING_INSTALLMENT_SUCCESS' ) {
						$ipn->txn_type					=	'subscr_payment';
					}
				} else {
					// if basket is not loaded: still check the INS so that we can verify and log the notification:
					$verificationResult					=	true;
					$ipn->txn_type						=	'web_accept';
				}
				if ( $verificationResult === true ) {
					if ( $basketLoaded  ) {
						if ( $message_type == 'REFUND_ISSUED' ) {
							$amount_2co_currency		=	cbGetParam( $postdata, 'item_list_amount_1' );
							$ammount_basket_currency	=	$this->_fromGatewayCurrency( $paymentBasket, $amount_2co_currency );
							$ipn->mc_currency			=	$paymentBasket->mc_currency;
							$ipn->mc_gross				=	- $ammount_basket_currency;
							if ( ( $invoice_status == 'approved' ) && is_object( $saleRecord ) ) {
								// while the invoice_status is only approved and not pending or deposited, the 2co fees are refunded too:
								$ipn->mc_fee			=	- $this->_fromGatewayCurrency( $paymentBasket, (float) ( (string) $invoice->fees_2co ) );
							}
						}
						elseif ( in_array( $invoice_status, array( 'approved', 'pending', 'deposited' ) ) ) {
							// this means basket total has been indeed approved for payment: we can't use the 'total' directly as it is in the gateway's currency.
							$ipn->mc_currency			=	$paymentBasket->mc_currency;
							$ipn->mc_gross				=	$paymentBasket->mc_gross;
							if ( $recurring ) {
								$ipn->mc_amount1		=	$paymentBasket->mc_amount1;
								$ipn->mc_amount3		=	$paymentBasket->mc_amount3;
								$ipn->period1			=	$paymentBasket->period1;
								$ipn->period3			=	$paymentBasket->period3;
							}
							if ( is_object( $saleRecord ) ) {
								if ( $invoiceLevelMessage || ( $message_type == 'RECURRING_INSTALLMENT_SUCCESS' ) ) {
									// first invoice and payment OR subsquent auto-recurring payments: find the corresponding invoice in the API:
									$fees				=	(string) $invoice->fees_2co;
								} else {
									$fees				=	null;
								}
								if ( $fees !== null ) {
									$ipn->mc_fee		=	$this->_fromGatewayCurrency( $paymentBasket, $fees );
								}
							}
						}
					} else {
						$cbpaidMoney					=	cbpaidMoney::getInstance();
						$twocoCurrency					=	$cbpaidMoney->currency( $this->getAccountParam( 'twocheckout_currency' ) );
						if ( $message_type == 'REFUND_ISSUED' ) {
							$amount_2co_currency		=	- cbGetParam( $postdata, 'item_list_amount_1' );
						} else {
							$amount_2co_currency		=	cbGetParam( $postdata, 'invoice_list_amount' );
						}
						$ipn->mc_currency				=	$twocoCurrency;
						$ipn->mc_gross					=	$amount_2co_currency;
					}
					$ipn->raw_result					=	'VERIFIED';
					if ( $basketLoaded ) {
						// lock the basket to 2co:
						$this->_bindIpnToBasket( $ipn, $paymentBasket );
						$autorecurring_type				=	( $recurring ? 2 : 0 );		// 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date
						$autorenew_type					=	( $recurring ? 2 : 0 );		// 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration
						$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $paymentStatus, $ipn, 1, $autorecurring_type, $autorenew_type, false );
					}
				} else {
					$ipn->raw_result					=	'ERROR';
					$ipn->raw_data						=	'$errorMessage="'
														.	(string) $verificationResult
														.	"\";\n"
														.	$ipn->raw_data;
					$ipn->log_type						=	'K';		// IPN notification not authenticating at payment gateway
				}
				break;

			// ITEM-LEVEL MESSAGES:
	//		case 'REFUND_ISSUED':
	//			break;
//			case 'RECURRING_INSTALLMENT_SUCCESS':		//TODO
//
//				if( ! $ipn->store() ) {
//					trigger_error( '2co log store error:' . htmlspecialchars( $_CB_database->stderr( true ) ), E_USER_NOTICE );
//				}
//				break;
			case 'RECURRING_INSTALLMENT_FAILED':
			case 'RECURRING_STOPPED':
			case 'RECURRING_COMPLETE':
			case 'RECURRING_RESTARTED':
			default:
				break;
		}
		$ipn->store();

		return null;
	}

	/**
	 * Handles a 2co Return to site:
	 * $postdata holds:
	 * order_number           2Checkout.com order number
	 * merchant_order_id      Payment basket id
	 * credit_card_processed  Y if successful (Approved).
	 * key                    the MD5 hash used to verify that the sale came from one of our servers
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  array                $postdata
	 * @return boolean
	 */
	private function _handle2coReturn( $paymentBasket, $postdata )
	{
		global $_CB_framework;

		// check that required params are here:
		if (!isset( $postdata['order_number'],
					$postdata['merchant_order_id'],
					$postdata['total'],
					$postdata['credit_card_processed'],
					$postdata['key']
					) ) {
			return false;
		}

		// check MD5 hash:
		if ( ! $this->_checkHash_Return( $postdata ) ) {
			return false;
		}

		// get the mandatory params, such as sale_id:
		$sale_id							=	cbGetParam( $postdata, 'order_number' );
		$paymentBasketId					=	cbGetParam( $postdata, 'merchant_order_id' );
		$total								=	cbGetParam( $postdata, 'total', '' );
		$credit_card_processed				=	cbGetParam( $postdata, 'credit_card_processed', '' );
		$key								=	cbGetParam( $postdata, 'key', '' );
		$demo_mode							=	cbGetParam( $postdata, 'demo', '' );
		// We are getting this from the IPN result instead:	$this->getAccountParam( 'normal_gateway' );

		if ( ( ! $sale_id ) || ( ! $paymentBasketId ) || ( ! $total ) || ( ! $credit_card_processed ) || ( ! $key ) ) {
			// don't even bother if mandatory params are missing:
			return false;
		}

		// Temporarily use what we have until check with API:
		$paymentTime						=	$_CB_framework->now();
		$paymentType						=	'credit card';
		$invoice_status						=	( $credit_card_processed === 'Y' ? 'approved' : ( $credit_card_processed === 'K' ? 'denied' : '' ) );
		$fraud_status						=	'wait';
		// Log the return (PDT):
		$log_type							=	'R';					// PDT
		$paymentStatus						=	$this->_paymentStatus( $invoice_status, $fraud_status, ( null ) );
		$reasonCode							=	null;

		$ipn								=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reasonCode, $paymentTime, 'utf-8' );
		$ipn->payment_type					=	$this->_getPayment_type_from_method( cbGetParam( $postdata, 'pay_method', '' ) );
		$ipn->test_ipn						=	( $demo_mode == 'Y' ? 1 : 0 );
		$ipn->raw_data						=	'$message_type="RETURN_TO_SITE";' . "\n"
											.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
											.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n"
											;
		$basketLoaded						=	$paymentBasket->load( (int) $paymentBasketId );
		if ( $basketLoaded ) {
			$ipn->user_id					=	$paymentBasket->user_id;
			$ipn->payment_status			=	$this->_paymentStatus( $invoice_status, $fraud_status, ( $paymentBasket->payment_status ) );
			if ( $ipn->payment_status == 'Pending' ) {
				$ipn->pending_reason		=	'paymentreview';
			}
			$ipn->bindBasket( $paymentBasket );
		}
		// array ( 'sid' => '1280154', 'ship_zip' => '', 'key' => '87E479CCF42F4A8ED537AF6068895BCA', 'state' => 'XX', 'email' => 'xxx@example.com', 'submit_y' => '21', 'order_number' => '4169073991', 'cart_id' => '1070', 'lang' => 'en', 'ship_state' => '', 'ptcoid' => '183fcd4f9c3f130ea430787afb2d4e78', 'total' => '0.68', 'ship_street_address2' => '', 'credit_card_processed' => 'Y', 'zip' => '1010', 'ship_name' => '', 'ship_method' => '', 'fixed' => 'Y', 'ship_country' => '', 'city' => 'Lausanne', 'street_address' => 'Road 1', 'ship_city' => '', 'cart_order_id' => '1070', 'country' => 'CHE', 'merchant_order_id' => 'U96613078773759', 'ip_country' => 'Switzerland', 'ship_street_address' => '', 'demo' => '', 'submit_x' => '148', 'middle_name' => '', 'pay_method' => 'CC', 'phone' => '0217654321 ', 'x_receipt_link_url' => 'http://www.example.com/~beat/1.0/index.php?option=com_comprofiler&task=pluginclass&user=37815&plugin=cbpaidsubscriptions&cbpaidsubscriptionsmethod=twocheckout&cbpaidsubscriptionsgacctno=11&cbpaidsubscriptionsbasket=1070&result=success&cbpid=cbp4b1fdff0ebdc9586614008', 'street_address2' => '', 'card_holder_name' => 'John Doe', )
		$insToIpn							=	array(	'merchant_order_id'	=>	'invoice',
												// 'list_currency'			=>	'mc_currency',
												// 'total'				=>	'mc_gross',			// do not use this as it can have another currency in it.
												'order_number'		=>	'sale_id',
												'invoice_id'		=>	'txn_id',
												// card_holder_name / middle_name	=> ...
												// 'customer_first_name'	=>	'first_name',
												// 'customer_last_name'	=>	'last_name',
												'card_holder_name'	=>	'address_name',
												'email'				=>	'payer_email',
												'phone'				=>	'contact_phone',
												'street_address'	=>	'address_street',		// + bill_street_address2 below
												'city'				=>	'address_city',
												'state'				=>	'address_state',
												'zip'				=>	'address_zip'
												// 'ip_country'		: 'Switzerland'		=>	???
												// 'pay_method' : 'CC'
											 );
		foreach ($insToIpn as $k => $v ) {
			$ipn->$v							=	cbGetParam( $postdata, $k );
		}
		$bill_address2							=	cbGetParam( $postdata, 'street_address2' );
		$bill_country_threeLetters				=	cbGetParam( $postdata, 'country' );
		if ( $bill_address2 ) {
			$ipn->address_street				.=	' ' . $bill_address2;
		}
		$ipn->address_country					=	$this->countryToLetters( $bill_country_threeLetters, -3 );
		$ipn->address_country_code				=	$this->countryToLetters( $bill_country_threeLetters, 32 );

		// 2co does not return first_name and last_name but middle_name and card_holder_name (with 2 spaces separating first and last name when there is no middle-name:
		// try to guess first and last name:
		$middle_name							=	cbGetParam( $postdata, 'middle_name' );
		if ( $middle_name ) {
			$card_names							=	explode( ' ' . $middle_name[0] . ' ', cbGetParam( $postdata, 'card_holder_name' ), 2 );
		} else {
			$card_names							=	explode( '  ', cbGetParam( $postdata, 'card_holder_name' ), 2 );
			if ( count( $card_names ) < 2 ) {
				$card_names						=	explode( ' ', cbGetParam( $postdata, 'card_holder_name' ), 2 );
				if ( count( $card_names ) < 2 ) {
					$card_names					=	array( '', cbGetParam( $postdata, 'card_holder_name' ) );
				}
			}
		}
		$ipn->first_name						=	$card_names[0];
		$ipn->last_name							=	$card_names[1];

		// in return, 2co does not say if it's a recurring payment or not, so we have to tell it ourselves from the URL:
		// note: as 2co has events for re-occurring payments, this is not a vuln.
		$recurring								=	$this->_getReqParam( 'recurring' );

		if ( $recurring ) {
			$ipn->txn_type						=	'subscr_signup';			//TODO: check if 'subscr_payment' is even needed...
		} else {
			$ipn->txn_type						=	'web_accept';
		}
		if ( $basketLoaded ) {
			$verificationResult					=	$this->_verifyReturnFraud( $postdata, $paymentBasket );
			if ( $verificationResult === true ) {
				if ( in_array( $invoice_status, array( 'approved', 'pending', 'deposited' ) ) ) {
					// this means basket total has been indeed approved for payment: we can't use the 'total' directly as it is in the gateway's currency.
					$ipn->mc_currency			=	$paymentBasket->mc_currency;
					$ipn->mc_gross				=	$paymentBasket->mc_gross;
					if ( $recurring ) {
						$ipn->mc_amount1		=	$paymentBasket->mc_amount1;
						$ipn->mc_amount3		=	$paymentBasket->mc_amount3;
						$ipn->period1			=	$paymentBasket->period1;
						$ipn->period3			=	$paymentBasket->period3;
					}
				}
				// get the official Sale record from 2co:
				$coAPI							=	$this->_api_getInstance();
				if ( $coAPI ) {
					$saleRecord					=	$coAPI->_api_detail_sale( $sale_id, null, $ipn->raw_data );
					// Refine with saleRecord if present:
					if ( is_object( $saleRecord ) && isset( $saleRecord->invoices[0] ) ) {
						$invoice				=	$saleRecord->invoices[0];
						$ipn->txn_id			=	(string) $invoice->invoice_id;
						$this->_api_updateInvoiceFraudStatus( $saleRecord, $invoice, $invoice_status, $fraud_status );
						$ipn->payment_date		=	$this->_api_saleDate( $saleRecord );
						if ( isset( $saleRecord->customer->pay_method->method ) ) {
							$ipn->payment_type	=	(string) $saleRecord->customer->pay_method->method;
						}
						$ipn->payment_status	=	$this->_paymentStatus( $invoice_status, $fraud_status, ( $paymentBasket->payment_status ) );
						$verificationResult		=	$this->_verifyReturnAPIFraud( $postdata, $saleRecord, $invoice, $paymentBasket );
						if ( $verificationResult === true ) {
							if ( ! $this->_api_InvoiceRefundedStatus( $saleRecord ) ) {
								$ipn->mc_fee	=	$this->_fromGatewayCurrency( $paymentBasket, (string) $invoice->fees_2co );
								$ipn->raw_result =	'VERIFIED';
							} else {
								$ipn->log_type	=	'1';		// PDT notification not matching state at payment gateway
								$ipn->raw_result =	'FRAUD';
								$ipn->raw_data	=	'$errorMessage="'
												.	'Amount already refunded: trial to repeat the return to basket while refunded.'
												.	"\";\n"
												.	$ipn->raw_data;
								$this->_setLogErrorMSG( 3, $ipn, '2Checkout: Fraud: return to sale of an already refunded sale: ' . $verificationResult, CBTxt::T("Sorry, this purchase has already been refunded by 2Checkout, and can't be used. Please contact your administrator to check his payment logs.") );
							}
						} else {
							$ipn->log_type		=	'L';		// PDT notification not authenticating at payment gateway
							$ipn->raw_result	=	'FRAUD';
							$ipn->raw_data		=	'$errorMessage="'
												.	(string) $verificationResult
												.	"\";\n"
												.	$ipn->raw_data;
							$this->_setLogErrorMSG( 3, $ipn, '2Checkout: Fraud: return does not match sales record by API: ' . $verificationResult, CBTxt::T("Sorry, the payment does not match the records at 2Checkout. Please contact your administrator to check his payment logs.") );
						}
					} elseif ( is_string( $saleRecord ) ) {
						// API defined, but API call failed...		//TBD: handle payment best we can ??? tradeoff security/customer friendlyness... if gateway works and record does not exist: ERROR, if gateway down temporarly: maybe NOT ERROR ?
						$ipn->log_type			=	'E';		// PDT payment gateway communication error
						$ipn->raw_result		=	'ERROR';
						$ipn->raw_data			=	'$errorMessage="'
												.	$saleRecord
												.	"\";\n"
												.	$ipn->raw_data;

						$ipn->store();

						$this->_setLogErrorMSG( 3, $ipn, '2Checkout: Error retrieving sales record by API: ' . $saleRecord, CBTxt::T("Sorry no response from 2Checkout for your payment. Please check your email and status later.") );

						return false;
					}
				} else {
					// No API defined: do without it:
					$ipn->raw_result			=	'SUCCESS';
					$ipn->payment_date			=	$this->_dateESTtoDateTZ( null );
				}

				if ( $recurring ) {
					// $ipn->subscr_id			=	???;
					$ipn->subscr_date			=	$ipn->payment_date;
					$ipn->recurring				=	1;
					$ipn->reattempt				=	0;
				}

			} else {
						$ipn->log_type			=	'G';		// PDT Amount/item mismatch / Fraud attempt
						$ipn->raw_result		=	'ERROR';
						$ipn->raw_data			=	'$errorMessage="'
												.	$verificationResult
												.	"\";\n"
												.	$ipn->raw_data;
						$this->_setLogErrorMSG( 3, $ipn, '2Checkout: Fraud detected: basket mismatch: ' . $verificationResult, CBTxt::T("Sorry, the payment does not match the purchased goods. Please contact your administrator to check his payment logs.") );
			}
		} else {
					$ipn->log_type				=	'2';		// PDT Basket non-existant
					$ipn->raw_result			=	'ERROR';
					$ipn->raw_data				=	'$errorMessage="'
												.	'Basket not found'
												.	"\";\n"
												.	$ipn->raw_data;
					$this->_setLogErrorMSG( 3, $ipn, '2Checkout: Basket not found', CBTxt::T("Sorry, the payment does not find your basket. Please contact your administrator to check his payment logs.") );
		}
		$ipn->store();

		// handle the message:
		if ( ( $ipn->raw_result == 'VERIFIED' ) || ( $ipn->raw_result == 'SUCCESS' ) ) {
			$this->_bindIpnToBasket( $ipn, $paymentBasket );
			$autorecurring_type					=	( $recurring ? 2 : 0 );		// 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date
			$autorenew_type						=	( $recurring ? 2 : 0 );		// 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration
			$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, false );
			return null;
		}
		return false;
	}

	/**
	 * USED by XML interface ONLY !!! Renders URL to set in the 2Checkout interface for notifications
	 *
	 * @param  string        $gatewayId  Id of gateway
	 * @return string                    HTML to display
	 */
	public function renderNotifyUrl( /** @noinspection PhpUnusedParameterInspection */ $gatewayId )
	{
		return $this->getNotifyUrl( null );
	}

	/**
	 * USED by XML interface ONLY !!! Renders URL to set in the 2Checkout Sandbox interface for approved return URL
	 *
	 * @param  string        $gatewayId  Id of gateway
	 * @return string                    HTML to display
	 */
	public function renderSuccessUrl( /** @noinspection PhpUnusedParameterInspection */ $gatewayId )
	{
		return $this->getSuccessUrlTwoCheckout( null, 0 );
	}

	/**
	 * parse period into period and type then calculate new period limitations:
	 *
	 * @param array $periodTypeArray
	 * @param int $recurrTimes
	 * @return array
	 */
	private function _periodsLimits( $periodTypeArray, $recurrTimes = 0 )
	{
		// parse period into period and type:
		$p					=	$periodTypeArray[0];
		$t					=	$periodTypeArray[1];

		// change single letter type to full length:
		if ( $t == 'Y' ) {
			$t				=	'Year';
		} elseif ( $t == 'M' ) {
			$t				=	'Month';
		} elseif  ( $t == 'W' ) {
			$t				=	'Week';
		} elseif ( $t == 'D' ) {
			if ( ( $p % 7 == 0 ) && ( $p < 7*99 ) ) {
				// Convert days to weeks:
				$p			=	( $p / 7 );
				$t			=	'Week';
			} elseif ( ( $p % 30 == 0 ) && ( $p < 30*99 ) ) {
				// Convert days to months:
				$p			=	( $p / 30 );
				$t			=	'Month';
			} elseif (  ( $p % 365 == 0 ) && ( $p < 365*99 ) ) {
				// Convert days to years:
				$p			=	( $p / 365 );
				$t			=	'Year';
			} else {
				$w			=	ceil( $p / 7 );
				if ( $w > 99 ) {
					$w		=	99;
				}
				$this->_setLogErrorMSG( 4, null, '2Checkout: Invalid plans duration', CBTxt::T( '2CHECKOUT_INVALID_PLANS_DURATION', '2Checkout: Trying to subscribe [days] days, not supported by 2Checkout. Subscription augmented to [weeks] week.', array( '[days]' => $p, '[weeks]' => $w ) ) );
				$p			=	$w;
				$t			=	'Week';
			}
		} else {
			trigger_error( __CLASS__ . '::' . __FUNCTION__ . ' wrong parameters.', E_USER_WARNING );
		}

		$d					=	$p . ' ' . $t;
		$r					=	( ! $recurrTimes ? 'Forever' : ( $p * $recurrTimes ) . ' ' . $t );

		return array( $d, $r );
	}
}	// end class cbpaidtwocheckout.

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * No methods need to be implemented or overridden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccounttwocheckout extends cbpaidGatewayAccounthostedpage
{
	/**
	 * USED by XML interface ONLY !!! Renders URL for notifications
	 *
	 * @param  string           $gatewayId  Id of gateway
	 * @param  ParamsInterface  $params     Params of gateway
	 * @return string                       HTML to display
	 */
	public function renderNotifyUrl( $gatewayId, /** @noinspection PhpUnusedParameterInspection */ $params )
	{
		/** @var $payClass cbpaidtwocheckout */
		$payClass				=	$this->getPayMean();
		return $payClass->renderNotifyUrl( $gatewayId );
	}

	/**
	 * USED by XML interface ONLY !!! Renders URL for Sandbox approved URL
	 *
	 * @param  string           $gatewayId  Id of gateway
	 * @param  ParamsInterface  $params     Params of gateway
	 * @return string                       HTML to display
	 */
	public function renderSuccessUrl( $gatewayId, /** @noinspection PhpUnusedParameterInspection */ $params )
	{
		/** @var $payClass cbpaidtwocheckout */
		$payClass				=	$this->getPayMean();
		return $payClass->renderSuccessUrl( $gatewayId );
	}

	/**
	 * USED by XML interface ONLY !!! Renders URL for notifications
	 *
	 * @param  string           $gatewayId  Id of gateway
	 * @param  ParamsInterface  $params     Params of gateway
	 * @return string                       HTML to display
	 */
	public function renderCompanyInfo( /** @noinspection PhpUnusedParameterInspection */ $gatewayId, $params )
	{
		/** @var $payClass cbpaidtwocheckout */
		$payClass				=	$this->getPayMean();
		$vendor_id				=	trim( $payClass->getAccountParam( 'twocheckout_account' ) );
		if ( $vendor_id ) {
			$companyInfo		=	$payClass->_api_getCompanyInfo( $vendor_id );
			if ( is_object( $companyInfo ) ) {
				$html			=	'';
				foreach ($companyInfo->children() as $k => $v ) {
					$html		.=	'<div class="cb_form_line"><label>'
								.	CBTxt::Th( htmlspecialchars( ucwords( str_replace( '_', ' ', $k ) ) ) )
								.	':</label><div class="cb_field">'
								.	htmlspecialchars( (string) $v )
								.	'</div></div>';
				}
			} elseif ( is_string( $companyInfo ) ) {
				$html			=	'<div class="alert alert-danger">' . CBTxt::Th("Error") . ': ' . htmlspecialchars( $companyInfo ) . '</div>';
			} else {
				$html			=	'<div class="alert alert-danger">' . CBTxt::Th("No API access: did you set yet the username and password of your API user ?") . '</div>';
			}
		} else {
			$html				=	'<div class="alert alert-danger">' . CBTxt::Th("No API access: Set first your 2Checkout Account number (vendor id at top of 2Checkout admin)") . '</div>';
		}
		return $html;
	}
}
