<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2023 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Application\Application;
use CBLib\Language\CBTxt;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use CB\Database\Table\UserTable;
use CB\Database\Table\TabTable;
use CB\Plugin\PMS\PMSHelper;
use CB\Plugin\PMS\UddeIM;
use CB\Plugin\PMS\Table\MessageTable;
use CBLib\Core\AutoLoader;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_PLUGINS;

AutoLoader::registerExactMap( '%^CB/Plugin/PMS/(.+)%i', __DIR__ . '/library/$1.php' );

$_PLUGINS->loadPluginGroup( 'user' );

$_PLUGINS->registerFunction( 'onBuildRoute', 'build', '\CB\Plugin\PMS\Trigger\RouterTrigger' );
$_PLUGINS->registerFunction( 'onParseRoute', 'parse', '\CB\Plugin\PMS\Trigger\RouterTrigger' );

$_PLUGINS->registerFunction( 'onAfterDeleteUser', 'deleteMessages', '\CB\Plugin\PMS\Trigger\UserTrigger' );

$_PLUGINS->registerFunction( 'mod_onCBAdminMenu', 'adminMenu', '\CB\Plugin\PMS\Trigger\AdminTrigger' );

$_PLUGINS->registerFunction( 'onAfterMenu', 'getMessages','\CB\Plugin\PMS\Trigger\MenuTrigger' );

class getmypmsproTab extends cbPMSHandler
{

	/**
	 * Sends a PMS message (HTML)
	 *
	 * @param  int     $toUserId        UserId of receiver
	 * @param  int     $fromUserId      UserId of sender
	 * @param  string  $subject         Subject of PMS message in HTML format
	 * @param  string  $message         Body of PMS message in HTML format
	 * @param  boolean $systemGenerated False: real user-to-user message; True: system-Generated by an action from user $fromid (if non-null)
	 * @param  string  $fromName        The name of the public sender
	 * @param  string  $fromEmail       The email address of the public sender
	 * @return boolean                  True: PM sent successfully; False: PM failed to send
	 */
	public function sendUserPMS( $toUserId, $fromUserId, $subject, $message, $systemGenerated = false, $fromName = null, $fromEmail = null )
	{
		global $_PLUGINS;

		if ( UddeIM::isUddeIM() ) {
			return UddeIM::sendUserPMS( $toUserId, $fromUserId, $subject, $message, $systemGenerated, $fromName, $fromEmail );
		}

		$fromUserId				=	(int) $fromUserId;

		if ( ( ! $fromUserId ) && ( ! $systemGenerated ) && ( $fromName || $fromEmail ) ) {
			if ( ! PMSHelper::getGlobalParams()->getInt( 'messages_public', 0 ) ) {
				$_PLUGINS->_setErrorMSG( CBTxt::T( 'PM_FAILED_SEND_ERROR', 'Message failed to send! Error: [error]', array( '[error]' => CBTxt::T( 'From not specified!' ) ) ) );

				return false;
			}
		} elseif ( ! $fromUserId ) {
			$systemGenerated	=	true;
		}

		$toUserId				=	(int) $toUserId;

		if ( ! $toUserId ) {
			$_PLUGINS->_setErrorMSG( CBTxt::T( 'PM_FAILED_SEND_ERROR', 'Message failed to send! Error: [error]', array( '[error]' => CBTxt::T( 'To not specified!' ) ) ) );

			return false;
		}

		if ( $subject ) {
			$message			=	$subject . "\n\n" . $message;
		}

		$row					=	new MessageTable();

		$row->set( 'from_user', $fromUserId );

		if ( ( ! $fromUserId ) && ( ! $systemGenerated ) ) {
			$row->set( 'from_name', $fromName );
			$row->set( 'from_email', $fromEmail );
		}

		$row->set( 'from_system', ( $systemGenerated ? 1 : 0 ) );
		$row->set( 'to_user', $toUserId );
		$row->set( 'message', $message );

		if ( $row->getError() || ( ! $row->check() ) ) {
			$_PLUGINS->_setErrorMSG( CBTxt::T( 'PM_FAILED_SEND_ERROR', 'Message failed to send! Error: [error]', array( '[error]' => $row->getError() ) ) );

			return false;
		}

		if ( $row->getError() || ( ! $row->store() ) ) {
			$_PLUGINS->_setErrorMSG( CBTxt::T( 'PM_FAILED_SEND_ERROR', 'Message failed to send! Error: [error]', array( '[error]' => $row->getError() ) ) );

			return false;
		}

		return true;
	}

	/**
	 * returns all the parameters needed for a hyperlink or a menu entry to do a pms action
	 *
	 * @param  int     $toUserId     UserId of receiver
	 * @param  int     $fromUserId   UserId of sender
	 * @param  string  $subject      Subject of PMS message
	 * @param  string  $message      Body of PMS message
	 * @param  int     $kind         kind of link: 1: link to compose new PMS message for $toid user. 2: link to inbox of $fromid user; 3: outbox, 4: trashbox, 5: link to edit pms options, 6: archive
	 * @return array|boolean         Array of string {"caption" => menu-text ,"url" => NON-cbSef relative url-link, "tooltip" => description} or false and errorMSG
	 */
	public function getPMSlink( $toUserId, $fromUserId, $subject, $message, $kind )
	{
		global $_CB_framework;

		if ( UddeIM::isUddeIM() ) {
			return UddeIM::getPMSlink( $toUserId, $fromUserId, $subject, $message, $kind );
		}

		switch( $kind ) {
			case 1: // Send PM
				if ( PMSHelper::canMessage( $fromUserId, $toUserId ) ) {
					return array(	'caption'	=>	CBTxt::T( 'PM_USER', 'Send Private Message' ),
									'url'		=>	$_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'message', 'func' => 'new', 'to' => (int) $toUserId, 'return' => PMSHelper::getReturn() ) ),
									'tooltip'	=>	CBTxt::T( 'PM_USER_DESC', 'Send a Private Message to this user' )
								);
				}
				break;
			case 2: // Inbox
			case 3: // Outbox
			case 4: // Trashcan
			case 5: // Options
			case 6: // Archive
				return array(	'caption'	=>	CBTxt::T( 'PM_INBOX', 'Show Private Inbox' ),
								'url'		=>	$_CB_framework->pluginClassUrl( $this->element, false, array( 'action' => 'messages' ) ),
								'tooltip'	=>	CBTxt::T( 'PM_INBOX_DESC', 'Show Received Private Messages' )
							);
				break;
		}

		return false;
	}

	/**
	 * Returs array of PMS capabilities or false if no compatible PMS is installed
	 *
	 * @return array|bool false: no compatible PMS installed; array: { 'subject' => boolean, 'body' => boolean, 'public' => boolean }
	 */
	public function getPMScapabilites()
	{
		if ( UddeIM::isUddeIM() ) {
			return UddeIM::getPMScapabilites();
		}

		return array( 'subject' => false, 'body' => true, 'public' => PMSHelper::getGlobalParams()->getInt( 'messages_public', 0 ) );
	}

	/**
	 * Counts number of unread uddeim messages (trashed and archived also excluded) for a user
	 *
	 * @param int $userId
	 * @return int
	 */
	public function getPMSunreadCount( $userId )
	{
		global $_CB_database;

		if ( UddeIM::isUddeIM() ) {
			return UddeIM::getPMSunreadCount( $userId );
		}

		static $cache			=	array();

		if ( ! isset( $cache[$userId] ) ) {
			$query				=	"SELECT COUNT(*)"
								.	"\n FROM " . $_CB_database->NameQuote( '#__comprofiler_plugin_messages' ) . " AS m"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__comprofiler_plugin_messages_read' ) . " AS r"
								.	" ON r." . $_CB_database->NameQuote( 'to_user' ) . " = " . (int) $userId
								.	" AND r." . $_CB_database->NameQuote( 'message' ) . " = m." . $_CB_database->NameQuote( 'id' )
								.	"\n WHERE ( ( m." . $_CB_database->NameQuote( 'from_user' ) . " != " . (int) $userId
								.	" AND m." . $_CB_database->NameQuote( 'to_user' ) . " = 0 )"
								.	" OR ( m." . $_CB_database->NameQuote( 'to_user' ) . " = " . (int) $userId
								.	" AND m." . $_CB_database->NameQuote( 'to_user_delete' ) . " = 0 ) )"
								.	"\n AND r." . $_CB_database->NameQuote( 'id' ) . " IS NULL";
			$_CB_database->setQuery( $query );
			$cache[$userId]		=	(int) $_CB_database->loadResult();
		}

		return $cache[$userId];
	}

	/**
	 * Generates the HTML to display the user profile tab
	 *
	 * @param  TabTable  $tab  the tab database entry
	 * @param  UserTable $user the user being displayed
	 * @param  int       $ui   1 for front-end, 2 for back-end
	 * @return string|boolean  Either string HTML for tab content, or false if ErrorMSG generated
	 */
	public function getDisplayTab( $tab, $user, $ui )
	{
		global $_CB_framework, $_CB_PMS, $_PLUGINS;

		$viewer							=	CBuser::getMyUserDataInstance();
		$features						=	$_CB_PMS->getPMScapabilites();
		$allowPublic					=	( isset( $features[0]['public'] ) && $features[0]['public'] );

		if ( ( ( ! $viewer->getInt( 'id', 0 ) ) && ( ! $allowPublic ) )
			 || ( ! $user->getInt( 'id', 0 ) )
			 || ( $viewer->getInt( 'id', 0 ) == $user->getInt( 'id', 0 ) )
		) {
			return null;
		}

		if ( ( ! UddeIM::isUddeIM() ) && ( ! PMSHelper::canMessage( $viewer->getInt( 'id', 0 ), $user->getInt( 'id', 0 ) ) ) ) {
			return null;
		}

		if ( ! ( $tab->params instanceof ParamsInterface ) ) {
			$tab->params				=	new Registry( $tab->params );
		}

		$messageLimit					=	( Application::MyUser()->isGlobalModerator() ? 0 : PMSHelper::getGlobalParams()->getInt( 'messages_characters', 2500 ) );
		$messageEditor					=	PMSHelper::getGlobalParams()->getInt( 'messages_editor', 2 );

		if ( UddeIM::isUddeIM() || ( ( $messageEditor == 3 ) && ( ! Application::MyUser()->isGlobalModerator() ) ) ) {
			$messageEditor				=	1;
		}

		$input							=	array();
		$input['from_name']				=	null;
		$input['from_email']			=	null;

		if ( ( ! $viewer->getInt( 'id', 0 ) ) && $allowPublic ) {
			$nameTooltip				=	cbTooltip( null, CBTxt::T( 'Input your name to be sent with your message.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

			$input['from_name']			=	'<input type="text" id="from_name" name="from_name" value="' . htmlspecialchars( $this->getInput()->getString( 'post/from_name', '' ) ) . '" class="form-control required" size="40"' . $nameTooltip . cbValidator::getRuleHtmlAttributes( 'maxlength', 100 ) . ' />';

			$emailTooltip				=	cbTooltip( null, CBTxt::T( 'Input your email address to be sent with your message. Note the user you are messaging will see your email address and replies to your message will be emailed to you.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

			$input['from_email']		=	'<input type="text" id="from_email" name="from_email" value="' . htmlspecialchars( $this->getInput()->getString( 'post/from_email', '' ) ) . '" class="form-control required" size="40"' . $emailTooltip . cbValidator::getRuleHtmlAttributes( 'email' ) . cbValidator::getRuleHtmlAttributes( 'maxlength', 100 ) . ' />';
		}

		if ( $messageEditor >= 2 ) {
			$input['message']			=	cbTooltip( null, CBTxt::T( 'Input your reply.' ), null, null, null, Application::Cms()->displayCmsEditor( 'message', $this->getInput()->getHtml( 'post/message', '' ), '100%', 175, 35, 6 ), null, 'class="d-block clearfix"' );
		} else {
			$messageTooltip				=	cbTooltip( null, CBTxt::T( 'Input your reply.' ), null, null, null, null, null, 'data-hascbtooltip="true"' );

			$input['message']			=	'<textarea id="message" name="message" class="w-100 form-control" cols="35" rows="6"' . $messageTooltip . ( $messageLimit ? cbValidator::getRuleHtmlAttributes( 'maxlength', $messageLimit ) : '' ) . '>' . htmlspecialchars( $this->getInput()->getString( 'post/message', '' ) ) . '</textarea>';
		}

		$input['message_limit']			=	null;

		if ( $messageLimit ) {
			static $JS_LOADED			=	0;

			if ( ! $JS_LOADED++ ) {
				$js						=	"$( '.pmMessageEditMessage textarea' ).on( 'change keyup', function() {"
										.		"$( '.pmMessageEditLimit' ).removeClass( 'hidden' );"
										.		"var inputLength = $( this ).val().length;"
										.		"if ( inputLength > $messageLimit ) {"
										.			"$( this ).val( $( this ).val().substr( 0, $messageLimit ) );"
										.			"$( '.pmMessageEditLimitCurrent' ).html( $messageLimit );"
										.		"} else {"
										.			"$( '.pmMessageEditLimitCurrent' ).html( $( this ).val().length );"
										.		"}"
										.	"});";

				if ( $messageEditor >= 2 ) {
					// Before attempting to bind to an editors events make absolutely sure it exists and its used functions eixst; otherwise hide the message limit and just trim on save:
					$js					.=	"if ( ( typeof Joomla != 'undefined' )"
										.		" && ( typeof Joomla.editors != 'undefined' )"
										.		" && ( typeof Joomla.editors.instances['message'] != 'undefined' )"
										.		" && ( typeof Joomla.editors.instances['message'].getValue != 'undefined' )"
										.		" && ( typeof Joomla.editors.instances['message'].setValue != 'undefined' )"
										.		" && ( typeof Joomla.editors.instances['message'].instance != 'undefined' )"
										.		" && ( typeof Joomla.editors.instances['message'].instance.on != 'undefined' ) ) {"
										.		"var messageEditor = Joomla.editors.instances['message'];"
										.		"messageEditor.instance.on( 'change keyup', function() {"
										.			"var inputValue = messageEditor.getValue();"
										.			"var inputLength = inputValue.length;"
										.			"if ( inputLength > $messageLimit ) {"
										.				"messageEditor.setValue( inputValue.substr( 0, $messageLimit ) );"
										.				"$( '.pmMessageEditLimitCurrent' ).html( $messageLimit );"
										.			"} else {"
										.				"$( '.pmMessageEditLimitCurrent' ).html( inputValue.length );"
										.			"}"
										.		"});"
										.	"} else {"
										.		"$( '.pmMessageEditLimit' ).addClass( 'hidden' );"
										.	"}";
				}

				$_CB_framework->outputCbJQuery( $js );
			}

			$input['message_limit']		=	'<div class="badge badge-secondary font-weight-normal align-bottom pmMessageEditLimit">'
										.		'<span class="pmMessageEditLimitCurrent">0</span> / <span class="pmMessageEditLimitMax">' . $messageLimit . '</span>'
										.	'</div>';
		}

		$input['captcha']				=	null;

		$showCaptcha					=	PMSHelper::getGlobalParams()->getInt( 'messages_captcha', 1 );

		if ( Application::MyUser()->isGlobalModerator() || ( ( $showCaptcha == 2 ) && $viewer->getInt( 'id', 0 ) ) ) {
			$showCaptcha				=	0;
		}

		if ( $showCaptcha ) {
			$input['captcha']			=	implode( '', $_PLUGINS->trigger( 'onGetCaptchaHtmlElements', array( true ) ) );
		}

		cbValidator::loadValidation();
		initToolTip();

		ob_start();
		require PMSHelper::getTemplate( null, 'tab' );
		$html							=	ob_get_contents();
		ob_end_clean();

		$class							=	PMSHelper::getGlobalParams()->getString( 'general_class', '' );

		$return							=	'<div class="cbPMS' . ( $class ? ' ' . htmlspecialchars( $class ) : null ) . '">'
										.		$html
										.	'</div>';

		return $return;
	}
}
