<?php
/**
 * @version $Id: cbpaidTimes.php 1541 2012-11-23 22:21:52Z beat $
 * @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
 * @subpackage Plugin for Paid Subscriptions
 * @copyright (C) 2007-2020 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

use CBLib\Registry\ParamsInterface;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
 * Class handling time and timezones
 */
class cbpaidTimes {
	/** Main params
	 *  @var ParamsInterface */
	protected $params;
	/**
	 * Constructor
	 * @private
	 */
	protected function __construct( ) {
		$this->params			=	cbpaidApp::settingsParams();
	}
	/**
	 * Gets a single instance of this class
	 *
	 * @return cbpaidTimes
	 */
	public static function getInstance( ) {
		static $singleInstance	=	null;
		if ( $singleInstance === null ) {
			$singleInstance		=	new cbpaidTimes();
		}
		return $singleInstance;
	}
	/**
	 * Transforms a SQL-formatted UTC datetime to a unix time
	 *
	 * @param  string   $sqlDateStr  'YYYY-MM-DD HH:II:SS' or NULL
	 * @return int                   unix-time
	 */
	public function strToTime( $sqlDateStr ) {
		if ( $sqlDateStr === null ) {
			$time = null;
		} else {
			list($y, $c, $d, $h, $m, $s) = sscanf($sqlDateStr, '%d-%d-%d %d:%d:%d');
			$time = gmmktime($h, $m, $s, $c, $d, $y);			// we do NOT use PHP strtotime, which is broken
		}
		return $time;
	}

	/**
	 * Returns the database-formatted (not quoted) date/time in UTC timezone format
	 *
	 * @param  int     $time      NULL: Now of script start time ($_SERVER['REQUEST_TIME'] or time())
	 * @return string             Unquoted SQL 'Y-m-d H:i:s' date string
	 */
	public function getUtcDateOfTime( $time = null )
	{
		if ( ! $time ) {
			$time	=	$this->startTime();
		}

		$date		=	new DateTime( '@' . (string) $time, new DateTimeZone( 'UTC' ) );

		return $date->format( 'Y-m-d H:i:s' );
	}

	/**
	 * Gets system's timezone
	 *
	 * @return string  Timezone of system (e.g. Europe/Zurich)
	 */
	public function systemTimeZone( )
	{
		global $_CB_framework;

		$timeZone	=	$_CB_framework->getCfg( 'system_timezone' );

		if ( $timeZone != null ) {
			return $timeZone;
		}

		// Temporary fallback for CB 2.0 < 2.0.6:
		return JFactory::getConfig()->get( 'offset' );
	}

	/**
	 * Returns the Unix start time in seconds since 1.1.1970 of pageload
	 *
	 * @return int     Unix-time in seconds
	 */
	public function startTime( )
	{
		global $_CB_framework;

		return $_CB_framework->now();
	}

	/**
	 * php function strtotime() but in GMT/UTC timezone
	 *
	 * @param  string    $time
	 * @param  int|null  $now
	 * @return int
	 */
	public function gmStrToTime( $time, $now = null )
	{
		$timeZone	=	new DateTimeZone( 'UTC' );
		$dateTime	=	new DateTime( '@' . (string) ( $now ? $now : $this->startTime() ), $timeZone );
		$dateTime->setTimezone( $timeZone );

		$dateTime->modify( $time );

		return $dateTime->getTimestamp();
	}

	/**
	 * php function strtotime but in system timezone
	 *
	 * @param      $time
	 * @param null $now
	 * @return int
	 */
	public function localStrToTime( $time, $now = null )
	{
		$timeZone	=	new DateTimeZone( $this->systemTimeZone() );
		$dateTime	=	new DateTime( '@' . (string) ( $now ? $now : $this->startTime() ), $timeZone );
		$dateTime->setTimezone( $timeZone );

		$dateTime->modify( $time );

		return $dateTime->getTimestamp();
	}

	/**
	 * gmdate() but with system-local timezone
	 *
	 * @param  string    $format
	 * @param  int|null  $timestamp
	 * @return string
	 */
	public function localDate( $format, $timestamp = null )
	{
		$timeZone	=	new DateTimeZone( $this->systemTimeZone() );
		$dateTime	=	new DateTime( '@' . (string) ( $timestamp ? $timestamp : $this->startTime() ), $timeZone );
		$dateTime->setTimezone( $timeZone );

		return $dateTime->format( $format );
	}

	/**
	 * Creates and returns a PHP DateInterval object from a CBSubs interval YYYY-MM-DD HH:ii:ss
	 * @since 4.0.0
	 *
	 * @param  string        $YmdHis
	 * @return DateInterval
	 */
	public function dateInterval( $YmdHis )
	{
		list( $y, $c, $d, $h, $m, $s)		=	sscanf( $YmdHis, '%d-%d-%d %d:%d:%d');

		return new DateInterval( 'P' . (int) $y . 'Y' . (int) $c . 'M' . (int) $d . 'D' . 'T' . (int) $h . 'H' . (int) $m . 'M' . (int) $s . 'S' );
	}

	/**
	 * Returns formatted time period ( xxx weeks , or xxx years xxx months xxx days xxx hours xxx minutes xxx seconds
	 *
	 * @param  array|string  $ycdhmsArray  array  of int = list( $years, $months, $days, $hours, $minutes, $seconds ) or SQL DATETIME string
	 * @param  int           $occurrences  [default: 1] multiply period by the occurrences before displaying
	 * @param  boolean       $displayOne   [default: true] displays also if only 1 unit of something
	 * @param  string        $prefix       text between number and period, e.g. 3 calendar months
	 * @return string
	 */
	public function renderPeriod( $ycdhmsArray, $occurrences = 1, $displayOne = true, $prefix = '' ) {
		$text = '';
		if ( $prefix ) {
			$prefix		=	$prefix . ' ';
		}
		if ( ! is_array( $ycdhmsArray ) ) {
			$ycdhmsArray	=	sscanf( $ycdhmsArray, '%d-%d-%d %d:%d:%d');
		}
		list($y, $c, $d, $h, $m, $s) = $ycdhmsArray;

		if ( $occurrences != 1 ) {
			$s *= $occurrences;
			$m *= $occurrences;
			$h *= $occurrences;
			$d *= $occurrences;
			$c *= $occurrences;
			$y *= $occurrences;
			if ( $c && ( ( $c % 12 ) == 0 ) ) {
				$y += $c / 12;
				$c = 0;
			}
		}

		if ( ( $y == 0 ) && ( $c == 0 ) && ( ( $d != 0 ) && ( ( $d % 7 ) == 0 ) ) && ( $h == 0 ) && ( $m == 0 ) && ( $s == 0 )  ) {
			$w = $d / 7;
			if ( $w == 1 ) {
				if ( $displayOne ) {
					$text = $w . ' ';
				}
				$text .= $prefix . CBPTXT::T("week") . ' ';
			} else {
				$text = $w . ' ' . $prefix . CBPTXT::T("weeks") . ' ';
			}
		} else {
			$text = $this->renderIntervalYMDHIS( $y, $c, $d, $h, $m, $s, $prefix );
			if ($text == '') {
				$text = CBPTXT::T( $this->params->get( 'regtextLifetime', "Lifetime subscription" ) );
			} elseif ( ! $displayOne ) {
				if ( ( ( $y + $c + $d + $h + $m + $s ) == 1 ) && ( substr( $text, 0, 2 ) == '1 ' ) ) {
					$text = substr( $text, 2 );
				}
			}
		}
		return trim( $text );
	}

	/**
	 * Checks if pluralization is done in language strings for dates
	 * TODO REMOVE IN CBSUBS 5.0:
	 *
	 * @return bool  Yes, language file is up to date
	 */
	private function isPluralizationAvailable() {
		return ( CBPTXT::T( 'X_YEARS' ) !== 'X_YEARS' );
	}

	/**
	 * Outputs a human-readable form of a time-interval.
	 * "Y years, M months, D days, H hours, I minutes and S seconds"
	 *
	 * @param  int    $y      Years
	 * @param  int    $m      Months
	 * @param  int    $d      Days
	 * @param  int    $h      Hours
	 * @param  int    $i      Minutes
	 * @param  int    $s      Seconds
	 * @param  string $prefix '' or 'calendar ' : Text between number and period, e.g. 3 calendar months
	 * @return string
	 */
	public function renderIntervalYMDHIS( $y, $m = 0, $d = 0, $h = 0, $i = 0, $s = 0, $prefix = '' )
	{
		$parts			=	array();

		if ( $this->isPluralizationAvailable() ) {
			if ( $prefix === '' ) {
				if ( $y ) {
					$parts[]	=	CBPTXT::T( 'X_YEARS', "one year|[X] years", array( '[X]' => $y ) );
				}
				if ( $m ) {
					$parts[]	=	CBPTXT::T( 'X_MONTHS', "one month|[X] months", array( '[X]' => $m ) );
				}
				if ( $d ) {
					$parts[]	=	CBPTXT::T( 'X_DAYS', "one day|[X] days", array( '[X]' => $d ) );
				}
			} else {
				if ( $y ) {
					$parts[]	=	CBPTXT::T( 'X_CALENDAR_YEARS', "one calendar year|[X] calendar years", array( '[X]' => $y ) );
				}
				if ( $m ) {
					$parts[]	=	CBPTXT::T( 'X_CALENDAR_MONTHS', "one calendar month|[X] calendar months", array( '[X]' => $m ) );
				}
				if ( $d ) {
					$parts[]	=	CBPTXT::T( 'X_CALENDAR_DAYS', "one calendar day|[X] calendar days", array( '[X]' => $d ) );
				}
			}
			if ( $h ) {
				$parts[]		=	CBPTXT::T( 'X_HOURS', "one hour|[X] hours", array( '[X]' => $h ) );
			}
			if ( $i ) {
				$parts[]		=	CBPTXT::T( 'X_MINUTES', "one minute|[X] minutes", array( '[X]' => $i ) );
			}
			if ( $s ) {
				$parts[]		=	CBPTXT::T( 'X_SECONDS', "one second|[X] seconds", array( '[X]' => $s ) );
			}

		} else {
			// TODO REMOVE IN CBSUBS 5.0:
			if ( $y ) {
				$parts[]		=	$y . ' ' . $prefix . ( $y == 1 ? CBPTXT::T( "year" ) : CBPTXT::T( "years" ) );
			}
			if ( $m ) {
				$parts[]		=	$m . ' ' . $prefix . ( $m == 1 ? CBPTXT::T("month")  : CBPTXT::T("months") );
			}
			if ( $d ) {
				$parts[]		=	$d . ' ' . $prefix . ( $d == 1 ? CBPTXT::T("day")	: CBPTXT::T("days")	);
			}
			if ( $h ) {
				$parts[]		=	$h . ' ' . $prefix . ( $h == 1 ? CBPTXT::T("hour")	: CBPTXT::T("hours") );
			}
			if ( $i ) {
				$parts[]		=	$i . ' ' . $prefix . ( $i == 1 ? CBPTXT::T("minute") : CBPTXT::T("minutes") );
			}
			if ( $s ) {
				$parts[]		=	$s . ' ' . $prefix . ( $s == 1 ? CBPTXT::T( "second" ) : CBPTXT::T( "seconds" ) );
			}
		}

		if ( count( $parts ) > 2 ) {
			$endText	=	array_pop( $parts );
			$beginText	=	implode( CBPTXT::T( 'CB_FIRST_COMMA_OF_YEARS_COMMA_MONTHS_AND_DAYS', ", " ), $parts );
			return $beginText . CBPTXT::T( 'CB_LAST_AND_OF_YEARS_COMMA_MONTHS_AND_DAYS', " and " ) . $endText;
		}
		return implode( CBPTXT::T( 'CB_ONLY_AND_OF_YEARS_AND_MONTHS', " and " ), $parts );
	}

	/**
	 * Outputs a date interval in human-readable translated form.
	 * "X years, y months and z days" or "x days and y hours" or "x hours and y minutes" or "x days" (if 3 to 90 days)
	 *
	 * @param  int  $fromTime  Start Epoch time
	 * @param  int  $toTime    End Epoch time
	 * @return string          Human-readable form
	 * @throws Exception
	 */
	public function timeIntervalToNaturalText( $fromTime, $toTime )
	{
		if ( $fromTime >= $toTime ) {
			return CBPTXT::T("already expired");
		}

		$from		=	new DateTime( '@' . (string) $fromTime, new DateTimeZone( 'UTC' ) );
		$to			=	new DateTime( '@' . (string) $toTime, new DateTimeZone( 'UTC' ) );
		$interval	=	$to->diff( $from );

		$days		=	$interval->format( 'a' );
		if ( ( $days > 2 ) && ( $days <= 90 ) ) {
			return $this->renderIntervalYMDHIS( 0, 0, $days );
		}

		if ( $interval->y || $interval->m ) {
			return $this->renderIntervalYMDHIS( $interval->y, $interval->m, $interval->d );
		}

		if ( $interval->d ) {
			return $this->renderIntervalYMDHIS( 0, 0, $interval->d, $interval->h );
		}

		return $this->renderIntervalYMDHIS( 0, 0, 0, $interval->h, $interval->i, ( $interval->h ? 0 : $interval->s ) );
	}

	/**
	 * Offsets date-time if time is present and $serverTimeOffset 1, then formats to CB's configured date-time format.
	 *
	 * @param  int|string   $date                In "Y-m-d H:i:s" format, or 	int : unix timestamp
	 * @param  boolean|null $showTime            false: hide time, true: force show time
	 * @return null|string
	 */
	public function cbFormatDateInOfficialTz( $date, $showTime = null )
	{
		$params				=	cbpaidApp::settingsParams();
		$showTime			=	( $showTime === true ) || ( ( $showTime === null ) && ( $params->get( 'showtime', '1' ) == '1' ) );

		// User time-zone would be:		$timeZone			=	Application::Cms()->getCmsUser( $userId )->getParam( 'timezone', $this->systemTimeZone() );
		$timeZone			=	$this->systemTimeZone();

		return cbFormatDate( $date, true, $showTime, null, null, $timeZone );
	}
}
