<?php
/**
 * @version $Id: cbpaidTimed.php 1541 2012-11-23 22:21:52Z beat $
 * @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
 * @subpackage Plugin for Paid Subscriptions
 * @copyright (C) 2007-2020 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

use CBLib\Database\DatabaseDriverInterface;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
 * Base class for timed items
 *
 */
abstract class cbpaidTimed extends cbpaidTable {
	public $validity;
	public $bonustime;
	public $bonustimeofreactivation;
	/**
	 * Constructor
	 *
	 *	@param string                   $table  name of the table in the db schema relating to child class
	 *	@param string                   $key    name of the primary key field in the table
	 *	@param DatabaseDriverInterface  $db     CB Database object
	 */
	public function __construct(  $table, $key, &$db = null ) {
		parent::__construct(  $table, $key, $db );
	}

	/**
	 * Gets the value of the class variable
	 *
	 * @param  string        $var      The name of the class variable
	 * @param  mixed         $default  The value to return if no value is found
	 * @param  string|array  $type     [optional] Default: null: GetterInterface::COMMAND. Or const int GetterInterface::COMMAND|GetterInterface::INT|... or array( const ) or array( $key => const )
	 * @return mixed                   The value of the class var (or null if no var of that name exists)
	 */
	public function get( $var, $default = null, $type = null )
	{
//		if ( ! property_exists( $this, $column ) ) {
//			trigger_error( 'cbpaidTimed::get ("' . htmlspecialchars( $column ) . '") innexistant attribute.', E_USER_ERROR );
//		}
		return parent::get( $var, $default, $type );
	}
	/**
	 * Set the value of the attribute of this timed item
	 *
	 * @param  string  $column  Name of attribute
	 * @param  mixed   $value   Nalue to assign to attribute
	 */
	public function set( $column, $value ) {
		if ( ! property_exists( $this, $column ) ) {
			trigger_error( 'cbpaidTimed::set ("' . htmlspecialchars( $column ) . '") innexistant attribute.', E_USER_ERROR );
		}

		parent::set( $column, $value );
	}
	/**
	 * Fix variable name 'first_validity' to 'validity' if there is no different first period
	 *
	 * @param  string  $varName   'first_validity' or 'validity'   !!! CHANGES (FIXES) THAT VAR NAME
	 */
	abstract public function fixVarName( &$varName );

	/**
	 * TIMING METHODS:
	 */

	/**
	 * Transforms a SQL-formatted datetime to a unix time
	 *
	 * @param  string   $sqlDateStr  'YYYY-MM-DD HH:II:SS' or NULL
	 * @return int                   unix-time
	 */
	public function strToTime( $sqlDateStr )
	{
		if ( ( $sqlDateStr === null ) || ( $sqlDateStr === '0000-00-00 00:00:00' ) ) {
			return null;
		}

		return cbpaidTimes::getInstance()->strToTime( $sqlDateStr );
	}


	/**
	 * Computes the real start and expiry time of the timed item based on start-time, first or succeeding periods, and,
	 * if succeeding periods the number of occurrences.
	 * WARNING: changes input $startTime by reference to reflect real starting time.
	 *
	 * @param  int     $startTime    RETURNS ALSO: IN: Unix-time of start OUT: Unix-time of REAL START (BONUS TIMES APPLY)
	 * @param  string  $varName      'first_validity' or 'validity'
	 * @param  int     $occurrences  number of occurrences
	 * @param  string  $reason       payment reason: 'N'=new subscription (default), 'R'=renewal, 'U'=update
	 * @param  string  $status       status of subscription
	 * @return int                   Unix-time
	 */
	public function getExpiryTime( &$startTime, $varName, $occurrences, $reason, $status )
	{
		$this->fixVarName( $varName );

		if ( $this->isLifetimeValidity() ) {
			// lifetime:
			return null;
		}

		// Compute $duration DateInterval and multiply its duration by the occurences:
		$duration		=	$this->validityDateInterval( $varName );

		if ( $occurrences !== 1 ) {
			foreach ( $duration as &$value ) {
				$value = $value * $occurrences;
			}
		}

		if ( $duration->y + $duration->m + $duration->d + $duration->h + $duration->i + $duration->s == 0 ) {
			return null; 	// Lifetime subscription
		}

		$systemTimeZone			=	new DateTimeZone( cbpaidTimes::getInstance()->systemTimeZone() );
		// Add 1 second to compensate for expiry at 23:59:59 instead of 00:00:00:
		$startTime				=	$startTime + 1;
		$startDateTime			=	new DateTime( '@' . $startTime, $systemTimeZone );
		$startDateTime->setTimezone( $systemTimeZone );

		if ( $this->isCalendarValidity( $varName ) ) {
			// Fix previous non-UTC saved dates:
			if ( ( $reason == 'R' ) && ( $status == 'A' ) && ( $startDateTime->format( 'H:i:s' ) != '00:00:00' ) && ( intval( $startDateTime->format( 'H' ) ) >= 12 ) ) {
				$startDateTime->modify( '+1 day' );
			}
			// Calendar-based expiries need to modify start date and time to correspond to calendar boundary:
			$startDateTime->modify( '00:00:00' );	// Calendar days start at midnight (system timezone)

			if ( ( $reason != 'R' ) || ( $status == 'X' ) ) {
				// advance startTime by bonusTime, it should change expiry only when it really should:
				$startDateTime->add( cbpaidTimes::getInstance()->dateInterval( $this->get( ( $reason == 'R' && $status == 'X') ? 'bonustimeofreactivation' : 'bonustime' ) ) );
			}

			$currentDateTimeWithBonus	=	clone $startDateTime;

			if ( $duration->y ) {
				// calendar years: Set starting Month/Day,

				list( $cs, $ds )	=	explode( '-', $this->calendarYearStart( $varName ) );
				$startDateTime->modify( (int) $cs . '/' . (int) $ds );

				if ( $currentDateTimeWithBonus->getTimestamp() < $startDateTime->getTimestamp() ) {
					$startDateTime->modify( '-1 year' );
				}

				// Add years of duration, then remove 1 second to make it 23:59:59 of the correct year:
			} elseif ( $duration->m ) {
				// calendar months: Set starting Day:
				list( , $ds )		=	explode( '-', $this->calendarYearStart( $varName ) );
				$startDateTime->modify( $startDateTime->format( 'm' ) . '/' . (int) $ds );

				if ( $currentDateTimeWithBonus->getTimestamp() < $startDateTime->getTimestamp() ) {
					$startDateTime->modify( '-1 month' );
				}

			} elseif ( $duration->d ) {
				// calendar days: Nothing to do as we are already doing what is needed!
			} else {
				trigger_error( 'getExpiryTime detected Calendar duration of less than a day', E_USER_NOTICE );
				// Well, let's guess 1 calendar year in this case, so we don't have undefineds:
				$startDateTime->modify( '01/01' );
			}

			// Change the start time according to computed calendar-start time:
			$startTime				=	$startDateTime->getTimestamp();
		}

		// Subscription-time relative based expiries continue here same as calendar-based ones:

		$endDateTime				=	clone $startDateTime;
		$endDateTime->add( $duration );

		// Remove the 1 second added at the begin to compensate for expiry at 23:59:59 instead of 00:00:00:
		$expiryTime					=	$endDateTime->getTimestamp() -1;

		return $expiryTime;
	}

	/**
	 * Enter description here...
	 *
	 * @param  string  $validity  time to subsctract ( 'Y-m-d H:i:s' format)
	 * @param  int     $time      UNIX-time
	 * @return int                UNIX-time
	 */
	public function substractValidityFromTime( $validity, $time )
	{
		if ( ( ! $validity ) || ( $validity == '0000-00-00 00:00:00' ) ) {
			return $time;
		} else {

			$dateInterval	=	cbpaidTimes::getInstance()->dateInterval( $validity );
			$systemTimeZone	=	new DateTimeZone( cbpaidTimes::getInstance()->systemTimeZone() );
			$date			=	new DateTime( '@' . $time, $systemTimeZone );
			$date->setTimezone( $systemTimeZone );

			$date->sub( $dateInterval );

			return $date->getTimestamp();
		}
	}

	/**
	 * says if validity is unlimitted
	 * OVERRIDE ! this tests only on validity !
	 *
	 * @return boolean	             true if lifetime, false if limitted time
	 */
	public function isLifetimeValidity() {
		return  ( $this->get( 'validity' ) == '0000-00-00 00:00:00' );
	}
	/**
	 * says if validity is calendar-based
	 *
	 * @param  string  $varName     variable name ( 'validity' or 'first_validity' )
	 * @return boolean              true if calendar-based, false if time-based
	 */
	public function isCalendarValidity( $varName ) {
		$this->fixVarName( $varName );
		return  ( strncasecmp( 'U:', $this->get( $varName ), 2 ) == 0 );
	}
	/**
	 * Gives Calendar Year start
	 *
	 * @param  string  $varName      'first_validity' or 'validity'
	 * @return string                'month-day', e.g. '01-01'
	 */
	public function calendarYearStart( /** @noinspection PhpUnusedParameterInspection */ $varName ) {
		return '01-01';
	}
	/**
	 * returns validity period as years, months, days, hours, minutes, seconds.
	 *
	 * @param  string  $varName     variable name ( 'validity' or 'first_validity' )
	 * @return array of int	        list( $years, $months, $days, $hours, $minutes, $seconds )
	 */
	public function getValidity( $varName ) {		//TODO: replace usage by getValidityDateInterval
		$this->fixVarName( $varName );
		if ( $this->isCalendarValidity( $varName ) ) {
			return sscanf( substr( $this->get( $varName ), 2 ), '%d-%d-%d %d:%d:%d');
		} else {
			// Subscription-time based expiries:
			return sscanf($this->get( $varName ), '%d-%d-%d %d:%d:%d');
		}
	}

	/**
	 * returns validity period as years, months, days, hours, minutes, seconds.
	 * @since 4.0.0
	 *
	 * @param  string        $varName  Variable name ( 'validity' or 'first_validity' )
	 * @return DateInterval	           List( $years, $months, $days, $hours, $minutes, $seconds )
	 */
	public function validityDateInterval( $varName )
	{
		$this->fixVarName( $varName );

		$interval	=	$this->get( $varName );

		if ( $this->isCalendarValidity( $varName ) ) {
			// Remove 'U:':
			$interval	=	substr( $interval, 2 );
		}

		return cbpaidTimes::getInstance()->dateInterval( $interval );
	}

	/**
	 * Returns the period of validitiy from startTime on in seconds
	 *
	 * @param  int     $startTime    starting time in unix time.
	 * @param  string  $varName      variable name ( 'validity' or 'first_validity' )
	 * @return int                   seconds of subscription or null for lifetime (no expiration)
	 */
	public function getFullPeriodValidityTime( $startTime = null, $varName )
	{
		$this->fixVarName( $varName );

		if ( $this->isLifetimeValidity() ) {
			// lifetime:
			return null;
		}

		// Compute $duration DateInterval:
		$duration			=	$this->validityDateInterval( $varName );

		if ( $duration->y + $duration->m + $duration->d + $duration->h + $duration->i + $duration->s == 0 ) {
			return null; 	// Lifetime subscription
		}

		$systemTimeZone		=	new DateTimeZone( cbpaidTimes::getInstance()->systemTimeZone() );
		$startDateTime		=	new DateTime( '@' . $startTime, $systemTimeZone );
		$startDateTime->setTimezone( $systemTimeZone );

		// Subscription-time relative based expiries continue here same as calendar-based ones:

		$endDateTime		=	clone $startDateTime;
		$endDateTime->add( $duration );

		$expiryTime			=	$endDateTime->getTimestamp();

		return $expiryTime - $startTime;
	}

	/**
	 * Checks if a given expiry date has passed a given time
	 *
	 * @param  string    $expiryDate   SQL-formatted expiry date or NULL for non-expiring item
	 * @param  int       $time         UNIX-formatted time (default: now)
	 * @return boolean                 TRUE if valid (not expired), FALSE otherwise
	 */
	public function checkValid( $expiryDate, $time = null ) {
		if ( $this->isLifetimeValidity() || ( $expiryDate == null ) ) {
			// lifetime:
			return true;
		} else {
			$expiryTime = $this->strToTime( $expiryDate );
			if ( $time === null ) {
				$time	=	cbpaidTimes::getInstance()->startTime();
			}
			return ( $time < $expiryTime);
		}
	}

	/**
	 * RENDERING METHODS:
	 */

	/**
	 * Returns formatted time period ( xxx weeks , or xxx years xxx months xxx days xxx hours xxx minutes xxx seconds
	 *
	 * @param  int[]    $ycdhmsArray  = list( $years, $months, $days, $hours, $minutes, $seconds )
	 * @param  int      $occurrences  [default: 1] multiply period by the occurrences before displaying
	 * @param  boolean  $displayOne   [default: true] displays also if only 1 unit of something
	 * @param  string   $prefix       text between number and period, e.g. 3 calendar months
	 * @return string
	 */
	public function renderPeriod( $ycdhmsArray, $occurrences = 1, $displayOne = true, $prefix = '' ) {
		$cbpaidTimes	=	cbpaidTimes::getInstance();
		return $cbpaidTimes->renderPeriod( $ycdhmsArray, $occurrences, $displayOne, $prefix );
	}
	/**
	 * Renders a calendar or time-period validity period, e.g. Year 2007, March - May 2007, December 2006 - January 2007, etc.
	 *
	 * @param  int      $startTime    Unix-time
	 * @param  string   $varName      variable name ( 'validity' (default) or 'first_validity' )
	 * @param  string   $reason       payment reason: 'N'=new subscription (default), 'R'=renewal, 'U'=update (needed only if $expiryTime is NULL)
	 * @param  int      $occurrences  number of occurrences (needed only if $expiryTime is NULL)
	 * @param  string   $status       status of subscription
	 * @return string
	 */
	public function getFormattedExpiryDate( $startTime, $varName, $reason = null, $occurrences = 1, $status = 'I' )
	{
		$params		=	cbpaidApp::settingsParams();

		$this->fixVarName( $varName );

		if ( $startTime === null ) {
			$startTime	=	cbpaidTimes::getInstance()->startTime();
		}
		$expiryTime		=	$this->getExpiryTime( $startTime, $varName, $occurrences, $reason, $status );		// WARNING: adjusts $startTime to the real Start-time, which is wanted here
		if ( $expiryTime ) {
			$text		=	cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $expiryTime );
		} else {
			$text		=	CBPTXT::T( $params->get( 'regtextLifetime', 'Lifetime Subscription' ) );
		}
		return $text;
	}

	/**
	 * Renders a calendar or time-period validity period, e.g. Year 2007, March - May 2007, December 2006 - January 2007, etc.
	 *
	 * @param  int|null  $expiryTime   Unix-time
	 * @param  int|null  $startTime    Unix-time
	 * @param  string    $varName      variable name ( 'validity' (default) or 'first_validity' )
	 * @param  string    $reason       payment reason: 'N'=new subscription (default), 'R'=renewal, 'U'=update (needed only if $expiryTime is NULL)
	 * @param  int       $occurrences  number of occurrences (needed only if $expiryTime is NULL)
	 * @param  boolean   $displayOne   Display significant 1s also if it's 1: e.g. TRUE: 1 year, FALSE: year
	 * @param  boolean   $html         true: Display for html with non-breaking spaces
	 * @param  string    $status       status of subscription
	 * @return string
	 */
	public function getFormattedValidity( $expiryTime, $startTime, $varName, $reason = null, $occurrences = 1, $displayOne = true, $html = false, $status = 'I' )
	{
		$this->fixVarName( $varName );

		$text = '';
		if ( $this->isCalendarValidity( $varName ) ) {
			$now		=	cbpaidTimes::getInstance()->startTime();

			if ( $startTime === null ) {
				$startTime = $now;
			}
			if ( $expiryTime === null ) {
				$expiryTime = $this->getExpiryTime( $startTime, $varName, $occurrences, $reason, $status );		// WARNING: adjusts $startTime to the real Start-time, which is wanted here
			}

			$isValid	= ( /*removed otherwise renew buttons display wrong: ( $startTime <= $now  ) && */ ( $now < $expiryTime ) );

			list( $y, $c, $d, /* $h */, /* $m */, /* $s */ ) = $this->getValidity( $varName );	// = sscanf( substr( $this->get( $varName ), 2 ), '%d-%d-%d %d:%d:%d' );
			list( $yn, $cn, $dn )	=	sscanf( cbpaidTimes::getInstance()->localDate( 'Y-m-d', $now ),		  '%d-%d-%d' );
			list( $ys, $cs, $ds )	=	sscanf( cbpaidTimes::getInstance()->localDate( 'Y-m-d', $startTime ),  '%d-%d-%d' );
			list( $ye, $ce, $de )	=	sscanf( cbpaidTimes::getInstance()->localDate( 'Y-m-d', $expiryTime ), '%d-%d-%d' );

			$calStart	=	$this->calendarYearStart( $varName );

			if ( $y && ( $calStart == '01-01' ) ) {

				if ( ( $y == 1 ) && ( $ye <= ( $ys + 1 ) ) ) {
					$text .= sprintf( $this->_htmlNbsp( CBPTXT::T("Year %s"), $html ), $ye );						// 'Year 2007'

					if ( $ye != $yn && $isValid ) {
						$text .= ' (' . CBPTXT::T("valid from now on") . ')';
					}
				} else {
					$years = $ye - $ys + 1;

					if ( ( ! $isValid ) || ( ( $y == $years ) && ( $ys == $yn ) ) ) {
						$text .= sprintf( $this->_htmlNbsp( CBPTXT::T("Years %s - %s"), $html ), $ys, $ye );		// 'Years 2006 - 2007'
					} else {

						if ( $y == $years ) {
							$text .= sprintf( $this->_htmlNbsp( CBPTXT::T("Years %s - %s"), $html ), $ys, $ye );	// 'Years 2007 - 2008'
						} else {
							$text .= sprintf( $this->_htmlNbsp( CBPTXT::T("Years %s - %s"), $html ), $ys + 1, $ye ); // 'Years 2007 - 2008'
						}
						$text .= ' (' . CBPTXT::T("valid from now on") . ')';
					}
				}
			} elseif ( $c || $y ) {

				if ( ( $calStart != '01-01' ) && ! preg_match( '/$..-01/', $calStart ) ) {
					// $text .= $calStart . date( 'Y-m-d H:i:s', $startTime ) .( $c + ( $y * 12 ) ) . '_' . ($ce - $cs + 1 + ( ( $ye - $ys ) * 12 )) . '_';
					$text .= CBPTXT::Tdate( 'j F', $startTime) . ( ( $ys != $ye ) ? $this->_htmlNbsp( ' ', $html ) . $ys : '' );	// 'January' or 'December 2006'
					$text .= $this->_htmlNbsp( ' - ', $html );														// ' - '
					$text .= CBPTXT::Tdate( 'j F', $expiryTime) . $this->_htmlNbsp( ' ', $html ) . $ye;						// 'February 2007'

					if ( ( ( $ys > $yn ) || ( ( $cs > $cn ) && ( $ys == $yn ) ) ) && $isValid ) {
						$text .= ' (' . CBPTXT::T("valid from now on") . ')';
					}
				} else {
					$months = $ce - $cs + 1 + ( ( $ye - $ys ) * 12 );

					if ( ( ( $c == 1 ) && ( $y == 0 ) ) || ( $months == 1 ) ) {
						$text .= CBPTXT::Tdate( 'F', $expiryTime) . $this->_htmlNbsp( ' ', $html ) . $ye;		// 'January 2007'
						if ( $ce != $cn  && $isValid ) {
							$text .= ' (' . CBPTXT::T("valid from now on") . ')';
						}
					} else {
						// if ( ( $c + ( $y * 12 ) ) == $months ) {
						$text .= CBPTXT::Tdate( 'F', $startTime) . ( ( $ys != $ye ) ? $this->_htmlNbsp( ' ', $html ) . $ys : '' );	// 'January' or 'December 2006'
						$text .= $this->_htmlNbsp( ' - ', $html );														// ' - '
						$text .= CBPTXT::Tdate( 'F', $expiryTime) . $this->_htmlNbsp( ' ', $html ) . $ye;						// 'February 2007'
						if ( ( ! ( ( $cs == $cn ) && ( $ys == $yn ) ) ) && $isValid ) {
							$text .= ' (' . CBPTXT::T("valid from now on") . ')';
						}
						// } else {		//TBD: check if this else is still needed
						/*	list($ynn, $cnn, $dnn, $hnn, $mnn, $snn) = sscanf( gmdate( 'Y-m-d H:i:s', $startTime ), '%d-%d-%d %d:%d:%d' );
							$cnn += 2;
							$dnn = 0;
							$nextMonthTime = mktime($hnn, $mnn, $snn, $cnn, $dnn, $ynn);
							$text .= gmdate( 'F', $nextMonthTime) . ( ( $ynn != $ye ) ? $this->_htmlNbsp( ' ', $html ) . $ynn : '' );	// 'January' or 'December 2006'
							$text .= $this->_htmlNbsp( ' - ', $html );														// ' - '
							$text .= gmdate( 'F', $expiryTime) . $this->_htmlNbsp( ' ', $html ) . $ye;						// 'February 2007'
							if ( $isValid ) {
								$text .= ' (' . CBPTXT::T("valid from now on") . ')';
							}
						*/
						// }
					}
				}
			} elseif ( $d ) {
				if ( $de == $dn ) {
					$text .= CBPTXT::T("Today");
				} elseif ( ( $de == ( $dn + 1 ) ) || ( ( $de == 1 ) && ( ( $expiryTime - $now ) < 48*3600 ) ) ) {
					if ( $d == 1 ) {
						$text .= CBPTXT::T("Tomorrow");
						if ( $isValid ) {
							$text .= ' (' . CBPTXT::T("valid from now on") . ')';
						}
					} else {
						$text .= CBPTXT::T("Today and tomorrow");
					}
				} else {
					if ( $isValid ) {
						$days = (int) floor( ( $expiryTime - $now ) / ( 24 * 3600 ) );
						if ( ( $days < $d ) && ( $ds == $dn ) ) {
							$t		=	CBPTXT::T("Today and next %d days");
							if ( $html ) {
								$t	=	str_replace( ' %d ', '&nbsp;%d&nbsp;', $t );
							}
							$text .= sprintf( $t, $days );
						} else {
							$text .= sprintf( $this->_htmlNbsp( CBPTXT::T("Next %d days"), $html ), $days ) . ' (' . CBPTXT::T("in addition of today, valid from now on") . ')';
						}
					} else {
						$expText	= cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $expiryTime, false );
						$startText	= cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $startTime, false );
						$text		.= $startText;
						if ( $startText != $expText ) {
							$text	.= $this->_htmlNbsp( ' - ', $html ) . $expText;
						}
					}
				}
			}

		} else {
			$text = $this->_htmlNbsp( $this->renderPeriod( $this->getValidity( $varName ), 1, $displayOne ), $html );
		}
		return trim( $text );
	}

	/**
	 * Utility replacing spaces by %nbsp;
	 *
	 * @param  string   $text
	 * @param  boolean  $html
	 * @return string
	 */
	private function _htmlNbsp( $text, $html ) {
		if ( $html ) {
			$text		=			str_replace( ' ', '&nbsp;', $text );
		}
		return $text;
	}
}
